# Tourist Dashboard API Documentation

## 🔗 Base URL
```
http://localhost/GHTouristApp/api/
```

## 🔐 Authentication
Most endpoints require authentication via JWT token stored in localStorage as `auth_token`.

## 📊 Tourist API Endpoints

### Get User Statistics
**Endpoint:** `GET /tourist.php?action=stats&user_id={id}`

**Description:** Retrieves dashboard statistics for a user

**Parameters:**
- `user_id` (required): User ID

**Response:**
```json
{
  "success": true,
  "stats": {
    "total_bookings": 5,
    "upcoming_events": 3,
    "accommodation_bookings": 2,
    "reviews_given": 8
  }
}
```

### Get User Bookings
**Endpoint:** `GET /tourist.php?action=bookings&user_id={id}`

**Description:** Retrieves all event bookings for a user

**Response:**
```json
{
  "success": true,
  "tickets": [
    {
      "id": 1,
      "event_title": "Ghana Music Festival",
      "ticket_type": "VIP",
      "event_date": "2025-12-25",
      "venue_name": "Accra Conference Centre",
      "payment_status": "completed",
      "price_paid": 100.00,
      "ticket_code": "TKT123ABC"
    }
  ]
}
```

### Book Event
**Endpoint:** `POST /tourist.php?action=book-event`

**Description:** Books tickets for an event

**Request Body:**
```json
{
  "event_id": 1,
  "user_id": 1,
  "ticket_type": "standard",
  "quantity": 2,
  "payment_method": "mobile_money"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Event booked successfully",
  "ticket_code": "TKT456DEF",
  "total_price": 100.00
}
```

### Submit Safety Report
**Endpoint:** `POST /tourist.php?action=safety-report`

**Description:** Submits a safety incident report

**Request Body:**
```json
{
  "user_id": 1,
  "type": "security",
  "location": "Main entrance",
  "description": "Suspicious activity observed",
  "urgency": "medium",
  "event_id": 1
}
```

**Response:**
```json
{
  "success": true,
  "message": "Safety report submitted successfully"
}
```

### Submit Review
**Endpoint:** `POST /tourist.php?action=submit-review`

**Description:** Submits a review for an attended event

**Request Body:**
```json
{
  "event_id": 1,
  "user_id": 1,
  "rating": 5,
  "comment": "Amazing event! Great organization."
}
```

**Response:**
```json
{
  "success": true,
  "message": "Review submitted successfully"
}
```

### Get User Reviews
**Endpoint:** `GET /tourist.php?action=reviews&user_id={id}`

**Description:** Retrieves all reviews submitted by a user

**Response:**
```json
{
  "success": true,
  "reviews": [
    {
      "id": 1,
      "event_title": "Ghana Music Festival",
      "rating": 5,
      "comment": "Amazing event!",
      "created_at": "2025-09-15"
    }
  ]
}
```

### Update Profile
**Endpoint:** `PUT /tourist.php?action=update-profile`

**Description:** Updates user profile information

**Request Body:**
```json
{
  "user_id": 1,
  "full_name": "John Doe",
  "email": "john@example.com",
  "phone": "+233123456789"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Profile updated successfully"
}
```

## 🏨 Accommodations API Endpoints

### Get Accommodations List
**Endpoint:** `GET /accommodations.php?action=list`

**Description:** Retrieves list of available accommodations

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "name": "Labadi Beach Hotel",
      "type": "hotel",
      "location": "Labadi, Accra",
      "price_per_night": 250.00,
      "rating": 4.5,
      "amenities": "WiFi, Pool, Restaurant, Spa",
      "available_rooms": 15,
      "contact_phone": "+233-30-277-2501"
    }
  ],
  "total": 6
}
```

### Search Accommodations
**Endpoint:** `GET /accommodations.php?action=search&location={location}&type={type}&min_price={min}&max_price={max}`

**Description:** Searches accommodations with filters

**Parameters:**
- `location` (optional): Location filter
- `type` (optional): Accommodation type
- `min_price` (optional): Minimum price
- `max_price` (optional): Maximum price
- `check_in` (optional): Check-in date
- `check_out` (optional): Check-out date

### Book Accommodation
**Endpoint:** `POST /accommodations.php?action=book`

**Description:** Books accommodation

**Request Body:**
```json
{
  "accommodation_id": 1,
  "user_id": 1,
  "check_in_date": "2025-01-15",
  "check_out_date": "2025-01-17",
  "guests": 2,
  "room_type": "standard",
  "special_requests": "Late check-in requested"
}
```

**Response:**
```json
{
  "success": true,
  "message": "Accommodation booked successfully",
  "booking_reference": "ACC789GHI",
  "total_price": 500.00,
  "nights": 2
}
```

### Get User Accommodation Bookings
**Endpoint:** `GET /accommodations.php?action=bookings&user_id={id}`

**Description:** Retrieves accommodation bookings for a user

**Response:**
```json
{
  "success": true,
  "bookings": [
    {
      "id": 1,
      "booking_reference": "ACC789GHI",
      "accommodation_name": "Labadi Beach Hotel",
      "check_in_date": "2025-01-15",
      "check_out_date": "2025-01-17",
      "guests": 2,
      "total_price": 500.00,
      "booking_status": "confirmed"
    }
  ]
}
```

## 📅 Events API Integration

### Get Events List
**Endpoint:** `GET /events.php?action=list`

**Description:** Retrieves list of available events

**Response:**
```json
{
  "success": true,
  "events": [
    {
      "id": 1,
      "title": "Ghana Music Festival",
      "description": "Annual music celebration",
      "start_datetime": "2025-12-25 18:00:00",
      "venue_name": "Accra Conference Centre",
      "ticket_price": 50.00,
      "status": "approved"
    }
  ]
}
```

## 💬 Chat API Integration

### Get Chat Rooms
**Endpoint:** `GET /chat.php?endpoint=get_chat_rooms&event_id={id}`

**Description:** Retrieves chat rooms for an event

### Send Message
**Endpoint:** `POST /chat.php?endpoint=send_message`

**Request Body:**
```json
{
  "chat_room_id": 1,
  "user_id": 1,
  "message": "Looking forward to this event!"
}
```

## 🎫 Tickets API Integration

### Get User Tickets
**Endpoint:** `GET /tickets.php?action=user-tickets&user_id={id}`

**Description:** Retrieves tickets for a user

## ⚠️ Error Handling

### Standard Error Response
```json
{
  "success": false,
  "error": "Error message description"
}
```

### Common HTTP Status Codes
- `200` - Success
- `400` - Bad Request (missing parameters)
- `401` - Unauthorized (invalid token)
- `404` - Not Found
- `500` - Internal Server Error

## 🔒 Security Considerations

### Input Validation
- All inputs are validated and sanitized
- SQL injection prevention via PDO prepared statements
- XSS protection through HTML escaping

### Authentication
- JWT tokens for session management
- Token expiration handling
- Role-based access control

### Rate Limiting
- API endpoints have rate limiting
- Prevents abuse and ensures fair usage

## 📝 Request/Response Examples

### Booking Flow Example
```javascript
// 1. Get user stats
const statsResponse = await fetch('/api/tourist.php?action=stats&user_id=1');

// 2. Book event
const bookingResponse = await fetch('/api/tourist.php?action=book-event', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    event_id: 1,
    user_id: 1,
    ticket_type: 'standard',
    quantity: 1,
    payment_method: 'mobile_money'
  })
});

// 3. Get updated bookings
const bookingsResponse = await fetch('/api/tourist.php?action=bookings&user_id=1');
```

### Safety Report Example
```javascript
const reportResponse = await fetch('/api/tourist.php?action=safety-report', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({
    user_id: 1,
    type: 'security',
    location: 'Main entrance',
    description: 'Suspicious activity',
    urgency: 'high'
  })
});
```

## 🧪 Testing

### API Testing Tools
- Use Postman or similar tools for testing
- Include proper headers and authentication
- Test error scenarios and edge cases

### Sample Test Data
- User ID: 1 (for testing)
- Event ID: 1 (Ghana Music Festival)
- Accommodation ID: 1 (Labadi Beach Hotel)

---

**API Version:** 1.0  
**Last Updated:** September 17, 2025  
**Contact:** tech-support@ghtouristapp.gov.gh
