# Tourist Dashboard Deployment Guide

## 🚀 Production Deployment

### Prerequisites
- **Web Server**: Apache 2.4+ or Nginx 1.18+
- **PHP**: 8.0 or higher with extensions (PDO, cURL, JSON, OpenSSL)
- **MySQL**: 8.0 or higher
- **SSL Certificate**: Required for production
- **Domain**: Registered domain name

### Server Requirements
- **RAM**: 4GB minimum (8GB recommended)
- **Storage**: 50GB minimum
- **CPU**: 2 cores minimum
- **Bandwidth**: Unlimited or high limit
- **OS**: Ubuntu 20.04+ or CentOS 8+

## 📋 Pre-Deployment Checklist

### 1. Environment Setup
```bash
# Update system packages
sudo apt update && sudo apt upgrade -y

# Install required packages
sudo apt install apache2 mysql-server php8.0 php8.0-mysql php8.0-curl php8.0-json php8.0-mbstring php8.0-xml
```

### 2. Database Configuration
```sql
-- Create production database
CREATE DATABASE gh_tourist_app_prod;

-- Create dedicated user
CREATE USER 'gh_app_user'@'localhost' IDENTIFIED BY 'secure_production_password';
GRANT ALL PRIVILEGES ON gh_tourist_app_prod.* TO 'gh_app_user'@'localhost';
FLUSH PRIVILEGES;

-- Import schema
USE gh_tourist_app_prod;
SOURCE database/schema.sql;
```

### 3. File Permissions
```bash
# Set proper ownership
sudo chown -R www-data:www-data /var/www/html/GHTouristApp/

# Set directory permissions
sudo find /var/www/html/GHTouristApp/ -type d -exec chmod 755 {} \;

# Set file permissions
sudo find /var/www/html/GHTouristApp/ -type f -exec chmod 644 {} \;

# Set writable directories
sudo chmod 775 /var/www/html/GHTouristApp/uploads/
sudo chmod 775 /var/www/html/GHTouristApp/logs/
sudo chmod 775 /var/www/html/GHTouristApp/cache/
```

## 🔧 Configuration Files

### 1. Database Configuration
Create `config/database.php`:
```php
<?php
class Database {
    private $host = 'localhost';
    private $db_name = 'gh_tourist_app_prod';
    private $username = 'gh_app_user';
    private $password = 'secure_production_password';
    private $conn;

    public function getConnection() {
        $this->conn = null;
        try {
            $this->conn = new PDO(
                "mysql:host=" . $this->host . ";dbname=" . $this->db_name,
                $this->username,
                $this->password,
                array(PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION)
            );
        } catch(PDOException $exception) {
            error_log("Connection error: " . $exception->getMessage());
        }
        return $this->conn;
    }
}
?>
```

### 2. Apache Virtual Host
Create `/etc/apache2/sites-available/ghtouristapp.conf`:
```apache
<VirtualHost *:80>
    ServerName ghtouristapp.gov.gh
    ServerAlias www.ghtouristapp.gov.gh
    DocumentRoot /var/www/html/GHTouristApp
    
    # Redirect HTTP to HTTPS
    Redirect permanent / https://ghtouristapp.gov.gh/
</VirtualHost>

<VirtualHost *:443>
    ServerName ghtouristapp.gov.gh
    ServerAlias www.ghtouristapp.gov.gh
    DocumentRoot /var/www/html/GHTouristApp
    
    # SSL Configuration
    SSLEngine on
    SSLCertificateFile /path/to/certificate.crt
    SSLCertificateKeyFile /path/to/private.key
    SSLCertificateChainFile /path/to/chain.crt
    
    # Security Headers
    Header always set X-Content-Type-Options nosniff
    Header always set X-Frame-Options DENY
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Strict-Transport-Security "max-age=63072000; includeSubDomains; preload"
    
    # API Rewrite Rules
    RewriteEngine On
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteRule ^api/(.*)$ api/$1.php [QSA,L]
    
    # Error and Access Logs
    ErrorLog ${APACHE_LOG_DIR}/ghtouristapp_error.log
    CustomLog ${APACHE_LOG_DIR}/ghtouristapp_access.log combined
</VirtualHost>
```

### 3. PHP Configuration
Update `/etc/php/8.0/apache2/php.ini`:
```ini
# Security Settings
expose_php = Off
display_errors = Off
log_errors = On
error_log = /var/log/php/error.log

# Performance Settings
memory_limit = 256M
max_execution_time = 30
max_input_time = 60
post_max_size = 50M
upload_max_filesize = 20M

# Session Security
session.cookie_httponly = 1
session.cookie_secure = 1
session.use_strict_mode = 1

# OPcache Settings
opcache.enable = 1
opcache.memory_consumption = 128
opcache.max_accelerated_files = 4000
opcache.revalidate_freq = 60
```

## 🔒 Security Hardening

### 1. Firewall Configuration
```bash
# Enable UFW firewall
sudo ufw enable

# Allow SSH, HTTP, and HTTPS
sudo ufw allow ssh
sudo ufw allow 80/tcp
sudo ufw allow 443/tcp

# Deny all other incoming traffic
sudo ufw default deny incoming
sudo ufw default allow outgoing
```

### 2. SSL Certificate Setup
```bash
# Install Certbot for Let's Encrypt
sudo apt install certbot python3-certbot-apache

# Obtain SSL certificate
sudo certbot --apache -d ghtouristapp.gov.gh -d www.ghtouristapp.gov.gh

# Set up automatic renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

### 3. Database Security
```sql
-- Remove test databases and users
DROP DATABASE IF EXISTS test;
DELETE FROM mysql.user WHERE User='';
DELETE FROM mysql.user WHERE User='root' AND Host NOT IN ('localhost', '127.0.0.1', '::1');
FLUSH PRIVILEGES;

-- Set strong root password
ALTER USER 'root'@'localhost' IDENTIFIED BY 'very_secure_root_password';
```

## 📊 Performance Optimization

### 1. Enable Compression
Add to `.htaccess`:
```apache
# Enable Gzip compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/plain
    AddOutputFilterByType DEFLATE text/html
    AddOutputFilterByType DEFLATE text/xml
    AddOutputFilterByType DEFLATE text/css
    AddOutputFilterByType DEFLATE application/xml
    AddOutputFilterByType DEFLATE application/xhtml+xml
    AddOutputFilterByType DEFLATE application/rss+xml
    AddOutputFilterByType DEFLATE application/javascript
    AddOutputFilterByType DEFLATE application/x-javascript
</IfModule>
```

### 2. Browser Caching
Add to `.htaccess`:
```apache
# Browser Caching
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/ico "access plus 1 year"
    ExpiresByType image/icon "access plus 1 year"
    ExpiresByType text/plain "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
</IfModule>
```

### 3. MySQL Optimization
Update `/etc/mysql/mysql.conf.d/mysqld.cnf`:
```ini
[mysqld]
# Performance Settings
innodb_buffer_pool_size = 2G
innodb_log_file_size = 256M
innodb_flush_log_at_trx_commit = 2
innodb_flush_method = O_DIRECT

# Query Cache
query_cache_type = 1
query_cache_size = 128M
query_cache_limit = 2M

# Connection Settings
max_connections = 200
wait_timeout = 300
interactive_timeout = 300
```

## 📈 Monitoring & Logging

### 1. Log Rotation
Create `/etc/logrotate.d/ghtouristapp`:
```
/var/www/html/GHTouristApp/logs/*.log {
    daily
    missingok
    rotate 30
    compress
    delaycompress
    notifempty
    create 644 www-data www-data
}
```

### 2. System Monitoring
```bash
# Install monitoring tools
sudo apt install htop iotop nethogs

# Set up log monitoring
sudo apt install logwatch
```

### 3. Application Monitoring
Create monitoring script `/usr/local/bin/monitor_app.sh`:
```bash
#!/bin/bash
# Check if application is responding
curl -f -s http://localhost/GHTouristApp/api/events.php?action=list > /dev/null
if [ $? -ne 0 ]; then
    echo "Application not responding" | mail -s "GH Tourist App Alert" admin@ghtouristapp.gov.gh
fi
```

## 🔄 Backup Strategy

### 1. Database Backup
Create `/usr/local/bin/backup_db.sh`:
```bash
#!/bin/bash
BACKUP_DIR="/var/backups/ghtouristapp"
DATE=$(date +%Y%m%d_%H%M%S)
mkdir -p $BACKUP_DIR

# Database backup
mysqldump -u gh_app_user -p'secure_production_password' gh_tourist_app_prod > $BACKUP_DIR/db_backup_$DATE.sql

# Compress backup
gzip $BACKUP_DIR/db_backup_$DATE.sql

# Remove backups older than 30 days
find $BACKUP_DIR -name "*.sql.gz" -mtime +30 -delete
```

### 2. File Backup
```bash
#!/bin/bash
# Application files backup
tar -czf /var/backups/ghtouristapp/files_backup_$(date +%Y%m%d).tar.gz /var/www/html/GHTouristApp/uploads/
```

### 3. Automated Backups
Add to crontab:
```bash
# Daily database backup at 2 AM
0 2 * * * /usr/local/bin/backup_db.sh

# Weekly file backup on Sundays at 3 AM
0 3 * * 0 /usr/local/bin/backup_files.sh
```

## 🚦 Health Checks

### 1. Application Health Check
Create `health-check.php`:
```php
<?php
header('Content-Type: application/json');

$health = [
    'status' => 'healthy',
    'timestamp' => date('Y-m-d H:i:s'),
    'checks' => []
];

// Database check
try {
    require_once 'config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
    $health['checks']['database'] = 'ok';
} catch (Exception $e) {
    $health['status'] = 'unhealthy';
    $health['checks']['database'] = 'error';
}

// File system check
if (is_writable('uploads/') && is_writable('logs/')) {
    $health['checks']['filesystem'] = 'ok';
} else {
    $health['status'] = 'unhealthy';
    $health['checks']['filesystem'] = 'error';
}

echo json_encode($health);
?>
```

## 📋 Deployment Checklist

### Pre-Deployment
- [ ] Server provisioned and configured
- [ ] SSL certificate installed
- [ ] Database created and configured
- [ ] File permissions set correctly
- [ ] Firewall configured
- [ ] Backup system in place

### Deployment
- [ ] Code deployed to production server
- [ ] Database schema imported
- [ ] Configuration files updated
- [ ] Apache/Nginx restarted
- [ ] Health checks passing

### Post-Deployment
- [ ] Application accessible via HTTPS
- [ ] All features tested
- [ ] Monitoring systems active
- [ ] Backup jobs scheduled
- [ ] Documentation updated
- [ ] Team notified

## 🆘 Troubleshooting

### Common Issues

**500 Internal Server Error**
- Check Apache error logs: `tail -f /var/log/apache2/error.log`
- Verify file permissions
- Check PHP error logs

**Database Connection Issues**
- Verify database credentials
- Check MySQL service status: `sudo systemctl status mysql`
- Review MySQL error logs: `sudo tail -f /var/log/mysql/error.log`

**SSL Certificate Issues**
- Verify certificate files exist and are readable
- Check certificate expiration: `openssl x509 -in certificate.crt -text -noout`
- Test SSL configuration: `openssl s_client -connect ghtouristapp.gov.gh:443`

### Emergency Contacts
- **System Administrator**: +233 (0) 302 123 456
- **Database Administrator**: +233 (0) 302 123 457
- **Security Team**: +233 (0) 302 123 458

---

**Deployment Version:** 1.0  
**Last Updated:** September 17, 2025  
**Contact:** devops@ghtouristapp.gov.gh
