<?php
require_once '../config/database.php';
require_once '../includes/logger.php';
require_once '../includes/api-logger.php';

class AdminAPI {
    private $conn;
    private $logger;
    private $apiLogger;

    public function __construct() {
        $database = new Database();
        $this->conn = $database->getConnection();
        $this->logger = new Logger();
        $this->apiLogger = new APILogger();
        
        // Set headers
        header('Content-Type: application/json');
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
        
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            exit(0);
        }
    }

    public function handleRequest() {
        try {
            $method = $_SERVER['REQUEST_METHOD'];
            $path = $_SERVER['PATH_INFO'] ?? '/';
            $pathParts = explode('/', trim($path, '/'));
            
            $this->logger->log('INFO', 'Admin API Request', [
                'method' => $method,
                'path' => $path,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            ], 'admin.php');

            switch ($method) {
                case 'GET':
                    $this->handleGet($pathParts);
                    break;
                case 'POST':
                    $this->handlePost($pathParts);
                    break;
                case 'PUT':
                    $this->handlePut($pathParts);
                    break;
                case 'DELETE':
                    $this->handleDelete($pathParts);
                    break;
                default:
                    http_response_code(405);
                    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
                    break;
            }
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Admin API Exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 'admin.php');
            
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => 'Internal server error']);
        }
    }

    private function handleGet($pathParts) {
        $endpoint = $pathParts[0] ?? '';
        $subEndpoint = $pathParts[1] ?? '';
        
        // Log the incoming request
        error_log("🔍 Admin API GET Request - Endpoint: $endpoint, SubEndpoint: $subEndpoint, Full Path: " . implode('/', $pathParts));
        
        switch ($endpoint) {
            case 'stats':
                $this->getDashboardStats();
                break;
            case 'users':
                if ($subEndpoint === 'pending') {
                    error_log("📋 Calling getPendingUsers()");
                    $this->getPendingUsers();
                } else {
                    error_log("👥 Calling getUsers()");
                    $this->getUsers();
                }
                break;
            case 'events':
                $this->getEvents();
                break;
            case 'venues':
                $this->getVenues();
                break;
            case 'categories':
                $this->getCategories();
                break;
            case 'vendors':
                $this->getVendors();
                break;
            case 'government-agencies':
                if ($subEndpoint && is_numeric($subEndpoint)) {
                    $this->getGovernmentAgency($subEndpoint);
                } else {
                    $this->getGovernmentAgencies();
                }
                break;
            case 'safety-reports':
                if ($subEndpoint && is_numeric($subEndpoint)) {
                    $this->getSafetyReport($subEndpoint);
                } else {
                    $this->getSafetyReports();
                }
                break;
            case 'payment-transactions':
                if ($subEndpoint && is_numeric($subEndpoint)) {
                    $this->getPaymentTransaction($subEndpoint);
                } else {
                    $this->getPaymentTransactions();
                }
                break;
            case 'event-permits':
                if ($subEndpoint && is_numeric($subEndpoint)) {
                    $this->getEventPermit($subEndpoint);
                } else {
                    $this->getEventPermits();
                }
                break;
            case 'roles':
                if ($subEndpoint && is_numeric($subEndpoint)) {
                    $this->getRole($subEndpoint);
                } else {
                    $this->getRoles();
                }
                break;
            case 'emergency-contacts':
                $this->getEmergencyContacts();
                break;
            case 'logs':
                $this->getSystemLogs();
                break;
            case 'analytics':
                $this->getAnalytics();
                break;
            default:
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Endpoint not found']);
                break;
        }
    }

    private function handlePost($pathParts) {
        $endpoint = $pathParts[0] ?? '';
        
        switch ($endpoint) {
            case 'emergency-contacts':
                $this->createEmergencyContact();
                break;
            case 'users':
                // Check if this is an approve request based on the path
                if (isset($pathParts[1]) && $pathParts[1] === 'approve') {
                    $this->approveUser();
                } else {
                    // Check JSON input for action
                    $input = json_decode(file_get_contents('php://input'), true);
                    if ($input && isset($input['action']) && $input['action'] === 'approve') {
                        $this->approveUser();
                    } else {
                        $this->createUser();
                    }
                }
                break;
            case 'vendors':
                $this->createVendor();
                break;
            case 'events':
                $this->approveEvent();
                break;
            case 'venues':
                $this->createVenue();
                break;
            case 'categories':
                $this->createCategory();
                break;
            case 'vendors':
                $this->createVendor();
                break;
            case 'government-agencies':
                $this->createGovernmentAgency();
                break;
            case 'safety-reports':
                $this->createSafetyReport();
                break;
            case 'payment-transactions':
                $this->createPaymentTransaction();
                break;
            case 'event-permits':
                $this->createEventPermit();
                break;
            case 'roles':
                $this->createRole();
                break;
            default:
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Endpoint not found']);
                break;
        }
    }

    private function handlePut($pathParts) {
        $endpoint = $pathParts[0] ?? '';
        $id = $pathParts[1] ?? null;
        
        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'ID required']);
            return;
        }

        switch ($endpoint) {
            case 'users':
                $this->updateUser($id);
                break;
            case 'events':
                $this->updateEvent($id);
                break;
            case 'venues':
                $this->updateVenue($id);
                break;
            case 'government-agencies':
                $this->updateGovernmentAgency($id);
                break;
            case 'safety-reports':
                $this->updateSafetyReport($id);
                break;
            case 'payment-transactions':
                $this->updatePaymentTransaction($id);
                break;
            case 'event-permits':
                $this->updateEventPermit($id);
                break;
            case 'roles':
                $this->updateRole($id);
                break;
            default:
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Endpoint not found']);
                break;
        }
    }

    private function handleDelete($pathParts) {
        $endpoint = $pathParts[0] ?? '';
        $id = $pathParts[1] ?? null;
        
        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'error' => 'ID required']);
            return;
        }

        switch ($endpoint) {
            case 'users':
                $this->deleteUser($id);
                break;
            case 'events':
                $this->deleteEvent($id);
                break;
            case 'venues':
                $this->deleteVenue($id);
                break;
            case 'government-agencies':
                $this->deleteGovernmentAgency($id);
                break;
            case 'safety-reports':
                $this->deleteSafetyReport($id);
                break;
            case 'payment-transactions':
                $this->deletePaymentTransaction($id);
                break;
            case 'event-permits':
                $this->deleteEventPermit($id);
                break;
            case 'roles':
                $this->deleteRole($id);
                break;
            default:
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Endpoint not found']);
                break;
        }
    }

    // Dashboard Statistics
    private function getDashboardStats() {
        try {
            $stats = [];
            
            // Total Users
            $query = "SELECT COUNT(*) as count FROM users";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $stats['total_users'] = $stmt->fetch()['count'];
            
            // Total Events
            $query = "SELECT COUNT(*) as count FROM events";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $stats['total_events'] = $stmt->fetch()['count'];
            
            // Total Venues
            $query = "SELECT COUNT(*) as count FROM venues";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $stats['total_venues'] = $stmt->fetch()['count'];
            
            // Pending Approvals
            $query = "SELECT COUNT(*) as count FROM events WHERE approval_status = 'pending'";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $stats['pending_approvals'] = $stmt->fetch()['count'];
            
            // Recent Activity
            $query = "SELECT 'event' as type, title as name, created_at FROM events 
                     UNION ALL 
                     SELECT 'user' as type, full_name as name, created_at FROM users 
                     UNION ALL 
                     SELECT 'venue' as type, name, created_at FROM venues 
                     ORDER BY created_at DESC LIMIT 10";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $stats['recent_activity'] = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $stats]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Users Management
    private function getUsers() {
        try {
            $query = "SELECT u.*, ur.role_type, ur.role_level, ur.department 
                     FROM users u 
                     LEFT JOIN user_roles ur ON u.id = ur.user_id 
                     WHERE u.status != 'pending_verification'
                     ORDER BY u.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $users = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $users]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getPendingUsers() {
        try {
            $query = "SELECT u.*, ur.role_type, ur.role_level, ur.department 
                     FROM users u 
                     LEFT JOIN user_roles ur ON u.id = ur.user_id 
                     WHERE u.status = 'pending_verification'
                     ORDER BY u.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $users = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $users]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function updateUser($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $query = "UPDATE users SET 
                     full_name = :full_name,
                     email = :email,
                     phone = :phone,
                     status = :status
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':full_name', $input['full_name']);
            $stmt->bindParam(':email', $input['email']);
            $stmt->bindParam(':phone', $input['phone']);
            $stmt->bindParam(':status', $input['status']);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'User updated successfully']);
            } else {
                throw new Exception('Failed to update user');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function approveUser() {
        try {
            $rawInput = file_get_contents('php://input');
            $this->logger->log('DEBUG', 'Raw input received: ' . $rawInput);
            
            $input = json_decode($rawInput, true);
            $this->logger->log('DEBUG', 'Decoded input: ' . json_encode($input));
            
            if (!$input || !isset($input['user_id'])) {
                $this->logger->log('ERROR', 'User ID missing from input');
                throw new Exception('User ID is required');
            }
            
            $userId = $input['user_id'];
            $status = isset($input['status']) ? $input['status'] : 'active';
            
            $this->logger->log('DEBUG', "Processing approval for user ID: {$userId}, status: {$status}");
            
            // Map status values to database enum values
            if ($status === 'approved') {
                $status = 'active';
            } elseif ($status === 'rejected') {
                $status = 'suspended';
            }
            
            $this->logger->log('DEBUG', "Mapped status to: {$status}");
            
            // Check if user exists
            $checkQuery = "SELECT id, status FROM users WHERE id = :user_id";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':user_id', $userId);
            $checkStmt->execute();
            $user = $checkStmt->fetch();
            
            $this->logger->log('DEBUG', 'User lookup result: ' . json_encode($user));
            
            if (!$user) {
                $this->logger->log('ERROR', "User not found with ID: {$userId}");
                throw new Exception('User not found');
            }
            
            $query = "UPDATE users SET 
                     status = :status,
                     updated_at = NOW()
                     WHERE id = :user_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':user_id', $userId);
            
            $this->logger->log('DEBUG', "Executing update query with status: {$status}, user_id: {$userId}");
            
            $executeResult = $stmt->execute();
            $rowCount = $stmt->rowCount();
            
            $this->logger->log('DEBUG', "Update result - Execute: " . ($executeResult ? 'true' : 'false') . ", Row count: {$rowCount}");
            
            if ($executeResult && $rowCount > 0) {
                $message = $status === 'active' ? 'User approved successfully' : 'User rejected successfully';
                $this->logger->log('INFO', $message . " for user ID: {$userId}");
                echo json_encode(['success' => true, 'message' => $message]);
            } else {
                $this->logger->log('ERROR', "Update failed - Execute: " . ($executeResult ? 'true' : 'false') . ", Row count: {$rowCount}");
                throw new Exception('Failed to update user status or no changes made');
            }
            
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Exception in approveUser: ' . $e->getMessage());
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createUser() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['username'])) {
                throw new Exception('Username is required');
            }
            
            $username = trim($input['username']);
            $email = trim($input['email'] ?? '');
            $fullName = trim($input['full_name'] ?? '');
            $phone = trim($input['phone'] ?? '');
            $password = trim($input['password'] ?? '');
            
            if (empty($username)) {
                throw new Exception('Username cannot be empty');
            }
            
            if (empty($email)) {
                throw new Exception('Email is required');
            }
            
            if (empty($fullName)) {
                throw new Exception('Full name is required');
            }
            
            if (empty($password)) {
                throw new Exception('Password is required');
            }
            
            // Check if username already exists
            $checkQuery = "SELECT id FROM users WHERE username = :username OR email = :email";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':username', $username);
            $checkStmt->bindParam(':email', $email);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('Username or email already exists');
            }
            
            // Hash password
            $passwordHash = password_hash($password, PASSWORD_DEFAULT);
            
            $query = "INSERT INTO users (username, email, password_hash, full_name, phone, status, created_at) 
                     VALUES (:username, :email, :password_hash, :full_name, :phone, 'active', NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':username', $username);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':password_hash', $passwordHash);
            $stmt->bindParam(':full_name', $fullName);
            $stmt->bindParam(':phone', $phone);
            
            if ($stmt->execute()) {
                $userId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'User created successfully',
                    'data' => ['id' => $userId]
                ]);
            } else {
                throw new Exception('Failed to create user');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Events Management
    private function getEvents() {
        try {
            $query = "SELECT e.*, u.full_name as organizer_name, v.name as venue_name, 
                            ec.name as category_name, ec.color as category_color
                     FROM events e 
                     LEFT JOIN users u ON e.organizer_id = u.id 
                     LEFT JOIN venues v ON e.venue_id = v.id 
                     LEFT JOIN event_categories ec ON e.category_id = ec.id 
                     ORDER BY e.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $events = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $events]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function approveEvent() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            $eventId = $input['event_id'];
            $status = $input['status']; // 'approved' or 'rejected'
            
            $query = "UPDATE events SET 
                     approval_status = :status,
                     approval_date = NOW()
                     WHERE id = :event_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':event_id', $eventId);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => "Event {$status} successfully"]);
            } else {
                throw new Exception('Failed to update event status');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Venues Management
    private function getVenues() {
        try {
            $query = "SELECT v.*, u.full_name as owner_name,
                            (SELECT COUNT(*) FROM events WHERE venue_id = v.id) as event_count
                     FROM venues v 
                     LEFT JOIN users u ON v.owner_id = u.id 
                     ORDER BY v.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $venues = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $venues]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createVenue() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['name'])) {
                throw new Exception('Venue name is required');
            }
            
            $name = trim($input['name']);
            $type = trim($input['type'] ?? '');
            $address = trim($input['address'] ?? '');
            $city = trim($input['city'] ?? '');
            $region = trim($input['region'] ?? '');
            $capacity = isset($input['capacity']) ? (int)$input['capacity'] : null;
            $phone = trim($input['phone'] ?? '');
            $description = trim($input['description'] ?? '');
            
            if (empty($name)) {
                throw new Exception('Venue name cannot be empty');
            }
            
            if (empty($type)) {
                throw new Exception('Venue type is required');
            }
            
            if (empty($address)) {
                throw new Exception('Address is required');
            }
            
            if (empty($city)) {
                throw new Exception('City is required');
            }
            
            if (empty($region)) {
                throw new Exception('Region is required');
            }
            
            // Check if venue name already exists in the same city
            $checkQuery = "SELECT id FROM venues WHERE name = :name AND city = :city";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':name', $name);
            $checkStmt->bindParam(':city', $city);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('A venue with this name already exists in this city');
            }
            
            $query = "INSERT INTO venues (owner_id, name, type, address, city, region, capacity, contact_phone, description, created_at) 
                     VALUES (1, :name, :type, :address, :city, :region, :capacity, :phone, :description, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':type', $type);
            $stmt->bindParam(':address', $address);
            $stmt->bindParam(':city', $city);
            $stmt->bindParam(':region', $region);
            $stmt->bindParam(':capacity', $capacity);
            $stmt->bindParam(':phone', $phone);
            $stmt->bindParam(':description', $description);
            
            if ($stmt->execute()) {
                $venueId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Venue created successfully',
                    'data' => ['id' => $venueId]
                ]);
            } else {
                throw new Exception('Failed to create venue');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Categories Management
    private function getCategories() {
        try {
            $query = "SELECT ec.*, 
                            (SELECT COUNT(*) FROM events WHERE category_id = ec.id) as event_count
                     FROM event_categories ec 
                     ORDER BY ec.name";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $categories = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $categories]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createCategory() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['name'])) {
                throw new Exception('Category name is required');
            }
            
            $name = trim($input['name']);
            $description = trim($input['description'] ?? '');
            $color = trim($input['color'] ?? '#6c757d');
            
            if (empty($name)) {
                throw new Exception('Category name cannot be empty');
            }
            
            // Check if category name already exists
            $checkQuery = "SELECT id FROM event_categories WHERE name = :name";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':name', $name);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('Category name already exists');
            }
            
            $query = "INSERT INTO event_categories (name, description, color, created_at) 
                     VALUES (:name, :description, :color, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':description', $description);
            $stmt->bindParam(':color', $color);
            
            if ($stmt->execute()) {
                $categoryId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Category created successfully',
                    'data' => ['id' => $categoryId]
                ]);
            } else {
                throw new Exception('Failed to create category');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Roles Management
    private function getRoles() {
        try {
            $query = "SELECT ur.*, u.full_name as user_name, u.email as user_email,
                            assigned_user.full_name as assigned_by_name
                     FROM user_roles ur 
                     LEFT JOIN users u ON ur.user_id = u.id 
                     LEFT JOIN users assigned_user ON ur.assigned_by = assigned_user.id 
                     ORDER BY ur.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $roles = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $roles]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getRole($id) {
        try {
            $query = "SELECT ur.*, u.full_name as user_name, u.email as user_email,
                            assigned_user.full_name as assigned_by_name
                     FROM user_roles ur 
                     LEFT JOIN users u ON ur.user_id = u.id 
                     LEFT JOIN users assigned_user ON ur.assigned_by = assigned_user.id 
                     WHERE ur.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $role = $stmt->fetch();

            if ($role) {
                echo json_encode(['success' => true, 'data' => $role]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Role not found']);
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Emergency Contacts Management
    private function getEmergencyContacts() {
        try {
            $query = "SELECT * FROM emergency_contacts ORDER BY contact_type, name";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $contacts = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $contacts]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Vendors Management
    private function getVendors() {
        try {
            $query = "SELECT v.*, 
                            (SELECT COUNT(*) FROM event_vendor_bookings WHERE vendor_id = v.id) as booking_count
                     FROM vendors v 
                     ORDER BY v.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $vendors = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $vendors]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createVendor() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['name'])) {
                throw new Exception('Vendor name is required');
            }
            
            $name = trim($input['name']);
            $type = trim($input['type'] ?? '');
            $email = trim($input['email'] ?? '');
            $phone = trim($input['phone'] ?? '');
            $address = trim($input['address'] ?? '');
            $services = trim($input['services'] ?? '');
            
            if (empty($name)) {
                throw new Exception('Vendor name cannot be empty');
            }
            
            if (empty($type)) {
                throw new Exception('Vendor type is required');
            }
            
            if (empty($email)) {
                throw new Exception('Email is required');
            }
            
            if (empty($phone)) {
                throw new Exception('Phone is required');
            }
            
            if (empty($address)) {
                throw new Exception('Address is required');
            }
            
            // Validate business type against enum values
            $validTypes = ['catering', 'decoration', 'sound_equipment', 'photography', 'security', 'transport', 'accommodation'];
            if (!in_array($type, $validTypes)) {
                throw new Exception('Invalid business type. Must be one of: ' . implode(', ', $validTypes));
            }
            
            // Check if vendor email already exists
            $checkQuery = "SELECT id FROM vendors WHERE contact_email = :email";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':email', $email);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('A vendor with this email already exists');
            }
            
            // Convert services to JSON array
            $servicesJson = json_encode([$services]);
            
            $query = "INSERT INTO vendors (owner_id, business_name, business_type, contact_email, contact_phone, address, services_offered, created_at) 
                     VALUES (1, :name, :type, :email, :phone, :address, :services, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':type', $type);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':phone', $phone);
            $stmt->bindParam(':address', $address);
            $stmt->bindParam(':services', $servicesJson);
            
            if ($stmt->execute()) {
                $vendorId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Vendor created successfully',
                    'data' => ['id' => $vendorId]
                ]);
            } else {
                throw new Exception('Failed to create vendor');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Government Agencies Management
    private function getGovernmentAgencies() {
        try {
            $query = "SELECT ga.*, 
                            (SELECT COUNT(*) FROM event_permits WHERE issuing_agency_id = ga.id) as permit_count
                     FROM government_agencies ga 
                     ORDER BY ga.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $agencies = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $agencies]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getGovernmentAgency($id) {
        try {
            $query = "SELECT ga.*, 
                            (SELECT COUNT(*) FROM event_permits WHERE issuing_agency_id = ga.id) as permit_count
                     FROM government_agencies ga 
                     WHERE ga.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $agency = $stmt->fetch();

            if ($agency) {
                echo json_encode(['success' => true, 'data' => $agency]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Government agency not found']);
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createGovernmentAgency() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['name'])) {
                throw new Exception('Agency name is required');
            }
            
            $name = trim($input['name']);
            $type = trim($input['type'] ?? '');
            $level = trim($input['level'] ?? 'national');
            $email = trim($input['contact_email'] ?? $input['email'] ?? '');
            $phone = trim($input['contact_phone'] ?? $input['phone'] ?? '');
            $address = trim($input['address'] ?? '');
            
            if (empty($name)) {
                throw new Exception('Agency name cannot be empty');
            }
            
            if (empty($type)) {
                throw new Exception('Agency type is required');
            }
            
            if (empty($email)) {
                throw new Exception('Email is required');
            }
            
            if (empty($phone)) {
                throw new Exception('Phone is required');
            }
            
            if (empty($address)) {
                throw new Exception('Address is required');
            }
            
            // Check if agency email already exists
            $checkQuery = "SELECT id FROM government_agencies WHERE contact_email = :email";
            $checkStmt = $this->conn->prepare($checkQuery);
            $checkStmt->bindParam(':email', $email);
            $checkStmt->execute();
            
            if ($checkStmt->fetch()) {
                throw new Exception('A government agency with this email already exists');
            }
            
            $query = "INSERT INTO government_agencies (name, type, level, contact_email, contact_phone, address, created_at) 
                     VALUES (:name, :type, :level, :email, :phone, :address, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':type', $type);
            $stmt->bindParam(':level', $level);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':phone', $phone);
            $stmt->bindParam(':address', $address);
            
            if ($stmt->execute()) {
                $agencyId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Government agency created successfully',
                    'data' => ['id' => $agencyId]
                ]);
            } else {
                throw new Exception('Failed to create government agency');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function updateGovernmentAgency($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $query = "UPDATE government_agencies SET 
                     name = :name,
                     type = :type,
                     level = :level,
                     contact_email = :email,
                     contact_phone = :phone,
                     address = :address
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $input['name']);
            $stmt->bindParam(':type', $input['type']);
            $stmt->bindParam(':level', $input['level']);
            $stmt->bindParam(':email', $input['email']);
            $stmt->bindParam(':phone', $input['phone']);
            $stmt->bindParam(':address', $input['address']);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Government agency updated successfully']);
            } else {
                throw new Exception('Failed to update government agency');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function deleteGovernmentAgency($id) {
        try {
            $query = "DELETE FROM government_agencies WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Government agency deleted successfully']);
            } else {
                throw new Exception('Failed to delete government agency');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Safety Reports Management
    private function getSafetyReports() {
        try {
            $query = "SELECT sr.*, u.full_name as reporter_name, e.title as event_title
                     FROM safety_reports sr 
                     LEFT JOIN users u ON sr.reporter_id = u.id
                     LEFT JOIN events e ON sr.event_id = e.id
                     ORDER BY sr.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $reports = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $reports]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getSafetyReport($id) {
        try {
            $query = "SELECT sr.*, u.full_name as reporter_name, e.title as event_title
                     FROM safety_reports sr 
                     LEFT JOIN users u ON sr.reporter_id = u.id
                     LEFT JOIN events e ON sr.event_id = e.id
                     WHERE sr.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $report = $stmt->fetch();

            if ($report) {
                echo json_encode(['success' => true, 'data' => $report]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Safety report not found']);
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createSafetyReport() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $eventId = $input['event_id'] ?? null;
            $reportType = trim($input['report_type'] ?? 'security_concern');
            $severity = trim($input['severity'] ?? '');
            $status = trim($input['status'] ?? 'reported');
            $location = trim($input['location_description'] ?? '');
            $description = trim($input['description'] ?? '');
            $resolutionNotes = trim($input['resolution_notes'] ?? '');
            
            if (empty($severity)) {
                throw new Exception('Severity is required');
            }
            
            if (empty($location)) {
                throw new Exception('Location description is required');
            }
            
            if (empty($description)) {
                throw new Exception('Description is required');
            }
            
            $query = "INSERT INTO safety_reports (event_id, report_type, severity, status, location_description, description, resolution_notes, reporter_id, created_at) 
                     VALUES (:event_id, :report_type, :severity, :status, :location, :description, :resolution_notes, 1, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':event_id', $eventId);
            $stmt->bindParam(':report_type', $reportType);
            $stmt->bindParam(':severity', $severity);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':location', $location);
            $stmt->bindParam(':description', $description);
            $stmt->bindParam(':resolution_notes', $resolutionNotes);
            
            if ($stmt->execute()) {
                $reportId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Safety report created successfully',
                    'data' => ['id' => $reportId]
                ]);
            } else {
                throw new Exception('Failed to create safety report');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function updateSafetyReport($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $query = "UPDATE safety_reports SET 
                     event_id = :event_id,
                     report_type = :report_type,
                     severity = :severity,
                     status = :status,
                     location_description = :location,
                     description = :description,
                     resolution_notes = :resolution_notes
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':event_id', $input['event_id']);
            $stmt->bindParam(':report_type', $input['report_type']);
            $stmt->bindParam(':severity', $input['severity']);
            $stmt->bindParam(':status', $input['status']);
            $stmt->bindParam(':location', $input['location_description']);
            $stmt->bindParam(':description', $input['description']);
            $stmt->bindParam(':resolution_notes', $input['resolution_notes']);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Safety report updated successfully']);
            } else {
                throw new Exception('Failed to update safety report');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function deleteSafetyReport($id) {
        try {
            $query = "DELETE FROM safety_reports WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Safety report deleted successfully']);
            } else {
                throw new Exception('Failed to delete safety report');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Payment Transactions Management
    private function getPaymentTransactions() {
        try {
            $query = "SELECT pt.*, u.full_name as user_name, 
                            CASE 
                                WHEN pt.transaction_type = 'ticket_purchase' THEN e.title
                                ELSE CONCAT(pt.transaction_type, ' (ID: ', pt.reference_id, ')')
                            END as event_title
                     FROM payment_transactions pt 
                     LEFT JOIN users u ON pt.user_id = u.id
                     LEFT JOIN events e ON pt.reference_id = e.id AND pt.transaction_type = 'ticket_purchase'
                     ORDER BY pt.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $transactions = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $transactions]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getPaymentTransaction($id) {
        try {
            $query = "SELECT pt.*, u.full_name as user_name, 
                            CASE 
                                WHEN pt.transaction_type = 'ticket_purchase' THEN e.title
                                ELSE CONCAT(pt.transaction_type, ' (ID: ', pt.reference_id, ')')
                            END as event_title
                     FROM payment_transactions pt 
                     LEFT JOIN users u ON pt.user_id = u.id
                     LEFT JOIN events e ON pt.reference_id = e.id AND pt.transaction_type = 'ticket_purchase'
                     WHERE pt.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $transaction = $stmt->fetch();

            if ($transaction) {
                echo json_encode(['success' => true, 'data' => $transaction]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Payment transaction not found']);
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createPaymentTransaction() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $userId = $input['user_id'] ?? null;
            $referenceId = $input['reference_id'] ?? null;
            $transactionType = trim($input['transaction_type'] ?? '');
            $amount = $input['amount'] ?? 0;
            $paymentMethod = trim($input['payment_method'] ?? '');
            $status = trim($input['status'] ?? 'pending');
            $paymentProvider = trim($input['payment_provider'] ?? '');
            $providerTransactionId = trim($input['provider_transaction_id'] ?? '');
            $fees = $input['fees'] ?? 0;
            $taxAmount = $input['tax_amount'] ?? 0;
            $providerReference = trim($input['provider_reference'] ?? '');
            $netAmount = $amount - $fees - $taxAmount;
            
            if (empty($transactionType)) {
                throw new Exception('Transaction type is required');
            }
            
            if ($amount <= 0) {
                throw new Exception('Amount must be greater than 0');
            }
            
            if (empty($paymentMethod)) {
                throw new Exception('Payment method is required');
            }
            
            $query = "INSERT INTO payment_transactions (user_id, reference_id, transaction_type, amount, payment_method, status, payment_provider, provider_reference, fees, tax_amount, net_amount, created_at) 
                     VALUES (:user_id, :reference_id, :transaction_type, :amount, :payment_method, :status, :payment_provider, :provider_reference, :fees, :tax_amount, :net_amount, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':reference_id', $referenceId);
            $stmt->bindParam(':transaction_type', $transactionType);
            $stmt->bindParam(':amount', $amount);
            $stmt->bindParam(':payment_method', $paymentMethod);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':payment_provider', $paymentProvider);
            $stmt->bindParam(':provider_reference', $providerReference);
            $stmt->bindParam(':fees', $fees);
            $stmt->bindParam(':tax_amount', $taxAmount);
            $stmt->bindParam(':net_amount', $netAmount);
            
            if ($stmt->execute()) {
                $transactionId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Payment transaction created successfully',
                    'data' => ['id' => $transactionId]
                ]);
            } else {
                throw new Exception('Failed to create payment transaction');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function updatePaymentTransaction($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $netAmount = ($input['amount'] ?? 0) - ($input['fees'] ?? 0) - ($input['tax_amount'] ?? 0);
            
            $query = "UPDATE payment_transactions SET 
                     user_id = :user_id,
                     reference_id = :reference_id,
                     transaction_type = :transaction_type,
                     amount = :amount,
                     payment_method = :payment_method,
                     status = :status,
                     payment_provider = :payment_provider,
                     provider_reference = :provider_reference,
                     fees = :fees,
                     tax_amount = :tax_amount,
                     net_amount = :net_amount
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $input['user_id']);
            $stmt->bindParam(':reference_id', $input['reference_id']);
            $stmt->bindParam(':transaction_type', $input['transaction_type']);
            $stmt->bindParam(':amount', $input['amount']);
            $stmt->bindParam(':payment_method', $input['payment_method']);
            $stmt->bindParam(':status', $input['status']);
            $stmt->bindParam(':payment_provider', $input['payment_provider']);
            $stmt->bindParam(':provider_reference', $input['provider_reference']);
            $stmt->bindParam(':fees', $input['fees']);
            $stmt->bindParam(':tax_amount', $input['tax_amount']);
            $stmt->bindParam(':net_amount', $netAmount);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Payment transaction updated successfully']);
            } else {
                throw new Exception('Failed to update payment transaction');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function deletePaymentTransaction($id) {
        try {
            $query = "DELETE FROM payment_transactions WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Payment transaction deleted successfully']);
            } else {
                throw new Exception('Failed to delete payment transaction');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Event Permits Management
    private function getEventPermits() {
        try {
            $query = "SELECT ep.*, e.title as event_title, ga.name as agency_name
                     FROM event_permits ep 
                     LEFT JOIN events e ON ep.event_id = e.id
                     LEFT JOIN government_agencies ga ON ep.issuing_agency_id = ga.id
                     ORDER BY ep.created_at DESC";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $permits = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $permits]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function getEventPermit($id) {
        try {
            $query = "SELECT ep.*, e.title as event_title, ga.name as agency_name
                     FROM event_permits ep 
                     LEFT JOIN events e ON ep.event_id = e.id
                     LEFT JOIN government_agencies ga ON ep.issuing_agency_id = ga.id
                     WHERE ep.id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            $permit = $stmt->fetch();

            if ($permit) {
                echo json_encode(['success' => true, 'data' => $permit]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Event permit not found']);
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function createEventPermit() {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $eventId = $input['event_id'] ?? null;
            $agencyId = $input['agency_id'] ?? null;
            $permitType = trim($input['permit_type'] ?? '');
            $status = trim($input['status'] ?? 'pending');
            $issueDate = $input['issue_date'] ?? null;
            $expiryDate = $input['expiry_date'] ?? null;
            $feeAmount = $input['fee_amount'] ?? 0;
            $requirements = trim($input['requirements'] ?? '');
            $notes = trim($input['notes'] ?? '');
            
            if (empty($eventId)) {
                throw new Exception('Event is required');
            }
            
            if (empty($agencyId)) {
                throw new Exception('Government agency is required');
            }
            
            if (empty($permitType)) {
                throw new Exception('Permit type is required');
            }
            
            $query = "INSERT INTO event_permits (event_id, issuing_agency_id, permit_type, status, issue_date, expiry_date, fee_amount, notes, created_at) 
                     VALUES (:event_id, :agency_id, :permit_type, :status, :issue_date, :expiry_date, :fee_amount, :notes, NOW())";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':event_id', $eventId);
            $stmt->bindParam(':agency_id', $agencyId);
            $stmt->bindParam(':permit_type', $permitType);
            $stmt->bindParam(':status', $status);
            $stmt->bindParam(':issue_date', $issueDate);
            $stmt->bindParam(':expiry_date', $expiryDate);
            $stmt->bindParam(':fee_amount', $feeAmount);
            $stmt->bindParam(':notes', $notes);
            
            if ($stmt->execute()) {
                $permitId = $this->conn->lastInsertId();
                echo json_encode([
                    'success' => true, 
                    'message' => 'Event permit created successfully',
                    'data' => ['id' => $permitId]
                ]);
            } else {
                throw new Exception('Failed to create event permit');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function updateEventPermit($id) {
        try {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $query = "UPDATE event_permits SET 
                     event_id = :event_id,
                     issuing_agency_id = :agency_id,
                     permit_type = :permit_type,
                     status = :status,
                     issue_date = :issue_date,
                     expiry_date = :expiry_date,
                     fee_amount = :fee_amount,
                     notes = :notes
                     WHERE id = :id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':event_id', $input['event_id']);
            $stmt->bindParam(':agency_id', $input['agency_id']);
            $stmt->bindParam(':permit_type', $input['permit_type']);
            $stmt->bindParam(':status', $input['status']);
            $stmt->bindParam(':issue_date', $input['issue_date']);
            $stmt->bindParam(':expiry_date', $input['expiry_date']);
            $stmt->bindParam(':fee_amount', $input['fee_amount']);
            $stmt->bindParam(':notes', $input['notes']);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Event permit updated successfully']);
            } else {
                throw new Exception('Failed to update event permit');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function deleteEventPermit($id) {
        try {
            $query = "DELETE FROM event_permits WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Event permit deleted successfully']);
            } else {
                throw new Exception('Failed to delete event permit');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // System Logs Management
    private function getSystemLogs() {
        try {
            $query = "SELECT * FROM system_logs ORDER BY created_at DESC LIMIT 100";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $logs = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $logs]);
            
        } catch (Exception $e) {
            // If system_logs table doesn't exist, return empty array
            echo json_encode(['success' => true, 'data' => []]);
        }
    }

    // Analytics Management
    private function getAnalytics() {
        try {
            $analytics = [];
            
            // Events by month
            $query = "SELECT DATE_FORMAT(created_at, '%Y-%m') as month, COUNT(*) as count 
                     FROM events 
                     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
                     GROUP BY month 
                     ORDER BY month";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $analytics['events_by_month'] = $stmt->fetchAll();
            
            // User registrations by month
            $query = "SELECT DATE_FORMAT(created_at, '%Y-%m') as month, COUNT(*) as count 
                     FROM users 
                     WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
                     GROUP BY month 
                     ORDER BY month";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $analytics['users_by_month'] = $stmt->fetchAll();
            
            // Popular event categories
            $query = "SELECT ec.name, COUNT(e.id) as count 
                     FROM event_categories ec 
                     LEFT JOIN events e ON ec.id = e.category_id 
                     GROUP BY ec.id, ec.name 
                     ORDER BY count DESC 
                     LIMIT 10";
            $stmt = $this->conn->prepare($query);
            $stmt->execute();
            $analytics['popular_categories'] = $stmt->fetchAll();

            echo json_encode(['success' => true, 'data' => $analytics]);
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    // Delete operations
    private function deleteUser($id) {
        try {
            $query = "DELETE FROM users WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'User deleted successfully']);
            } else {
                throw new Exception('Failed to delete user');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }

    private function deleteEvent($id) {
        try {
            $query = "DELETE FROM events WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':id', $id);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Event deleted successfully']);
            } else {
                throw new Exception('Failed to delete event');
            }
            
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
    }
}

// Initialize and handle request
$api = new AdminAPI();
$api->handleRequest();
?>
