<?php
require_once '../config/database.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'events':
            if ($method === 'GET') {
                getEventRecommendations();
            }
            break;
            
        case 'venues':
            if ($method === 'GET') {
                getVenueRecommendations();
            }
            break;
            
        case 'vendors':
            if ($method === 'GET') {
                getVendorRecommendations();
            }
            break;
            
        case 'similar-users':
            if ($method === 'GET') {
                getSimilarUsers();
            }
            break;
            
        case 'preferences':
            if ($method === 'GET') {
                getUserPreferences();
            } elseif ($method === 'PUT') {
                updateUserPreferences();
            }
            break;
            
        case 'trending':
            if ($method === 'GET') {
                getTrendingEvents();
            }
            break;
            
        case 'personalized-feed':
            if ($method === 'GET') {
                getPersonalizedFeed();
            }
            break;
            
        case 'log-activity':
            if ($method === 'POST') {
                logUserActivity();
            }
            break;
            
        case 'analytics':
            if ($method === 'GET') {
                getRecommendationAnalytics();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function getEventRecommendations() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $limit = min(50, max(1, intval($_GET['limit'] ?? 10)));
    $algorithm = $_GET['algorithm'] ?? 'hybrid';
    
    $recommendations = [];
    
    switch ($algorithm) {
        case 'collaborative':
            $recommendations = getCollaborativeEventRecommendations($user['id'], $limit);
            break;
        case 'content':
            $recommendations = getContentBasedEventRecommendations($user['id'], $limit);
            break;
        case 'popularity':
            $recommendations = getPopularEventRecommendations($user['id'], $limit);
            break;
        default:
            $recommendations = getHybridEventRecommendations($user['id'], $limit);
    }
    
    // Log recommendation request
    logRecommendationRequest($user['id'], 'events', $algorithm);
    
    echo json_encode([
        'success' => true,
        'algorithm' => $algorithm,
        'recommendations' => $recommendations,
        'count' => count($recommendations)
    ]);
}

function getVenueRecommendations() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $limit = min(50, max(1, intval($_GET['limit'] ?? 10)));
    $venue_type = $_GET['type'] ?? '';
    $region = $_GET['region'] ?? '';
    
    // Get user preferences
    $preferences = getUserPreferencesData($user['id']);
    
    $where_conditions = ['v.status = ?'];
    $params = ['active'];
    
    if ($venue_type) {
        $where_conditions[] = 'v.type = ?';
        $params[] = $venue_type;
    }
    
    if ($region) {
        $where_conditions[] = 'v.region = ?';
        $params[] = $region;
    } elseif (!empty($preferences['preferred_regions'])) {
        $regions = json_decode($preferences['preferred_regions'], true);
        if (is_array($regions) && !empty($regions)) {
            $placeholders = str_repeat('?,', count($regions) - 1) . '?';
            $where_conditions[] = "v.region IN ($placeholders)";
            $params = array_merge($params, $regions);
        }
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Calculate recommendation score based on multiple factors
    $stmt = $pdo->prepare("
        SELECT v.*, 
               (v.rating * 0.3 + 
                (v.total_reviews / 100) * 0.2 + 
                CASE WHEN v.government_verified THEN 0.2 ELSE 0 END +
                (SELECT COUNT(*) FROM venue_bookings vb WHERE vb.venue_id = v.id AND vb.booking_status = 'completed') * 0.001 * 0.3
               ) as recommendation_score,
               (SELECT COUNT(*) FROM venue_bookings vb WHERE vb.venue_id = v.id AND vb.user_id = ?) as user_booking_count
        FROM venues v
        WHERE $where_clause
        ORDER BY recommendation_score DESC, v.rating DESC
        LIMIT ?
    ");
    
    $params = array_merge([$user['id']], $params, [$limit]);
    $stmt->execute($params);
    $venues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Enhance with additional data
    foreach ($venues as &$venue) {
        $venue['amenities'] = json_decode($venue['amenities'] ?? '[]', true);
        $venue['images'] = json_decode($venue['images'] ?? '[]', true);
        $venue['recommendation_score'] = round($venue['recommendation_score'], 3);
        $venue['has_user_booked'] = $venue['user_booking_count'] > 0;
        unset($venue['user_booking_count']);
    }
    
    echo json_encode([
        'success' => true,
        'recommendations' => $venues,
        'count' => count($venues)
    ]);
}

function getVendorRecommendations() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $limit = min(50, max(1, intval($_GET['limit'] ?? 10)));
    $business_type = $_GET['business_type'] ?? '';
    $event_id = $_GET['event_id'] ?? '';
    
    $where_conditions = ['v.status = ?'];
    $params = ['active'];
    
    if ($business_type) {
        $where_conditions[] = 'v.business_type = ?';
        $params[] = $business_type;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get vendor recommendations with scoring
    $stmt = $pdo->prepare("
        SELECT v.*, 
               (v.rating * 0.4 + 
                (v.total_reviews / 50) * 0.2 + 
                CASE WHEN v.government_verified THEN 0.2 ELSE 0 END +
                (SELECT COUNT(*) FROM event_vendor_bookings evb WHERE evb.vendor_id = v.id AND evb.booking_status = 'completed') * 0.002 * 0.2
               ) as recommendation_score,
               (SELECT COUNT(*) FROM event_vendor_bookings evb 
                JOIN events e ON evb.event_id = e.id 
                WHERE evb.vendor_id = v.id AND e.organizer_id = ?) as user_booking_count
        FROM vendors v
        WHERE $where_clause
        ORDER BY recommendation_score DESC, v.rating DESC
        LIMIT ?
    ");
    
    $params = array_merge([$user['id']], $params, [$limit]);
    $stmt->execute($params);
    $vendors = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Enhance with additional data
    foreach ($vendors as &$vendor) {
        $vendor['services_offered'] = json_decode($vendor['services_offered'] ?? '[]', true);
        $vendor['portfolio_images'] = json_decode($vendor['portfolio_images'] ?? '[]', true);
        $vendor['pricing_info'] = json_decode($vendor['pricing_info'] ?? '{}', true);
        $vendor['recommendation_score'] = round($vendor['recommendation_score'], 3);
        $vendor['has_user_booked'] = $vendor['user_booking_count'] > 0;
        unset($vendor['user_booking_count']);
    }
    
    echo json_encode([
        'success' => true,
        'recommendations' => $vendors,
        'count' => count($vendors)
    ]);
}

function getSimilarUsers() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $limit = min(20, max(1, intval($_GET['limit'] ?? 5)));
    
    // Find users with similar event preferences and activity patterns
    $stmt = $pdo->prepare("
        SELECT u.id, u.full_name, u.profile_image,
               COUNT(DISTINCT et1.event_id) as common_events,
               COUNT(DISTINCT ual1.entity_id) as common_activities
        FROM users u
        JOIN event_tickets et1 ON u.id = et1.user_id
        JOIN event_tickets et2 ON et1.event_id = et2.event_id AND et2.user_id = ?
        LEFT JOIN user_activity_log ual1 ON u.id = ual1.user_id
        LEFT JOIN user_activity_log ual2 ON ual1.entity_id = ual2.entity_id 
                                         AND ual1.activity_type = ual2.activity_type 
                                         AND ual2.user_id = ?
        WHERE u.id != ? AND u.status = 'active'
        GROUP BY u.id, u.full_name, u.profile_image
        HAVING common_events > 0
        ORDER BY (common_events * 0.7 + common_activities * 0.3) DESC
        LIMIT ?
    ");
    
    $stmt->execute([$user['id'], $user['id'], $user['id'], $limit]);
    $similar_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'similar_users' => $similar_users,
        'count' => count($similar_users)
    ]);
}

function getUserPreferences() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $preferences = getUserPreferencesData($user['id']);
    
    if (!$preferences) {
        // Create default preferences
        $preferences = createDefaultPreferences($user['id']);
    }
    
    // Parse JSON fields
    $preferences['preferred_event_categories'] = json_decode($preferences['preferred_event_categories'] ?? '[]', true);
    $preferences['preferred_venues'] = json_decode($preferences['preferred_venues'] ?? '[]', true);
    $preferences['preferred_regions'] = json_decode($preferences['preferred_regions'] ?? '[]', true);
    $preferences['budget_range'] = json_decode($preferences['budget_range'] ?? '{}', true);
    $preferences['accessibility_needs'] = json_decode($preferences['accessibility_needs'] ?? '[]', true);
    $preferences['dietary_restrictions'] = json_decode($preferences['dietary_restrictions'] ?? '[]', true);
    $preferences['language_preferences'] = json_decode($preferences['language_preferences'] ?? '[]', true);
    
    echo json_encode([
        'success' => true,
        'preferences' => $preferences
    ]);
}

function updateUserPreferences() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $stmt = $pdo->prepare("
        INSERT INTO user_preferences (
            user_id, preferred_event_categories, preferred_venues, 
            preferred_regions, budget_range, accessibility_needs,
            dietary_restrictions, language_preferences, notification_frequency
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
        preferred_event_categories = VALUES(preferred_event_categories),
        preferred_venues = VALUES(preferred_venues),
        preferred_regions = VALUES(preferred_regions),
        budget_range = VALUES(budget_range),
        accessibility_needs = VALUES(accessibility_needs),
        dietary_restrictions = VALUES(dietary_restrictions),
        language_preferences = VALUES(language_preferences),
        notification_frequency = VALUES(notification_frequency),
        updated_at = NOW()
    ");
    
    $stmt->execute([
        $user['id'],
        json_encode($input['preferred_event_categories'] ?? []),
        json_encode($input['preferred_venues'] ?? []),
        json_encode($input['preferred_regions'] ?? []),
        json_encode($input['budget_range'] ?? []),
        json_encode($input['accessibility_needs'] ?? []),
        json_encode($input['dietary_restrictions'] ?? []),
        json_encode($input['language_preferences'] ?? []),
        $input['notification_frequency'] ?? 'daily'
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Preferences updated successfully'
    ]);
}

function getTrendingEvents() {
    global $pdo;
    
    $limit = min(50, max(1, intval($_GET['limit'] ?? 10)));
    $timeframe = $_GET['timeframe'] ?? 'week'; // week, month, all
    
    $date_condition = '';
    switch ($timeframe) {
        case 'week':
            $date_condition = 'AND e.created_at >= DATE_SUB(NOW(), INTERVAL 1 WEEK)';
            break;
        case 'month':
            $date_condition = 'AND e.created_at >= DATE_SUB(NOW(), INTERVAL 1 MONTH)';
            break;
    }
    
    // Calculate trending score based on multiple factors
    $stmt = $pdo->prepare("
        SELECT e.*, ec.name as category_name, v.name as venue_name,
               (
                   (e.current_attendees / GREATEST(e.max_attendees, 1)) * 0.3 +
                   (SELECT COUNT(*) FROM event_tickets et WHERE et.event_id = e.id) * 0.001 * 0.2 +
                   (SELECT COUNT(*) FROM user_activity_log ual WHERE ual.entity_id = e.id AND ual.activity_type = 'event_view' AND ual.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)) * 0.01 * 0.3 +
                   (SELECT AVG(rating) FROM event_feedback ef WHERE ef.event_id = e.id) * 0.2 / 5
               ) as trending_score
        FROM events e
        LEFT JOIN event_categories ec ON e.category_id = ec.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE e.approval_status = 'approved' 
          AND e.start_datetime > NOW()
          $date_condition
        ORDER BY trending_score DESC, e.current_attendees DESC
        LIMIT ?
    ");
    
    $stmt->execute([$limit]);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Enhance with additional data
    foreach ($events as &$event) {
        $event['images'] = json_decode($event['images'] ?? '[]', true);
        $event['tags'] = json_decode($event['tags'] ?? '[]', true);
        $event['trending_score'] = round($event['trending_score'], 3);
    }
    
    echo json_encode([
        'success' => true,
        'trending_events' => $events,
        'timeframe' => $timeframe,
        'count' => count($events)
    ]);
}

function getPersonalizedFeed() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    
    // Get cached recommendations if available
    $cached = getCachedRecommendations($user['id'], 'events');
    if ($cached && strtotime($cached['expires_at']) > time()) {
        echo json_encode([
            'success' => true,
            'feed' => json_decode($cached['recommendations'], true),
            'cached' => true,
            'expires_at' => $cached['expires_at']
        ]);
        return;
    }
    
    // Generate fresh recommendations
    $hybrid_recommendations = getHybridEventRecommendations($user['id'], $limit);
    
    // Cache the results
    cacheRecommendations($user['id'], 'events', $hybrid_recommendations, 3600); // 1 hour cache
    
    echo json_encode([
        'success' => true,
        'feed' => $hybrid_recommendations,
        'cached' => false,
        'count' => count($hybrid_recommendations)
    ]);
}

function logUserActivity() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['activity_type', 'entity_type', 'entity_id'];
    foreach ($required_fields as $field) {
        if (!isset($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO user_activity_log (
            user_id, activity_type, entity_type, entity_id, 
            metadata, session_id, ip_address, user_agent
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $user['id'],
        $input['activity_type'],
        $input['entity_type'],
        $input['entity_id'],
        json_encode($input['metadata'] ?? []),
        session_id(),
        $_SERVER['REMOTE_ADDR'] ?? '',
        $_SERVER['HTTP_USER_AGENT'] ?? ''
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Activity logged successfully'
    ]);
}

function getRecommendationAnalytics() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!hasPermission($user['id'], 'recommendation_analytics')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    // Recommendation usage statistics
    $stmt = $pdo->prepare("
        SELECT 
            recommendation_type,
            COUNT(*) as total_requests,
            COUNT(DISTINCT user_id) as unique_users,
            AVG(score) as avg_score
        FROM recommendation_cache
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY recommendation_type
    ");
    $stmt->execute();
    $usage_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // User activity trends
    $stmt = $pdo->prepare("
        SELECT 
            activity_type,
            DATE(created_at) as date,
            COUNT(*) as activity_count
        FROM user_activity_log
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY activity_type, DATE(created_at)
        ORDER BY date DESC
    ");
    $stmt->execute();
    $activity_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'analytics' => [
            'usage_stats' => $usage_stats,
            'activity_trends' => $activity_trends
        ]
    ]);
}

// Helper functions
function getCollaborativeEventRecommendations($user_id, $limit) {
    global $pdo;
    
    // Find events liked by similar users
    $stmt = $pdo->prepare("
        SELECT DISTINCT e.*, ec.name as category_name, v.name as venue_name,
               COUNT(et2.user_id) as similar_user_count
        FROM events e
        LEFT JOIN event_categories ec ON e.category_id = ec.id
        LEFT JOIN venues v ON e.venue_id = v.id
        JOIN event_tickets et1 ON e.id = et1.event_id
        JOIN event_tickets et2 ON et1.user_id != et2.user_id
        JOIN event_tickets et3 ON et2.user_id = et3.user_id
        JOIN event_tickets et4 ON et3.event_id = et4.event_id AND et4.user_id = ?
        WHERE e.approval_status = 'approved' 
          AND e.start_datetime > NOW()
          AND e.id NOT IN (SELECT event_id FROM event_tickets WHERE user_id = ?)
        GROUP BY e.id
        ORDER BY similar_user_count DESC, e.current_attendees DESC
        LIMIT ?
    ");
    
    $stmt->execute([$user_id, $user_id, $limit]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getContentBasedEventRecommendations($user_id, $limit) {
    global $pdo;
    
    // Get user's preferred categories from past events
    $preferences = getUserPreferencesData($user_id);
    $preferred_categories = json_decode($preferences['preferred_event_categories'] ?? '[]', true);
    
    if (empty($preferred_categories)) {
        // Fallback to user's event history
        $stmt = $pdo->prepare("
            SELECT DISTINCT e.category_id
            FROM events e
            JOIN event_tickets et ON e.id = et.event_id
            WHERE et.user_id = ?
            ORDER BY e.created_at DESC
            LIMIT 5
        ");
        $stmt->execute([$user_id]);
        $preferred_categories = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'category_id');
    }
    
    if (empty($preferred_categories)) {
        return getPopularEventRecommendations($user_id, $limit);
    }
    
    $placeholders = str_repeat('?,', count($preferred_categories) - 1) . '?';
    
    $stmt = $pdo->prepare("
        SELECT e.*, ec.name as category_name, v.name as venue_name
        FROM events e
        LEFT JOIN event_categories ec ON e.category_id = ec.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE e.approval_status = 'approved' 
          AND e.start_datetime > NOW()
          AND e.category_id IN ($placeholders)
          AND e.id NOT IN (SELECT event_id FROM event_tickets WHERE user_id = ?)
        ORDER BY e.current_attendees DESC, e.created_at DESC
        LIMIT ?
    ");
    
    $params = array_merge($preferred_categories, [$user_id, $limit]);
    $stmt->execute($params);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getPopularEventRecommendations($user_id, $limit) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT e.*, ec.name as category_name, v.name as venue_name
        FROM events e
        LEFT JOIN event_categories ec ON e.category_id = ec.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE e.approval_status = 'approved' 
          AND e.start_datetime > NOW()
          AND e.id NOT IN (SELECT event_id FROM event_tickets WHERE user_id = ?)
        ORDER BY e.current_attendees DESC, e.created_at DESC
        LIMIT ?
    ");
    
    $stmt->execute([$user_id, $limit]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getHybridEventRecommendations($user_id, $limit) {
    // Combine collaborative, content-based, and popularity-based recommendations
    $collaborative = getCollaborativeEventRecommendations($user_id, intval($limit * 0.4));
    $content_based = getContentBasedEventRecommendations($user_id, intval($limit * 0.4));
    $popular = getPopularEventRecommendations($user_id, intval($limit * 0.2));
    
    // Merge and deduplicate
    $all_recommendations = array_merge($collaborative, $content_based, $popular);
    $unique_recommendations = [];
    $seen_ids = [];
    
    foreach ($all_recommendations as $rec) {
        if (!in_array($rec['id'], $seen_ids)) {
            $unique_recommendations[] = $rec;
            $seen_ids[] = $rec['id'];
        }
    }
    
    return array_slice($unique_recommendations, 0, $limit);
}

function getUserPreferencesData($user_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT * FROM user_preferences WHERE user_id = ?");
    $stmt->execute([$user_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function createDefaultPreferences($user_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO user_preferences (user_id) VALUES (?)
    ");
    $stmt->execute([$user_id]);
    
    return getUserPreferencesData($user_id);
}

function getCachedRecommendations($user_id, $type) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        SELECT * FROM recommendation_cache 
        WHERE user_id = ? AND recommendation_type = ? AND expires_at > NOW()
        ORDER BY created_at DESC LIMIT 1
    ");
    $stmt->execute([$user_id, $type]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function cacheRecommendations($user_id, $type, $recommendations, $ttl_seconds) {
    global $pdo;
    
    $expires_at = date('Y-m-d H:i:s', time() + $ttl_seconds);
    
    $stmt = $pdo->prepare("
        INSERT INTO recommendation_cache (
            user_id, recommendation_type, recommendations, expires_at
        ) VALUES (?, ?, ?, ?)
    ");
    
    $stmt->execute([
        $user_id,
        $type,
        json_encode($recommendations),
        $expires_at
    ]);
}

function logRecommendationRequest($user_id, $type, $algorithm) {
    global $pdo;
    
    $stmt = $pdo->prepare("
        INSERT INTO user_activity_log (
            user_id, activity_type, entity_type, entity_id, metadata
        ) VALUES (?, 'recommendation_request', 'system', 0, ?)
    ");
    
    $stmt->execute([
        $user_id,
        json_encode(['type' => $type, 'algorithm' => $algorithm])
    ]);
}
?>
