<?php
/**
 * Ghana National Event Management Platform
 * Authentication API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/logger.php';
require_once '../includes/api-logger.php';

class AuthAPI {
    private $auth;
    private $logger;
    private $apiLogger;

    public function __construct() {
        $this->auth = new Auth();
        $this->logger = new Logger();
        $this->apiLogger = new ApiLogger();
        
        // Log database connection attempt
        $this->logger->log('DATABASE', 'Attempting database connection', [
            'host' => 'localhost',
            'database' => 'gh_tourist_app',
            'username' => 'root'
        ], 'auth.php');
        
        try {
            $db = new Database();
            $conn = $db->getConnection();
            $this->logger->log('DATABASE', 'Database connection successful', [], 'auth.php');
        } catch (Exception $e) {
            $this->logger->log('ERROR', 'Database connection failed', [
                'error' => $e->getMessage()
            ], 'auth.php');
        }
    }

    /**
     * Handle API requests
     */
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        
        $action = end($pathParts);
        
        // Log API request
        $this->logger->log('INFO', 'API Request', [
            'method' => $method,
            'action' => $action,
            'query_params' => $_GET,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ], 'auth.php');
        
        // Log to API calls log
        $this->apiLogger->logRequest($method, $action, [
            'endpoint' => $action,
            'method' => $method
        ] + $_GET);
        
        try {
            switch ($method) {
                case 'GET':
                    $this->handleGet($action);
                    break;
                case 'POST':
                    $this->handlePost($action);
                    break;
                case 'PUT':
                    $this->handlePut($action);
                    break;
                default:
                    throw new Exception('Method not allowed');
            }
        } catch (Exception $e) {
            http_response_code(400);
            $errorResponse = ['error' => $e->getMessage()];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'API request failed', [
                'error' => $e->getMessage(),
                'method' => $method,
                'action' => $action
            ], 'auth.php');
            
            $this->apiLogger->logResponse($method, $action, $errorResponse);
        }
    }

    /**
     * Handle GET requests
     */
    private function handleGet($action) {
        switch ($action) {
            case 'profile':
                $this->getProfile();
                break;
            case 'verify-token':
                $this->verifyToken();
                break;
            default:
                http_response_code(404);
                echo json_encode(['success' => false, 'error' => 'Endpoint not found']);
                break;
        }
    }

    /**
     * Handle POST requests
     */
    private function handlePost($action) {
        switch ($action) {
            case 'register':
                $this->register();
                break;
            case 'login':
                $this->login();
                break;
            case 'logout':
                $this->logout();
                break;
            case 'forgot-password':
                $this->forgotPassword();
                break;
            case 'reset-password':
                $this->resetPassword();
                break;
            case 'verify-email':
                $this->verifyEmail();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Handle PUT requests
     */
    private function handlePut($action) {
        switch ($action) {
            case 'profile':
                $this->updateProfile();
                break;
            case 'password':
                $this->changePassword();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * User registration
     */
    private function register() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Log registration request
        $this->logger->log('INFO', 'Registration Request', [
            'input' => $input
        ], 'auth.php');
        
        $this->apiLogger->logRequest('POST', 'register', [
            'endpoint' => 'register',
            'method' => 'POST',
            'input' => $input
        ]);
        
        if (!$input) {
            http_response_code(400);
            $errorResponse = [
                'success' => false,
                'error' => 'Invalid JSON input'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Invalid JSON input', [
                'input' => $input
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'register', $errorResponse);
            return;
        }

        // Validate required fields
        $requiredFields = ['full_name', 'username', 'email', 'password', 'phone', 'role_type'];
        $missingFields = [];
        
        foreach ($requiredFields as $field) {
            if (empty($input[$field])) {
                $missingFields[] = $field;
            }
        }
        
        if (!empty($missingFields)) {
            http_response_code(400);
            $errorResponse = [
                'success' => false,
                'error' => 'Missing required fields: ' . implode(', ', $missingFields)
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Missing required fields', [
                'missingFields' => $missingFields
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'register', $errorResponse);
            return;
        }

        // Check for existing users before attempting registration
        $db = new Database();
        $conn = $db->getConnection();
        
        // Check for existing username
        $checkUsername = "SELECT id FROM users WHERE username = :username";
        $stmt = $conn->prepare($checkUsername);
        $stmt->bindParam(':username', $input['username']);
        $stmt->execute();
        
        if ($stmt->fetch()) {
            http_response_code(409);
            $errorResponse = [
                'success' => false,
                'error' => 'Username already exists. Please choose a different username.',
                'field' => 'username'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Username already exists', [
                'username' => $input['username']
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'register', $errorResponse);
            return;
        }
        
        // Check for existing email
        $checkEmail = "SELECT id FROM users WHERE email = :email";
        $stmt = $conn->prepare($checkEmail);
        $stmt->bindParam(':email', $input['email']);
        $stmt->execute();
        
        if ($stmt->fetch()) {
            http_response_code(409);
            $errorResponse = [
                'success' => false,
                'error' => 'An account with this email already exists. Please use a different email or try logging in.',
                'field' => 'email'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Email already exists', [
                'email' => $input['email']
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'register', $errorResponse);
            return;
        }

        // Attempt registration
        try {
            $result = $this->auth->register($input);
            
            if ($result['success']) {
                http_response_code(201);
                $successResponse = [
                    'success' => true,
                    'message' => 'Account created successfully! Please check your email for verification instructions.',
                    'user_id' => $result['user_id'] ?? null
                ];
                echo json_encode($successResponse);
                
                $this->logger->log('INFO', 'Registration successful', [
                    'user_id' => $result['user_id'] ?? null
                ], 'auth.php');
                
                $this->apiLogger->logResponse('POST', 'register', $successResponse);
            } else {
                http_response_code(400);
                $errorResponse = [
                    'success' => false,
                    'error' => $result['message'] ?? 'Registration failed. Please try again.'
                ];
                echo json_encode($errorResponse);
                
                $this->logger->log('ERROR', 'Registration failed', [
                    'error' => $result['message'] ?? 'Registration failed. Please try again.'
                ], 'auth.php');
                
                $this->apiLogger->logResponse('POST', 'register', $errorResponse);
            }
        } catch (Exception $e) {
            error_log("Registration error: " . $e->getMessage());
            http_response_code(500);
            $errorResponse = [
                'success' => false,
                'error' => 'An unexpected error occurred. Please try again later.'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Registration error', [
                'error' => $e->getMessage()
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'register', $errorResponse);
        }
    }

    /**
     * Handle user login
     */
    private function login() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        $this->logger->log('INFO', 'Login Request', [
            'input' => array_merge($input, ['password' => '[REDACTED]'])
        ], 'auth.php');
        
        $this->apiLogger->logRequest('POST', 'login', $input);
        
        // Validate required fields
        $requiredFields = ['email', 'password'];
        $missingFields = [];
        
        foreach ($requiredFields as $field) {
            if (empty($input[$field])) {
                $missingFields[] = $field;
            }
        }
        
        if (!empty($missingFields)) {
            http_response_code(400);
            $errorResponse = [
                'success' => false,
                'error' => 'Missing required fields: ' . implode(', ', $missingFields)
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Missing required fields', [
                'missingFields' => $missingFields
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'login', $errorResponse);
            return;
        }
        
        try {
            $result = $this->auth->login($input['email'], $input['password']);
            
            if ($result['success']) {
                http_response_code(200);
                $successResponse = [
                    'success' => true,
                    'message' => 'Login successful',
                    'token' => $result['token'],
                    'user' => $result['user'],
                    'expires_in' => $result['expires_in']
                ];
                
                // Add debugging log to see what user data contains
                $this->logger->log('DEBUG', 'Login response data', [
                    'user_data' => $result['user'],
                    'has_role_type' => isset($result['user']['role_type']),
                    'role_type_value' => $result['user']['role_type'] ?? 'NOT_SET'
                ], 'auth.php');
                
                echo json_encode($successResponse);
                
                $this->logger->log('INFO', 'Login successful', [
                    'user_id' => $result['user']['id'],
                    'username' => $result['user']['username']
                ], 'auth.php');
                
                $this->apiLogger->logResponse('POST', 'login', $successResponse);
            } else {
                http_response_code(401);
                $errorResponse = [
                    'success' => false,
                    'error' => $result['message'] ?? 'Login failed'
                ];
                echo json_encode($errorResponse);
                
                $this->logger->log('ERROR', 'Login failed', [
                    'username' => $input['email'],
                    'error' => $result['message'] ?? 'Login failed'
                ], 'auth.php');
                
                $this->apiLogger->logResponse('POST', 'login', $errorResponse);
            }
        } catch (Exception $e) {
            http_response_code(500);
            $errorResponse = [
                'success' => false,
                'error' => 'An unexpected error occurred. Please try again later.'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Login error', [
                'error' => $e->getMessage()
            ], 'auth.php');
            
            $this->apiLogger->logResponse('POST', 'login', $errorResponse);
        }
    }

    /**
     * Verify JWT token
     */
    private function verifyToken() {
        $headers = getallheaders();
        $authHeader = $headers['Authorization'] ?? '';
        
        if (!$authHeader || !str_starts_with($authHeader, 'Bearer ')) {
            http_response_code(401);
            $errorResponse = [
                'success' => false,
                'error' => 'No token provided'
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Token verification failed - No token', [], 'auth.php');
            $this->apiLogger->logResponse('GET', 'verify-token', $errorResponse);
            return;
        }
        
        $token = substr($authHeader, 7);
        
        try {
            $decoded = $this->auth->verifyJWT($token);
            
            if ($decoded) {
                // Get fresh user data
                $user = $this->auth->getUserById($decoded['user_id']);
                
                if ($user) {
                    http_response_code(200);
                    $successResponse = [
                        'success' => true,
                        'user' => $this->auth->sanitizeUserData($user),
                        'valid' => true
                    ];
                    echo json_encode($successResponse);
                    
                    $this->logger->log('INFO', 'Token verified successfully', [
                        'user_id' => $user['id']
                    ], 'auth.php');
                    
                    $this->apiLogger->logResponse('GET', 'verify-token', $successResponse);
                } else {
                    throw new Exception('User not found');
                }
            } else {
                throw new Exception('Invalid token');
            }
        } catch (Exception $e) {
            http_response_code(401);
            $errorResponse = [
                'success' => false,
                'error' => 'Invalid token',
                'valid' => false
            ];
            echo json_encode($errorResponse);
            
            $this->logger->log('ERROR', 'Token verification failed', [
                'error' => $e->getMessage()
            ], 'auth.php');
            
            $this->apiLogger->logResponse('GET', 'verify-token', $errorResponse);
        }
    }

    // ... rest of the code remains the same ...
}

// Handle the request
$api = new AuthAPI();
$api->handleRequest();
?>
