<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';
require_once '../config/auth.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit();
}

try {
    // Verify authentication
    $headers = getallheaders();
    $authHeader = $headers['Authorization'] ?? '';
    
    if (!$authHeader || !str_starts_with($authHeader, 'Bearer ')) {
        throw new Exception('Authentication required');
    }
    
    $token = substr($authHeader, 7);
    $user = verifyToken($token);
    
    if (!$user) {
        throw new Exception('Invalid authentication token');
    }

    // Validate required fields
    if (!isset($_POST['event_id']) || !isset($_POST['user_id']) || !isset($_POST['message_type'])) {
        throw new Exception('Missing required fields');
    }

    $event_id = (int)$_POST['event_id'];
    $user_id = (int)$_POST['user_id'];
    $room_id = $_POST['room_id'] ?? $event_id;
    $message_type = $_POST['message_type'];

    // Verify user matches token
    if ($user['id'] != $user_id) {
        throw new Exception('User ID mismatch');
    }

    // Check if file was uploaded
    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('File upload failed');
    }

    $file = $_FILES['file'];
    
    // Validate file
    $maxFileSize = 10 * 1024 * 1024; // 10MB
    $allowedTypes = [
        'image/jpeg', 'image/png', 'image/gif', 'image/webp',
        'video/mp4', 'video/webm', 'video/quicktime',
        'application/pdf', 'application/msword', 
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        'text/plain', 'application/zip', 'application/x-rar-compressed'
    ];

    if ($file['size'] > $maxFileSize) {
        throw new Exception('File too large. Maximum size is 10MB');
    }

    if (!in_array($file['type'], $allowedTypes)) {
        throw new Exception('File type not supported');
    }

    // Create uploads directory if it doesn't exist
    $uploadDir = '../uploads/chat/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }

    // Generate unique filename
    $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $fileName = uniqid('chat_' . $user_id . '_') . '.' . $fileExtension;
    $filePath = $uploadDir . $fileName;

    // Move uploaded file
    if (!move_uploaded_file($file['tmp_name'], $filePath)) {
        throw new Exception('Failed to save file');
    }

    // Get file URL for database storage
    $fileUrl = '/GHTouristApp/uploads/chat/' . $fileName;

    // Connect to database
    $pdo = getDBConnection();

    // Insert message into chat_messages table
    $stmt = $pdo->prepare("
        INSERT INTO chat_messages (
            room_id, user_id, content, message_type, file_url, file_name, file_size, created_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");

    $content = $message_type === 'image' ? 'Shared an image' : 
              ($message_type === 'video' ? 'Shared a video' : 'Shared a file: ' . $file['name']);

    $stmt->execute([
        $room_id,
        $user_id,
        $content,
        $message_type,
        $fileUrl,
        $file['name'],
        $file['size']
    ]);

    $messageId = $pdo->lastInsertId();

    // Also insert into simple chat system
    try {
        // Create enhanced simple chat messages table if it doesn't exist
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS simple_chat_messages (
                id INT PRIMARY KEY AUTO_INCREMENT,
                event_id INT NOT NULL,
                user_id INT NOT NULL,
                message TEXT NOT NULL,
                message_type ENUM('text', 'image', 'video', 'file') DEFAULT 'text',
                file_url VARCHAR(500) NULL,
                file_name VARCHAR(255) NULL,
                file_size INT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (event_id) REFERENCES events(id),
                FOREIGN KEY (user_id) REFERENCES users(id)
            )
        ");
        
        $stmt = $pdo->prepare("
            INSERT INTO simple_chat_messages (
                event_id, user_id, message, message_type, file_url, file_name, file_size
            ) VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $event_id,
            $user_id,
            $content,
            $message_type,
            $fileUrl,
            $file['name'],
            $file['size']
        ]);
    } catch (Exception $e) {
        // Log error but don't fail the main operation
        error_log("Simple chat table insert failed: " . $e->getMessage());
    }

    echo json_encode([
        'success' => true,
        'message_id' => $messageId,
        'file_url' => $fileUrl,
        'file_name' => $file['name'],
        'file_size' => $file['size'],
        'message' => 'Media uploaded successfully'
    ]);

} catch (Exception $e) {
    error_log("Chat media upload error: " . $e->getMessage());
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
