<?php
require_once '../config/database.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Initialize database connection
$db = new Database();
$pdo = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'rooms':
            if ($method === 'GET') {
                getChatRooms();
            } elseif ($method === 'POST') {
                createChatRoom();
            }
            break;
        
        case 'join':
            if ($method === 'POST') {
                joinChatRoom();
            }
            break;
            
        case 'leave':
            if ($method === 'POST') {
                leaveChatRoom();
            }
            break;
            
        case 'messages':
            if ($method === 'GET') {
                getChatMessages();
            } elseif ($method === 'POST') {
                sendMessage();
            }
            break;
            
        case 'moderate':
            if ($method === 'POST') {
                moderateMessage();
            }
            break;
            
        case 'participants':
            if ($method === 'GET') {
                getRoomParticipants();
            }
            break;
            
        case 'ban-user':
            if ($method === 'POST') {
                banUser();
            }
            break;
            
        case 'room-settings':
            if ($method === 'PUT') {
                updateRoomSettings();
            }
            break;
            
        case 'typing':
            if ($method === 'POST') {
                updateTypingStatus();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function getChatRooms() {
    global $pdo;
    
    // Temporarily bypass authentication for organizer dashboard
    $organizer_id = $_GET['organizer_id'] ?? null;
    $event_id = $_GET['event_id'] ?? '';
    
    if (!$organizer_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Organizer ID required']);
        return;
    }
    
    if ($event_id) {
        // Get rooms for specific event
        $stmt = $pdo->prepare("
            SELECT cr.*, e.title as event_title,
                   cr.current_participants as participant_count,
                   COUNT(DISTINCT cm.id) as message_count,
                   MAX(cm.created_at) as last_message_at
            FROM event_chat_rooms cr
            LEFT JOIN events e ON cr.event_id = e.id
            LEFT JOIN chat_messages cm ON cr.id = cm.chat_room_id
            WHERE cr.event_id = ? AND cr.is_active = 1
            GROUP BY cr.id
            ORDER BY cr.created_at DESC
        ");
        $stmt->execute([$event_id]);
    } else {
        // Get all rooms for organizer
        $stmt = $pdo->prepare("
            SELECT cr.*, e.title as event_title,
                   cr.current_participants as participant_count,
                   COUNT(DISTINCT cm.id) as message_count,
                   MAX(cm.created_at) as last_message_at
            FROM event_chat_rooms cr
            LEFT JOIN events e ON cr.event_id = e.id
            LEFT JOIN chat_messages cm ON cr.id = cm.chat_room_id
            WHERE cr.moderator_id = ? AND cr.is_active = 1
            GROUP BY cr.id
            ORDER BY cr.created_at DESC
            LIMIT 50
        ");
        $stmt->execute([$organizer_id]);
    }
    
    $rooms = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add room_type and status fields for compatibility
    foreach ($rooms as &$room) {
        $room['room_type'] = 'public'; // Default room type
        $room['status'] = $room['is_active'] ? 'active' : 'inactive';
        $room['max_participants'] = $room['max_participants'] ?? 1000;
    }
    
    echo json_encode([
        'success' => true,
        'rooms' => $rooms
    ]);
}

function createChatRoom() {
    global $pdo;
    
    // Temporarily bypass authentication for organizer dashboard
    $organizer_id = $_GET['organizer_id'] ?? $_POST['organizer_id'] ?? null;
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$organizer_id && isset($input['organizer_id'])) {
        $organizer_id = $input['organizer_id'];
    }
    
    if (!$organizer_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Organizer ID required']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['name', 'event_id', 'room_type'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Create chat room
        $stmt = $pdo->prepare("
            INSERT INTO event_chat_rooms (
                name, description, event_id, 
                is_moderated, moderator_id, max_participants, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $input['name'],
            $input['description'] ?? '',
            $input['event_id'],
            $input['moderated'] ?? true,
            $organizer_id,
            $input['max_participants'] ?? 1000
        ]);
        
        $room_id = $pdo->lastInsertId();
        
        // Send welcome message
        $stmt = $pdo->prepare("
            INSERT INTO chat_messages (chat_room_id, user_id, message_type, content, created_at)
            VALUES (?, ?, 'system', 'Welcome to the chat room! Please be respectful and follow community guidelines.', NOW())
        ");
        $stmt->execute([$room_id, $organizer_id]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Chat room created successfully',
            'room_id' => $room_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function joinChatRoom() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    $room_id = $input['room_id'] ?? '';
    
    if (!$room_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Room ID required']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check if room exists and is active
        $stmt = $pdo->prepare("
            SELECT cr.*, COUNT(crp.user_id) as current_participants
            FROM chat_rooms cr
            LEFT JOIN chat_room_participants crp ON cr.id = crp.room_id AND crp.status = 'active'
            WHERE cr.id = ? AND cr.status = 'active'
            GROUP BY cr.id
        ");
        $stmt->execute([$room_id]);
        $room = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$room) {
            http_response_code(404);
            echo json_encode(['error' => 'Room not found or inactive']);
            return;
        }
        
        // Check if room is full
        if ($room['current_participants'] >= $room['max_participants']) {
            http_response_code(409);
            echo json_encode(['error' => 'Room is full']);
            return;
        }
        
        // Check if user is banned
        $stmt = $pdo->prepare("
            SELECT status FROM chat_room_participants 
            WHERE room_id = ? AND user_id = ? AND status = 'banned'
        ");
        $stmt->execute([$room_id, $user['id']]);
        if ($stmt->fetch()) {
            http_response_code(403);
            echo json_encode(['error' => 'You are banned from this room']);
            return;
        }
        
        // Check if user needs event ticket for private rooms
        if ($room['room_type'] === 'private') {
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM event_tickets 
                WHERE event_id = ? AND user_id = ? AND status = 'confirmed'
            ");
            $stmt->execute([$room['event_id'], $user['id']]);
            if ($stmt->fetchColumn() == 0) {
                http_response_code(403);
                echo json_encode(['error' => 'Event ticket required to join this room']);
                return;
            }
        }
        
        // Add or update participant
        $stmt = $pdo->prepare("
            INSERT INTO chat_room_participants (room_id, user_id, role, status, joined_at)
            VALUES (?, ?, 'participant', 'active', NOW())
            ON DUPLICATE KEY UPDATE 
            status = 'active', joined_at = NOW()
        ");
        $stmt->execute([$room_id, $user['id']]);
        
        // Send join notification
        $stmt = $pdo->prepare("
            INSERT INTO chat_messages (room_id, user_id, message_type, content, created_at)
            VALUES (?, ?, 'system', CONCAT(?, ' joined the chat'), NOW())
        ");
        $stmt->execute([$room_id, $user['id'], $user['full_name']]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Successfully joined chat room'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function sendMessage() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['room_id', 'content'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check if user is active participant
        $stmt = $pdo->prepare("
            SELECT crp.role, cr.settings
            FROM chat_room_participants crp
            JOIN chat_rooms cr ON crp.room_id = cr.id
            WHERE crp.room_id = ? AND crp.user_id = ? AND crp.status = 'active'
        ");
        $stmt->execute([$input['room_id'], $user['id']]);
        $participation = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$participation) {
            http_response_code(403);
            echo json_encode(['error' => 'Not authorized to send messages in this room']);
            return;
        }
        
        $settings = json_decode($participation['settings'] ?? '{}', true);
        
        // Check rate limiting
        if (isset($settings['rate_limit'])) {
            $stmt = $pdo->prepare("
                SELECT COUNT(*) FROM chat_messages 
                WHERE room_id = ? AND user_id = ? 
                AND created_at >= DATE_SUB(NOW(), INTERVAL 1 MINUTE)
            ");
            $stmt->execute([$input['room_id'], $user['id']]);
            $recent_messages = $stmt->fetchColumn();
            
            if ($recent_messages >= $settings['rate_limit']) {
                http_response_code(429);
                echo json_encode(['error' => 'Rate limit exceeded. Please slow down.']);
                return;
            }
        }
        
        // Apply profanity filter if enabled
        $content = $input['content'];
        if ($settings['profanity_filter'] ?? true) {
            $content = applyProfanityFilter($content);
        }
        
        // Insert message
        $stmt = $pdo->prepare("
            INSERT INTO chat_messages (
                room_id, user_id, message_type, content, 
                media_url, reply_to, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $status = ($settings['moderated'] ?? true) && $participation['role'] === 'participant' ? 'pending' : 'approved';
        
        $stmt->execute([
            $input['room_id'],
            $user['id'],
            $input['message_type'] ?? 'text',
            $content,
            $input['media_url'] ?? null,
            $input['reply_to'] ?? null,
            $status
        ]);
        
        $message_id = $pdo->lastInsertId();
        
        // Update room last activity
        $stmt = $pdo->prepare("
            UPDATE chat_rooms SET last_activity = NOW() WHERE id = ?
        ");
        $stmt->execute([$input['room_id']]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Message sent successfully',
            'message_id' => $message_id,
            'status' => $status
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getChatMessages() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $room_id = $_GET['room_id'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(100, max(1, intval($_GET['limit'] ?? 50)));
    $offset = ($page - 1) * $limit;
    
    if (!$room_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Room ID required']);
        return;
    }
    
    // Check if user has access to room
    $stmt = $pdo->prepare("
        SELECT role FROM chat_room_participants 
        WHERE room_id = ? AND user_id = ? AND status = 'active'
    ");
    $stmt->execute([$room_id, $user['id']]);
    $participation = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$participation) {
        http_response_code(403);
        echo json_encode(['error' => 'Access denied to this room']);
        return;
    }
    
    // Get messages
    $status_filter = $participation['role'] === 'moderator' ? '' : "AND cm.status = 'approved'";
    
    $stmt = $pdo->prepare("
        SELECT cm.*, u.full_name as sender_name, u.profile_image,
               rm.content as reply_content, ru.full_name as reply_sender
        FROM chat_messages cm
        LEFT JOIN users u ON cm.user_id = u.id
        LEFT JOIN chat_messages rm ON cm.reply_to = rm.id
        LEFT JOIN users ru ON rm.user_id = ru.id
        WHERE cm.room_id = ? $status_filter
        ORDER BY cm.created_at DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$room_id, $limit, $offset]);
    $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Reverse to show oldest first
    $messages = array_reverse($messages);
    
    echo json_encode([
        'success' => true,
        'messages' => $messages
    ]);
}

function moderateMessage() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $message_id = $input['message_id'] ?? '';
    $action = $input['action'] ?? ''; // 'approve', 'reject', 'delete'
    
    if (!$message_id || !$action) {
        http_response_code(400);
        echo json_encode(['error' => 'Message ID and action required']);
        return;
    }
    
    // Check if user is moderator
    $stmt = $pdo->prepare("
        SELECT cm.room_id, crp.role
        FROM chat_messages cm
        JOIN chat_room_participants crp ON cm.room_id = crp.room_id
        WHERE cm.id = ? AND crp.user_id = ? AND crp.role IN ('moderator', 'admin')
    ");
    $stmt->execute([$message_id, $user['id']]);
    $moderation = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$moderation) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        switch ($action) {
            case 'approve':
                $stmt = $pdo->prepare("
                    UPDATE chat_messages 
                    SET status = 'approved', moderated_by = ?, moderated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$user['id'], $message_id]);
                break;
                
            case 'reject':
                $stmt = $pdo->prepare("
                    UPDATE chat_messages 
                    SET status = 'rejected', moderated_by = ?, moderated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$user['id'], $message_id]);
                break;
                
            case 'delete':
                $stmt = $pdo->prepare("
                    UPDATE chat_messages 
                    SET status = 'deleted', moderated_by = ?, moderated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$user['id'], $message_id]);
                break;
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => "Message {$action}d successfully"
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function applyProfanityFilter($content) {
    // Basic profanity filter - in production, use a comprehensive filter
    $profanity = ['damn', 'hell', 'stupid', 'idiot']; // Add more words
    $replacement = '***';
    
    foreach ($profanity as $word) {
        $content = str_ireplace($word, $replacement, $content);
    }
    
    return $content;
}

function getRoomParticipants() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $room_id = $_GET['room_id'] ?? '';
    
    if (!$room_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Room ID required']);
        return;
    }
    
    // Check access
    $stmt = $pdo->prepare("
        SELECT role FROM chat_room_participants 
        WHERE room_id = ? AND user_id = ? AND status = 'active'
    ");
    $stmt->execute([$room_id, $user['id']]);
    if (!$stmt->fetch()) {
        http_response_code(403);
        echo json_encode(['error' => 'Access denied']);
        return;
    }
    
    // Get participants
    $stmt = $pdo->prepare("
        SELECT crp.*, u.full_name, u.profile_image
        FROM chat_room_participants crp
        JOIN users u ON crp.user_id = u.id
        WHERE crp.room_id = ? AND crp.status = 'active'
        ORDER BY crp.role DESC, crp.joined_at ASC
    ");
    $stmt->execute([$room_id]);
    $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'participants' => $participants
    ]);
}
?>
