<?php
/**
 * Ghana National Event Management Platform
 * Event Management API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/logger.php'; // Include the Logger class

class EventAPI {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Handle API requests
     */
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        
        // Extract action from query parameter
        $action = $_GET['action'] ?? 'list';
        
        // Log API request details
        Logger::info("API Request", [
            'method' => $method,
            'action' => $action,
            'query_params' => $_GET,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);
        
        try {
            switch ($method) {
                case 'GET':
                    $this->handleGet($action);
                    break;
                case 'POST':
                    $this->handlePost($action);
                    break;
                case 'PUT':
                    $this->handlePut($action);
                    break;
                case 'DELETE':
                    $this->handleDelete($action);
                    break;
                default:
                    throw new Exception('Method not allowed');
            }
        } catch (Exception $e) {
            http_response_code(400);
            echo json_encode(['error' => $e->getMessage()]);
            Logger::error("API Error", [
                'error' => $e->getMessage(),
                'method' => $method,
                'action' => $action,
                'query_params' => $_GET,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            ]);
        }
    }

    /**
     * Handle GET requests
     */
    private function handleGet($action) {
        switch ($action) {
            case 'list':
                $this->getEvents();
                break;
            case 'details':
                $this->getEventDetails();
                break;
            case 'categories':
                $this->getEventCategories();
                break;
            case 'my-events':
                $this->getMyEvents();
                break;
            case 'pending-approval':
                $this->getPendingEvents();
                break;
            case 'analytics':
                $this->getEventAnalytics();
                break;
            case 'government-dashboard':
                $this->getGovernmentDashboard();
                break;
            case 'organizer-stats':
                $this->getOrganizerStats();
                break;
            case 'organizer-events':
                $this->getOrganizerEvents();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Handle POST requests
     */
    private function handlePost($action) {
        switch ($action) {
            case 'create':
                $this->createEvent();
                break;
            case 'approve':
                $this->approveEvent();
                break;
            case 'reject':
                $this->rejectEvent();
                break;
            case 'register':
                $this->registerForEvent();
                break;
            case 'feedback':
                $this->submitFeedback();
                break;
            case 'safety-report':
                $this->submitSafetyReport();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Handle PUT requests
     */
    private function handlePut($action) {
        switch ($action) {
            case 'update':
                $this->updateEvent();
                break;
            case 'cancel':
                $this->cancelEvent();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Get events list with filtering and pagination
     */
    private function getEvents() {
        $page = $_GET['page'] ?? 1;
        $limit = min($_GET['limit'] ?? Config::DEFAULT_PAGE_SIZE, Config::MAX_PAGE_SIZE);
        $offset = ($page - 1) * $limit;
        
        $category = $_GET['category'] ?? null;
        $city = $_GET['city'] ?? null;
        $region = $_GET['region'] ?? null;
        $dateFrom = $_GET['date_from'] ?? null;
        $dateTo = $_GET['date_to'] ?? null;
        $search = $_GET['search'] ?? null;
        $status = $_GET['status'] ?? 'approved,pending';

        $whereConditions = [];
        $params = [];

        // Handle status filtering - can be comma-separated values
        if ($status) {
            $statusArray = explode(',', $status);
            $statusPlaceholders = [];
            foreach ($statusArray as $i => $statusValue) {
                $statusPlaceholders[] = ":status$i";
                $params[":status$i"] = trim($statusValue);
            }
            $whereConditions[] = 'e.approval_status IN (' . implode(',', $statusPlaceholders) . ')';
        }

        if ($category) {
            $whereConditions[] = 'e.category_id = :category';
            $params[':category'] = $category;
        }

        if ($city) {
            $whereConditions[] = 'v.city = :city';
            $params[':city'] = $city;
        }

        if ($region) {
            $whereConditions[] = 'v.region = :region';
            $params[':region'] = $region;
        }

        if ($dateFrom) {
            $whereConditions[] = 'e.start_datetime >= :date_from';
            $params[':date_from'] = $dateFrom;
        }

        if ($dateTo) {
            $whereConditions[] = 'e.start_datetime <= :date_to';
            $params[':date_to'] = $dateTo;
        }

        if ($search) {
            $whereConditions[] = '(e.title LIKE :search OR e.description LIKE :search)';
            $params[':search'] = "%$search%";
        }

        $whereClause = implode(' AND ', $whereConditions);

        $query = "SELECT e.*, ec.name as category_name, ec.color as category_color,
                         v.name as venue_name, v.address as venue_address, v.city, v.region,
                         u.full_name as organizer_name,
                         COUNT(et.id) as registered_count
                  FROM events e
                  LEFT JOIN event_categories ec ON e.category_id = ec.id
                  LEFT JOIN venues v ON e.venue_id = v.id
                  LEFT JOIN users u ON e.organizer_id = u.id
                  LEFT JOIN event_tickets et ON e.id = et.event_id AND et.payment_status = 'completed'
                  WHERE $whereClause
                  GROUP BY e.id
                  ORDER BY e.start_datetime ASC
                  LIMIT :limit OFFSET :offset";

        $stmt = $this->conn->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();

        $events = $stmt->fetchAll();

        // Get total count
        $countQuery = "SELECT COUNT(DISTINCT e.id) as total
                      FROM events e
                      LEFT JOIN venues v ON e.venue_id = v.id
                      WHERE $whereClause";
        
        $countStmt = $this->conn->prepare($countQuery);
        foreach ($params as $key => $value) {
            $countStmt->bindValue($key, $value);
        }
        $countStmt->execute();
        $total = $countStmt->fetch()['total'];

        $response = [
            'success' => true,
            'data' => $events,
            'pagination' => [
                'page' => (int)$page,
                'limit' => (int)$limit,
                'total' => (int)$total,
                'pages' => ceil($total / $limit)
            ]
        ];

        Logger::info("API Response", [
            'method' => 'GET',
            'action' => 'list',
            'query_params' => $_GET,
            'response' => $response,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);

        echo json_encode($response);
    }

    /**
     * Get event details
     */
    private function getEventDetails() {
        $eventId = $_GET['id'] ?? null;
        if (!$eventId) {
            throw new Exception('Event ID is required');
        }

        $query = "SELECT e.*, ec.name as category_name, ec.color as category_color,
                         v.name as venue_name, v.address as venue_address, v.city, v.region,
                         v.latitude, v.longitude, v.amenities as venue_amenities,
                         u.full_name as organizer_name, u.email as organizer_email,
                         COUNT(et.id) as registered_count,
                         AVG(ef.rating) as average_rating,
                         COUNT(ef.id) as review_count
                  FROM events e
                  LEFT JOIN event_categories ec ON e.category_id = ec.id
                  LEFT JOIN venues v ON e.venue_id = v.id
                  LEFT JOIN users u ON e.organizer_id = u.id
                  LEFT JOIN event_tickets et ON e.id = et.event_id AND et.payment_status = 'completed'
                  LEFT JOIN event_feedback ef ON e.id = ef.event_id AND ef.is_public = 1
                  WHERE e.id = :event_id
                  GROUP BY e.id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':event_id', $eventId);
        $stmt->execute();

        $event = $stmt->fetch();
        if (!$event) {
            throw new Exception('Event not found');
        }

        // Get recent feedback
        $feedbackQuery = "SELECT ef.rating, ef.comment, ef.created_at, u.full_name as reviewer_name
                         FROM event_feedback ef
                         JOIN users u ON ef.user_id = u.id
                         WHERE ef.event_id = :event_id AND ef.is_public = 1
                         ORDER BY ef.created_at DESC
                         LIMIT 10";
        
        $feedbackStmt = $this->conn->prepare($feedbackQuery);
        $feedbackStmt->bindParam(':event_id', $eventId);
        $feedbackStmt->execute();
        $event['recent_feedback'] = $feedbackStmt->fetchAll();

        // Get permits status
        $permitsQuery = "SELECT permit_type, status, expiry_date
                        FROM event_permits
                        WHERE event_id = :event_id";
        
        $permitsStmt = $this->conn->prepare($permitsQuery);
        $permitsStmt->bindParam(':event_id', $eventId);
        $permitsStmt->execute();
        $event['permits'] = $permitsStmt->fetchAll();

        $response = [
            'success' => true,
            'data' => $event
        ];

        Logger::info("API Response", [
            'method' => 'GET',
            'action' => 'details',
            'query_params' => $_GET,
            'response' => $response,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);

        echo json_encode($response);
    }

    /**
     * Create new event
     */
    private function createEvent() {
        // Get user ID from token
        $headers = getallheaders();
        $token = null;
        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                $token = $matches[1];
            }
        }

        if (!$token) {
            http_response_code(401);
            echo json_encode(['error' => 'Authorization required']);
            return;
        }

        // For now, use organizer ID from token or default
        $organizerId = $_GET['organizer_id'] ?? 6; // Default to dorisanuson's ID
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $required = ['title', 'description', 'category_id', 'start_datetime', 'end_datetime'];
        foreach ($required as $field) {
            if (empty($input[$field])) {
                throw new Exception("$field is required");
            }
        }

        // Validate dates
        $startDate = new DateTime($input['start_datetime']);
        $endDate = new DateTime($input['end_datetime']);
        $now = new DateTime();

        if ($startDate <= $now) {
            throw new Exception('Event start date must be in the future');
        }

        if ($endDate <= $startDate) {
            throw new Exception('Event end date must be after start date');
        }

        // Insert event with simplified fields
        $query = "INSERT INTO events (
                    organizer_id, venue_id, category_id, title, description, 
                    start_datetime, end_datetime, max_attendees, ticket_price, 
                    is_free, approval_status, created_at
                  ) VALUES (
                    :organizer_id, :venue_id, :category_id, :title, :description,
                    :start_datetime, :end_datetime, :max_attendees, :ticket_price,
                    :is_free, 'pending', NOW()
                  )";

        $stmt = $this->conn->prepare($query);
        // Prepare variables for binding
        $maxAttendees = $input['max_attendees'] ?? null;
        $ticketPrice = $input['ticket_price'] ?? 0.00;
        $isFree = ($input['ticket_price'] ?? 0) == 0 ? 1 : 0;
        
        $stmt->bindParam(':organizer_id', $organizerId);
        $stmt->bindParam(':venue_id', $input['venue_id']);
        $stmt->bindParam(':category_id', $input['category_id']);
        $stmt->bindParam(':title', $input['title']);
        $stmt->bindParam(':description', $input['description']);
        $stmt->bindParam(':start_datetime', $input['start_datetime']);
        $stmt->bindParam(':end_datetime', $input['end_datetime']);
        $stmt->bindParam(':max_attendees', $maxAttendees);
        $stmt->bindParam(':ticket_price', $ticketPrice);
        $stmt->bindParam(':is_free', $isFree, PDO::PARAM_INT);

        if ($stmt->execute()) {
            $eventId = $this->conn->lastInsertId();
            
            // Create chat room if enabled
            if ($input['chat_room_enabled'] ?? true) {
                $this->createEventChatRoom($eventId, $input['title']);
            }

            // Send notification to government officials for approval
            $this->notifyGovernmentForApproval($eventId);

            $response = [
                'success' => true,
                'message' => 'Event created successfully and submitted for approval',
                'event_id' => $eventId
            ];

            Logger::info("API Response", [
                'method' => 'POST',
                'action' => 'create',
                'request_body' => $input,
                'response' => $response,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            ]);

            echo json_encode($response);
        } else {
            throw new Exception('Failed to create event');
        }
    }

    /**
     * Approve event (Government officials only)
     */
    private function approveEvent() {
        $user = AuthMiddleware::requirePermission('approve_events');
        
        $input = json_decode(file_get_contents('php://input'), true);
        $eventId = $input['event_id'] ?? null;
        
        if (!$eventId) {
            throw new Exception('Event ID is required');
        }

        $query = "UPDATE events SET 
                  approval_status = 'approved',
                  approved_by = :approved_by,
                  approval_date = NOW(),
                  government_verified = 1
                  WHERE id = :event_id AND approval_status = 'pending'";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':approved_by', $user['id']);
        $stmt->bindParam(':event_id', $eventId);

        if ($stmt->execute() && $stmt->rowCount() > 0) {
            // Notify event organizer
            $this->notifyEventApproval($eventId, 'approved');
            
            $response = [
                'success' => true,
                'message' => 'Event approved successfully'
            ];

            Logger::info("API Response", [
                'method' => 'POST',
                'action' => 'approve',
                'request_body' => $input,
                'response' => $response,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            ]);

            echo json_encode($response);
        } else {
            throw new Exception('Event not found or already processed');
        }
    }

    /**
     * Get events pending approval (Government officials only)
     */
    private function getPendingEvents() {
        $user = AuthMiddleware::requirePermission('approve_events');

        $query = "SELECT e.*, ec.name as category_name,
                         v.name as venue_name, v.address as venue_address,
                         u.full_name as organizer_name, u.email as organizer_email
                  FROM events e
                  LEFT JOIN event_categories ec ON e.category_id = ec.id
                  LEFT JOIN venues v ON e.venue_id = v.id
                  LEFT JOIN users u ON e.organizer_id = u.id
                  WHERE e.approval_status = 'pending'
                  ORDER BY e.created_at ASC";

        $stmt = $this->conn->prepare($query);
        $stmt->execute();

        $response = [
            'success' => true,
            'data' => $stmt->fetchAll()
        ];

        Logger::info("API Response", [
            'method' => 'GET',
            'action' => 'pending-approval',
            'response' => $response,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);

        echo json_encode($response);
    }

    /**
     * Get government dashboard data
     */
    private function getGovernmentDashboard() {
        $user = AuthMiddleware::requirePermission('view_analytics');

        // Get event statistics
        $statsQuery = "SELECT 
                       COUNT(*) as total_events,
                       COUNT(CASE WHEN approval_status = 'pending' THEN 1 END) as pending_events,
                       COUNT(CASE WHEN approval_status = 'approved' THEN 1 END) as approved_events,
                       COUNT(CASE WHEN start_datetime >= CURDATE() THEN 1 END) as upcoming_events,
                       COUNT(CASE WHEN start_datetime < CURDATE() AND end_datetime > CURDATE() THEN 1 END) as ongoing_events
                       FROM events";

        $statsStmt = $this->conn->prepare($statsQuery);
        $statsStmt->execute();
        $stats = $statsStmt->fetch();

        // Get revenue statistics
        $revenueQuery = "SELECT 
                        SUM(amount) as total_revenue,
                        SUM(tax_amount) as total_tax,
                        COUNT(*) as total_transactions
                        FROM payment_transactions 
                        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)";

        $revenueStmt = $this->conn->prepare($revenueQuery);
        $revenueStmt->execute();
        $revenue = $revenueStmt->fetch();

        // Get safety reports
        $safetyQuery = "SELECT 
                       COUNT(*) as total_reports,
                       COUNT(CASE WHEN severity = 'critical' THEN 1 END) as critical_reports,
                       COUNT(CASE WHEN status = 'reported' THEN 1 END) as pending_reports
                       FROM safety_reports 
                       WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)";

        $safetyStmt = $this->conn->prepare($safetyQuery);
        $safetyStmt->execute();
        $safety = $safetyStmt->fetch();

        // Get recent events by region
        $regionQuery = "SELECT v.region, COUNT(e.id) as event_count
                       FROM events e
                       JOIN venues v ON e.venue_id = v.id
                       WHERE e.start_datetime >= CURDATE()
                       GROUP BY v.region
                       ORDER BY event_count DESC";

        $regionStmt = $this->conn->prepare($regionQuery);
        $regionStmt->execute();
        $regionData = $regionStmt->fetchAll();

        $response = [
            'success' => true,
            'data' => [
                'statistics' => $stats,
                'revenue' => $revenue,
                'safety' => $safety,
                'regions' => $regionData
            ]
        ];

        Logger::info("API Response", [
            'method' => 'GET',
            'action' => 'government-dashboard',
            'response' => $response,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);

        echo json_encode($response);
    }

    /**
     * Submit safety report
     */
    private function submitSafetyReport() {
        $user = AuthMiddleware::requireAuth();
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        $required = ['event_id', 'report_type', 'severity', 'description'];
        foreach ($required as $field) {
            if (empty($input[$field])) {
                throw new Exception("$field is required");
            }
        }

        $query = "INSERT INTO safety_reports (
                    event_id, reporter_id, report_type, severity, location_description,
                    latitude, longitude, description, images, is_anonymous
                  ) VALUES (
                    :event_id, :reporter_id, :report_type, :severity, :location_description,
                    :latitude, :longitude, :description, :images, :is_anonymous
                  )";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':event_id', $input['event_id']);
        $stmt->bindParam(':reporter_id', $input['is_anonymous'] ? null : $user['id']);
        $stmt->bindParam(':report_type', $input['report_type']);
        $stmt->bindParam(':severity', $input['severity']);
        $stmt->bindParam(':location_description', $input['location_description'] ?? null);
        $stmt->bindParam(':latitude', $input['latitude'] ?? null);
        $stmt->bindParam(':longitude', $input['longitude'] ?? null);
        $stmt->bindParam(':description', $input['description']);
        $stmt->bindParam(':images', json_encode($input['images'] ?? []));
        $stmt->bindParam(':is_anonymous', $input['is_anonymous'] ?? false, PDO::PARAM_BOOL);

        if ($stmt->execute()) {
            $reportId = $this->conn->lastInsertId();
            
            // Notify emergency responders for critical reports
            if ($input['severity'] === 'critical') {
                $this->notifyEmergencyResponders($reportId);
            }

            $response = [
                'success' => true,
                'message' => 'Safety report submitted successfully',
                'report_id' => $reportId
            ];

            Logger::info("API Response", [
                'method' => 'POST',
                'action' => 'safety-report',
                'request_body' => $input,
                'response' => $response,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
                'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
            ]);

            echo json_encode($response);
        } else {
            throw new Exception('Failed to submit safety report');
        }
    }

    /**
     * Get event categories
     */
    private function getEventCategories() {
        $query = "SELECT * FROM event_categories ORDER BY name";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();

        $response = [
            'success' => true,
            'data' => $stmt->fetchAll()
        ];

        Logger::info("API Response", [
            'method' => 'GET',
            'action' => 'categories',
            'response' => $response,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);

        echo json_encode($response);
    }

    /**
     * Helper methods
     */
    private function createEventChatRoom($eventId, $eventTitle) {
        $query = "INSERT INTO event_chat_rooms (event_id, name, description) 
                 VALUES (:event_id, :name, :description)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':event_id', $eventId);
        $stmt->bindValue(':name', "Chat - $eventTitle");
        $stmt->bindValue(':description', "Official chat room for $eventTitle");
        $stmt->execute();
    }

    private function notifyGovernmentForApproval($eventId) {
        // Implementation for notifying government officials
        // This would send notifications to relevant government users
    }

    private function notifyEventApproval($eventId, $status) {
        // Implementation for notifying event organizer about approval/rejection
    }

    private function notifyEmergencyResponders($reportId) {
        // Implementation for notifying emergency responders about critical safety reports
    }

    /**
     * Get organizer statistics
     */
    private function getOrganizerStats() {
        // Get user ID from token
        $headers = getallheaders();
        $token = null;
        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                $token = $matches[1];
            }
        }

        if (!$token) {
            http_response_code(401);
            echo json_encode(['error' => 'Authorization required']);
            return;
        }

        // For now, get organizer ID from URL parameter or use a default
        $organizerId = $_GET['organizer_id'] ?? 6; // Default to dorisanuson's ID

        try {
            // Get event statistics for this organizer
            $statsQuery = "SELECT 
                COUNT(*) as total_events,
                SUM(CASE WHEN approval_status = 'approved' THEN 1 ELSE 0 END) as approved_events,
                SUM(CASE WHEN approval_status = 'pending' THEN 1 ELSE 0 END) as pending_events,
                SUM(current_attendees) as total_attendees,
                SUM(CASE WHEN is_free = 0 THEN ticket_price * current_attendees ELSE 0 END) as total_revenue
                FROM events 
                WHERE organizer_id = :organizer_id";

            $stmt = $this->conn->prepare($statsQuery);
            $stmt->bindParam(':organizer_id', $organizerId);
            $stmt->execute();
            $stats = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'data' => [
                    'total_events' => (int)$stats['total_events'],
                    'total_tickets' => (int)$stats['total_attendees'],
                    'total_revenue' => (float)$stats['total_revenue'],
                    'pending_approvals' => (int)$stats['pending_events'],
                    'approved_events' => (int)$stats['approved_events']
                ]
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to fetch organizer statistics']);
        }
    }

    /**
     * Get organizer events
     */
    private function getOrganizerEvents() {
        // Get user ID from token
        $headers = getallheaders();
        $token = null;
        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                $token = $matches[1];
            }
        }

        // Temporarily bypass authentication for debugging
        if (!$token) {
            // For debugging purposes, allow access without token
            // TODO: Re-enable authentication after fixing the issue
            // http_response_code(401);
            // echo json_encode(['error' => 'Authorization required', 'success' => false]);
            // return;
        }

        // For now, get organizer ID from URL parameter or use a default
        $organizerId = $_GET['organizer_id'] ?? 6; // Default to dorisanuson's ID
        $limit = $_GET['limit'] ?? 10;

        try {
            $query = "SELECT e.*, ec.name as category_name, v.name as venue_name
                     FROM events e
                     LEFT JOIN event_categories ec ON e.category_id = ec.id
                     LEFT JOIN venues v ON e.venue_id = v.id
                     WHERE e.organizer_id = :organizer_id
                     ORDER BY e.created_at DESC
                     LIMIT :limit";

            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':organizer_id', $organizerId);
            $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
            $stmt->execute();
            $events = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'data' => $events
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to fetch organizer events', 'success' => false, 'debug' => $e->getMessage()]);
        }
    }
}

// Handle the request
$api = new EventAPI();
$api->handleRequest();
?>
