<?php
require_once '../config/database.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'dashboard':
            if ($method === 'GET') {
                getGovernmentDashboard();
            }
            break;
        
        case 'analytics':
            if ($method === 'GET') {
                getAnalytics();
            }
            break;
            
        case 'event-monitoring':
            if ($method === 'GET') {
                getEventMonitoring();
            }
            break;
            
        case 'revenue-tracking':
            if ($method === 'GET') {
                getRevenueTracking();
            }
            break;
            
        case 'compliance-report':
            if ($method === 'GET') {
                getComplianceReport();
            }
            break;
            
        case 'emergency-overview':
            if ($method === 'GET') {
                getEmergencyOverview();
            }
            break;
            
        case 'tourism-metrics':
            if ($method === 'GET') {
                getTourismMetrics();
            }
            break;
            
        case 'regional-analysis':
            if ($method === 'GET') {
                getRegionalAnalysis();
            }
            break;
            
        case 'export-data':
            if ($method === 'POST') {
                exportData();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function getGovernmentDashboard() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'government_dashboard')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    // Get comprehensive dashboard data
    $dashboard = [];
    
    // Event Statistics
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_events,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_approval,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_events,
            SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_events,
            SUM(CASE WHEN start_date >= CURDATE() THEN 1 ELSE 0 END) as upcoming_events,
            SUM(CASE WHEN start_date >= CURDATE() AND start_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as events_this_week
        FROM events
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
    ");
    $stmt->execute();
    $dashboard['event_stats'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Revenue Statistics
    $stmt = $pdo->prepare("
        SELECT 
            SUM(amount) as total_revenue,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN amount ELSE 0 END) as monthly_revenue,
            COUNT(*) as total_transactions,
            AVG(amount) as avg_transaction_value
        FROM payment_transactions
        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
    ");
    $stmt->execute();
    $dashboard['revenue_stats'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Safety & Security
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_reports,
            SUM(CASE WHEN severity = 'high' THEN 1 ELSE 0 END) as high_priority,
            SUM(CASE WHEN status = 'resolved' THEN 1 ELSE 0 END) as resolved_reports,
            AVG(TIMESTAMPDIFF(HOUR, created_at, responded_at)) as avg_response_hours
        FROM safety_reports
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    ");
    $stmt->execute();
    $dashboard['safety_stats'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // User Growth
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_users,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as new_users_month,
            SUM(CASE WHEN last_login >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as active_users_week
        FROM users
        WHERE status = 'active'
    ");
    $stmt->execute();
    $dashboard['user_stats'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Venue & Vendor Stats
    $stmt = $pdo->prepare("
        SELECT 
            (SELECT COUNT(*) FROM venues WHERE status = 'approved') as approved_venues,
            (SELECT COUNT(*) FROM venues WHERE status = 'pending') as pending_venues,
            (SELECT COUNT(*) FROM vendors WHERE status = 'approved') as approved_vendors,
            (SELECT COUNT(*) FROM vendors WHERE status = 'pending') as pending_vendors
    ");
    $stmt->execute();
    $dashboard['venue_vendor_stats'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Recent Activities
    $stmt = $pdo->prepare("
        SELECT 'event' as type, title as description, created_at, status
        FROM events
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        UNION ALL
        SELECT 'safety_report' as type, CONCAT('Safety report: ', incident_type) as description, created_at, status
        FROM safety_reports
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        UNION ALL
        SELECT 'venue' as type, CONCAT('Venue registration: ', name) as description, created_at, status
        FROM venues
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        ORDER BY created_at DESC
        LIMIT 20
    ");
    $stmt->execute();
    $dashboard['recent_activities'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'dashboard' => $dashboard
    ]);
}

function getAnalytics() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'analytics_access')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $period = $_GET['period'] ?? '12months';
    $analytics = [];
    
    // Event trends over time
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            COUNT(*) as event_count,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_count
        FROM events
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $analytics['event_trends'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Revenue trends
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            SUM(amount) as total_revenue,
            COUNT(*) as transaction_count,
            AVG(amount) as avg_amount
        FROM payment_transactions
        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $analytics['revenue_trends'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Event categories popularity
    $stmt = $pdo->prepare("
        SELECT 
            ec.name as category,
            COUNT(e.id) as event_count,
            SUM(et.quantity) as total_tickets_sold
        FROM event_categories ec
        LEFT JOIN events e ON ec.id = e.category_id
        LEFT JOIN event_tickets et ON e.id = et.event_id AND et.status = 'confirmed'
        WHERE e.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY ec.id, ec.name
        ORDER BY event_count DESC
    ");
    $stmt->execute();
    $analytics['category_popularity'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Regional distribution
    $stmt = $pdo->prepare("
        SELECT 
            region,
            COUNT(*) as event_count,
            SUM(expected_attendees) as total_expected_attendees
        FROM events
        WHERE status = 'approved' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY region
        ORDER BY event_count DESC
    ");
    $stmt->execute();
    $analytics['regional_distribution'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // User engagement metrics
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            COUNT(*) as new_registrations
        FROM users
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $analytics['user_growth'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'analytics' => $analytics
    ]);
}

function getEventMonitoring() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'event_monitoring')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $status = $_GET['status'] ?? '';
    $region = $_GET['region'] ?? '';
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($status) {
        $where_conditions[] = 'e.status = ?';
        $params[] = $status;
    }
    
    if ($region) {
        $where_conditions[] = 'e.region = ?';
        $params[] = $region;
    }
    
    if ($date_from) {
        $where_conditions[] = 'e.start_date >= ?';
        $params[] = $date_from;
    }
    
    if ($date_to) {
        $where_conditions[] = 'e.start_date <= ?';
        $params[] = $date_to;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get events with monitoring data
    $stmt = $pdo->prepare("
        SELECT 
            e.*,
            u.full_name as organizer_name,
            v.name as venue_name,
            ec.name as category_name,
            COUNT(DISTINCT et.id) as tickets_sold,
            COUNT(DISTINCT sr.id) as safety_reports,
            AVG(ef.rating) as avg_rating
        FROM events e
        LEFT JOIN users u ON e.organizer_id = u.id
        LEFT JOIN venues v ON e.venue_id = v.id
        LEFT JOIN event_categories ec ON e.category_id = ec.id
        LEFT JOIN event_tickets et ON e.id = et.event_id AND et.status = 'confirmed'
        LEFT JOIN safety_reports sr ON e.id = sr.event_id
        LEFT JOIN event_feedback ef ON e.id = ef.event_id
        WHERE $where_clause
        GROUP BY e.id
        ORDER BY e.start_date DESC
        LIMIT 100
    ");
    $stmt->execute($params);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Add risk assessment for each event
    foreach ($events as &$event) {
        $risk_score = calculateEventRiskScore($event);
        $event['risk_score'] = $risk_score;
        $event['risk_level'] = getRiskLevel($risk_score);
    }
    
    echo json_encode([
        'success' => true,
        'events' => $events
    ]);
}

function getRevenueTracking() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'revenue_tracking')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $revenue = [];
    
    // Total revenue breakdown
    $stmt = $pdo->prepare("
        SELECT 
            payment_method,
            SUM(amount) as total_amount,
            COUNT(*) as transaction_count,
            AVG(amount) as avg_amount
        FROM payment_transactions
        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY payment_method
        ORDER BY total_amount DESC
    ");
    $stmt->execute();
    $revenue['by_payment_method'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Revenue by event category
    $stmt = $pdo->prepare("
        SELECT 
            ec.name as category,
            SUM(pt.amount) as total_revenue,
            COUNT(DISTINCT e.id) as event_count,
            AVG(pt.amount) as avg_ticket_price
        FROM event_categories ec
        JOIN events e ON ec.id = e.category_id
        JOIN event_tickets et ON e.id = et.event_id
        JOIN payment_transactions pt ON et.payment_id = pt.id
        WHERE pt.status = 'completed' AND pt.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY ec.id, ec.name
        ORDER BY total_revenue DESC
    ");
    $stmt->execute();
    $revenue['by_category'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Tax and levy calculations
    $stmt = $pdo->prepare("
        SELECT 
            SUM(amount * 0.125) as vat_collected,
            SUM(amount * 0.05) as tourism_levy,
            SUM(amount * 0.02) as platform_fee,
            SUM(amount) as gross_revenue
        FROM payment_transactions
        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
    ");
    $stmt->execute();
    $revenue['tax_breakdown'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Monthly revenue trends
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            SUM(amount) as revenue,
            COUNT(*) as transactions,
            SUM(amount * 0.125) as vat,
            SUM(amount * 0.05) as tourism_levy
        FROM payment_transactions
        WHERE status = 'completed' AND created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $revenue['monthly_trends'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'revenue' => $revenue
    ]);
}

function getTourismMetrics() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'tourism_analytics')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $metrics = [];
    
    // Tourist vs Local attendance
    $stmt = $pdo->prepare("
        SELECT 
            CASE 
                WHEN u.country != 'Ghana' THEN 'International'
                WHEN u.region != e.region THEN 'Domestic Tourist'
                ELSE 'Local'
            END as visitor_type,
            COUNT(*) as attendee_count,
            SUM(pt.amount) as total_spent
        FROM event_tickets et
        JOIN users u ON et.user_id = u.id
        JOIN events e ON et.event_id = e.id
        JOIN payment_transactions pt ON et.payment_id = pt.id
        WHERE et.status = 'confirmed' AND pt.status = 'completed'
        AND et.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY visitor_type
    ");
    $stmt->execute();
    $metrics['visitor_breakdown'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Popular destinations
    $stmt = $pdo->prepare("
        SELECT 
            e.region,
            e.city,
            COUNT(DISTINCT e.id) as event_count,
            COUNT(et.id) as total_attendees,
            SUM(pt.amount) as revenue_generated
        FROM events e
        JOIN event_tickets et ON e.id = et.event_id
        JOIN payment_transactions pt ON et.payment_id = pt.id
        WHERE e.status = 'approved' AND et.status = 'confirmed' 
        AND pt.status = 'completed'
        AND e.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY e.region, e.city
        ORDER BY total_attendees DESC
        LIMIT 20
    ");
    $stmt->execute();
    $metrics['popular_destinations'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Seasonal trends
    $stmt = $pdo->prepare("
        SELECT 
            MONTH(e.start_date) as month,
            MONTHNAME(e.start_date) as month_name,
            COUNT(DISTINCT e.id) as events,
            COUNT(et.id) as attendees,
            SUM(pt.amount) as revenue
        FROM events e
        JOIN event_tickets et ON e.id = et.event_id
        JOIN payment_transactions pt ON et.payment_id = pt.id
        WHERE e.status = 'approved' AND et.status = 'confirmed'
        AND pt.status = 'completed'
        AND e.start_date >= DATE_SUB(NOW(), INTERVAL 24 MONTH)
        GROUP BY MONTH(e.start_date), MONTHNAME(e.start_date)
        ORDER BY month
    ");
    $stmt->execute();
    $metrics['seasonal_trends'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Economic impact
    $stmt = $pdo->prepare("
        SELECT 
            SUM(pt.amount) as direct_revenue,
            COUNT(DISTINCT et.user_id) as unique_visitors,
            COUNT(DISTINCT e.id) as total_events,
            AVG(e.expected_attendees) as avg_event_size,
            SUM(pt.amount) * 2.5 as estimated_economic_impact
        FROM events e
        JOIN event_tickets et ON e.id = et.event_id
        JOIN payment_transactions pt ON et.payment_id = pt.id
        WHERE e.status = 'approved' AND et.status = 'confirmed'
        AND pt.status = 'completed'
        AND e.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
    ");
    $stmt->execute();
    $metrics['economic_impact'] = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'tourism_metrics' => $metrics
    ]);
}

function calculateEventRiskScore($event) {
    $risk_score = 0;
    
    // Base risk factors
    if ($event['expected_attendees'] > 5000) $risk_score += 3;
    elseif ($event['expected_attendees'] > 1000) $risk_score += 2;
    elseif ($event['expected_attendees'] > 500) $risk_score += 1;
    
    // Safety reports factor
    if ($event['safety_reports'] > 5) $risk_score += 3;
    elseif ($event['safety_reports'] > 2) $risk_score += 2;
    elseif ($event['safety_reports'] > 0) $risk_score += 1;
    
    // Rating factor (inverse)
    if ($event['avg_rating'] && $event['avg_rating'] < 3) $risk_score += 2;
    elseif ($event['avg_rating'] && $event['avg_rating'] < 4) $risk_score += 1;
    
    // Event type factors
    if (stripos($event['title'], 'concert') !== false) $risk_score += 1;
    if (stripos($event['title'], 'festival') !== false) $risk_score += 1;
    
    return min($risk_score, 10); // Cap at 10
}

function getRiskLevel($score) {
    if ($score >= 7) return 'High';
    elseif ($score >= 4) return 'Medium';
    else return 'Low';
}
?>
