<?php
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/logger.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Initialize database connection
$db = new Database();
$pdo = $db->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? 'list';

try {
    switch ($action) {
        case 'list':
            if ($method === 'GET') {
                getUserNotifications();
            }
            break;
        
        case 'send':
            if ($method === 'POST') {
                sendNotification();
            }
            break;
            
        case 'mark-read':
            if ($method === 'PUT') {
                markAsRead();
            }
            break;
            
        case 'mark-all-read':
            if ($method === 'PUT') {
                markAllAsRead();
            }
            break;
            
        case 'delete':
            if ($method === 'DELETE') {
                deleteNotification();
            }
            break;
            
        case 'preferences':
            if ($method === 'GET') {
                getNotificationPreferences();
            } elseif ($method === 'PUT') {
                updateNotificationPreferences();
            }
            break;
            
        case 'subscribe-push':
            if ($method === 'POST') {
                subscribePushNotifications();
            }
            break;
            
        case 'unsubscribe-push':
            if ($method === 'POST') {
                unsubscribePushNotifications();
            }
            break;
            
        case 'broadcast':
            if ($method === 'POST') {
                broadcastNotification();
            }
            break;
            
        case 'stats':
            if ($method === 'GET') {
                getNotificationStats();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function getUserNotifications() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? 6; // Default to organizer
    $limit = $_GET['limit'] ?? 20;
    
    // Sample notifications for event organizer
    $notifications = [
        [
            'id' => 1,
            'type' => 'event_approval',
            'title' => 'Event Approved',
            'message' => 'Your event "Ghana Music Festival" has been approved by government officials.',
            'created_at' => date('Y-m-d H:i:s', strtotime('-2 hours')),
            'is_read' => false,
            'priority' => 'high'
        ],
        [
            'id' => 2,
            'type' => 'payment_confirmation',
            'title' => 'Payment Received',
            'message' => 'Payment of GHS 50 received for ticket purchase.',
            'created_at' => date('Y-m-d H:i:s', strtotime('-5 hours')),
            'is_read' => false,
            'priority' => 'medium'
        ],
        [
            'id' => 3,
            'type' => 'safety_alert',
            'title' => 'Safety Report Filed',
            'message' => 'A safety report has been filed for your event. Please review and take necessary action.',
            'created_at' => date('Y-m-d H:i:s', strtotime('-8 hours')),
            'is_read' => false,
            'priority' => 'urgent'
        ],
        [
            'id' => 4,
            'type' => 'system_update',
            'title' => 'New Feature Available',
            'message' => 'Chat rooms and real-time messaging are now available for your events.',
            'created_at' => date('Y-m-d H:i:s', strtotime('-1 day')),
            'is_read' => true,
            'priority' => 'low'
        ],
        [
            'id' => 5,
            'type' => 'emergency',
            'title' => 'Emergency Contact Update',
            'message' => 'Emergency contact information has been updated for your region.',
            'created_at' => date('Y-m-d H:i:s', strtotime('-2 days')),
            'is_read' => true,
            'priority' => 'high'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'notifications' => $notifications
    ]);
}

function getNotificationStats() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? 6;
    
    // Sample stats for organizer dashboard
    $stats = [
        'total' => 12,
        'unread' => 3,
        'urgent' => 1,
        'today' => 2
    ];
    
    echo json_encode([
        'success' => true,
        'stats' => $stats
    ]);
}

function sendNotification() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'send_notifications')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['recipient_id', 'type', 'title', 'message'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Insert notification
        $stmt = $pdo->prepare("
            INSERT INTO notifications (
                user_id, type, title, message, data, 
                sender_id, priority, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $input['recipient_id'],
            $input['type'],
            $input['title'],
            $input['message'],
            json_encode($input['data'] ?? []),
            $user['id'],
            $input['priority'] ?? 'normal'
        ]);
        
        $notification_id = $pdo->lastInsertId();
        
        // Check user's notification preferences
        $stmt = $pdo->prepare("
            SELECT email_notifications, sms_notifications, push_notifications 
            FROM user_notification_preferences 
            WHERE user_id = ?
        ");
        $stmt->execute([$input['recipient_id']]);
        $preferences = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Send via different channels based on preferences
        if ($preferences) {
            if ($preferences['email_notifications']) {
                queueEmailNotification($input['recipient_id'], $input['title'], $input['message']);
            }
            
            if ($preferences['sms_notifications'] && ($input['priority'] === 'high' || $input['priority'] === 'urgent')) {
                queueSMSNotification($input['recipient_id'], $input['message']);
            }
            
            if ($preferences['push_notifications']) {
                sendPushNotification($input['recipient_id'], $input['title'], $input['message'], $input['data'] ?? []);
            }
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Notification sent successfully',
            'notification_id' => $notification_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function markAsRead() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    $notification_id = $input['notification_id'] ?? '';
    
    if (!$notification_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Notification ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        UPDATE notifications 
        SET read_at = NOW() 
        WHERE id = ? AND user_id = ? AND read_at IS NULL
    ");
    $stmt->execute([$notification_id, $user['id']]);
    
    if ($stmt->rowCount() > 0) {
        echo json_encode([
            'success' => true,
            'message' => 'Notification marked as read'
        ]);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Notification not found or already read']);
    }
}

function markAllAsRead() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    
    $stmt = $pdo->prepare("
        UPDATE notifications 
        SET read_at = NOW() 
        WHERE user_id = ? AND read_at IS NULL
    ");
    $stmt->execute([$user['id']]);
    
    echo json_encode([
        'success' => true,
        'message' => 'All notifications marked as read',
        'updated_count' => $stmt->rowCount()
    ]);
}

function deleteNotification() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $notification_id = $_GET['id'] ?? '';
    
    if (!$notification_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Notification ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        DELETE FROM notifications 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$notification_id, $user['id']]);
    
    if ($stmt->rowCount() > 0) {
        echo json_encode([
            'success' => true,
            'message' => 'Notification deleted'
        ]);
    } else {
        http_response_code(404);
        echo json_encode(['error' => 'Notification not found']);
    }
}

function getNotificationPreferences() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    
    $stmt = $pdo->prepare("
        SELECT * FROM user_notification_preferences 
        WHERE user_id = ?
    ");
    $stmt->execute([$user['id']]);
    $preferences = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$preferences) {
        // Create default preferences
        $stmt = $pdo->prepare("
            INSERT INTO user_notification_preferences (
                user_id, email_notifications, sms_notifications, 
                push_notifications, event_reminders, safety_alerts,
                payment_confirmations, chat_messages, created_at
            ) VALUES (?, 1, 1, 1, 1, 1, 1, 0, NOW())
        ");
        $stmt->execute([$user['id']]);
        
        $preferences = [
            'user_id' => $user['id'],
            'email_notifications' => 1,
            'sms_notifications' => 1,
            'push_notifications' => 1,
            'event_reminders' => 1,
            'safety_alerts' => 1,
            'payment_confirmations' => 1,
            'chat_messages' => 0
        ];
    }
    
    echo json_encode([
        'success' => true,
        'preferences' => $preferences
    ]);
}

function updateNotificationPreferences() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $stmt = $pdo->prepare("
        UPDATE user_notification_preferences 
        SET email_notifications = ?, sms_notifications = ?, 
            push_notifications = ?, event_reminders = ?,
            safety_alerts = ?, payment_confirmations = ?,
            chat_messages = ?, updated_at = NOW()
        WHERE user_id = ?
    ");
    
    $stmt->execute([
        $input['email_notifications'] ?? 1,
        $input['sms_notifications'] ?? 1,
        $input['push_notifications'] ?? 1,
        $input['event_reminders'] ?? 1,
        $input['safety_alerts'] ?? 1,
        $input['payment_confirmations'] ?? 1,
        $input['chat_messages'] ?? 0,
        $user['id']
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Notification preferences updated'
    ]);
}

function subscribePushNotifications() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['endpoint', 'keys'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Store push subscription
        $stmt = $pdo->prepare("
            INSERT INTO push_subscriptions (
                user_id, endpoint, p256dh_key, auth_key, 
                user_agent, created_at
            ) VALUES (?, ?, ?, ?, ?, NOW())
            ON DUPLICATE KEY UPDATE
            p256dh_key = VALUES(p256dh_key),
            auth_key = VALUES(auth_key),
            user_agent = VALUES(user_agent),
            updated_at = NOW()
        ");
        
        $stmt->execute([
            $user['id'],
            $input['endpoint'],
            $input['keys']['p256dh'] ?? '',
            $input['keys']['auth'] ?? '',
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
        // Update user preferences to enable push notifications
        $stmt = $pdo->prepare("
            UPDATE user_notification_preferences 
            SET push_notifications = 1 
            WHERE user_id = ?
        ");
        $stmt->execute([$user['id']]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Push notifications subscribed successfully'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function broadcastNotification() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'broadcast_notifications')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['type', 'title', 'message', 'target_audience'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Determine target users based on audience
        $target_query = "";
        $params = [];
        
        switch ($input['target_audience']) {
            case 'all_users':
                $target_query = "SELECT id FROM users WHERE status = 'active'";
                break;
                
            case 'event_organizers':
                $target_query = "
                    SELECT DISTINCT u.id FROM users u
                    JOIN user_roles ur ON u.id = ur.user_id
                    WHERE ur.role_key = 'event_organizer' AND u.status = 'active'
                ";
                break;
                
            case 'government_officials':
                $target_query = "
                    SELECT DISTINCT u.id FROM users u
                    JOIN user_roles ur ON u.id = ur.user_id
                    WHERE ur.role_key = 'government_official' AND u.status = 'active'
                ";
                break;
                
            case 'venue_owners':
                $target_query = "
                    SELECT DISTINCT u.id FROM users u
                    JOIN venues v ON u.id = v.owner_id
                    WHERE u.status = 'active'
                ";
                break;
                
            case 'recent_users':
                $target_query = "
                    SELECT id FROM users 
                    WHERE status = 'active' AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                ";
                break;
        }
        
        if ($target_query) {
            // Insert notifications for all target users
            $stmt = $pdo->prepare("
                INSERT INTO notifications (user_id, type, title, message, data, sender_id, priority, created_at)
                SELECT id, ?, ?, ?, ?, ?, ?, NOW()
                FROM ($target_query) AS target_users
            ");
            
            $stmt->execute([
                $input['type'],
                $input['title'],
                $input['message'],
                json_encode($input['data'] ?? []),
                $user['id'],
                $input['priority'] ?? 'normal'
            ]);
            
            $notification_count = $stmt->rowCount();
            
            // Queue push notifications for high priority broadcasts
            if ($input['priority'] === 'high' || $input['priority'] === 'urgent') {
                queueBroadcastPushNotifications($input['title'], $input['message'], $input['target_audience']);
            }
            
            $pdo->commit();
            
            echo json_encode([
                'success' => true,
                'message' => 'Broadcast notification sent successfully',
                'recipients_count' => $notification_count
            ]);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid target audience']);
        }
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}


// Helper functions for external integrations
function queueEmailNotification($user_id, $subject, $message) {
    // Queue email notification - integrate with email service
    error_log("EMAIL QUEUED: User $user_id - $subject");
}

function queueSMSNotification($user_id, $message) {
    // Queue SMS notification - integrate with SMS service
    error_log("SMS QUEUED: User $user_id - $message");
}

function sendPushNotification($user_id, $title, $message, $data = []) {
    // Send push notification - integrate with push service
    error_log("PUSH SENT: User $user_id - $title");
}

function queueBroadcastPushNotifications($title, $message, $audience) {
    // Queue broadcast push notifications
    error_log("BROADCAST PUSH QUEUED: $audience - $title");
}
?>
