<?php
/**
 * Ghana National Event Management Platform
 * Payment Processing API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';
require_once '../includes/auth.php';

class PaymentAPI {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Handle API requests
     */
    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $pathParts = explode('/', trim($path, '/'));
        
        $action = end($pathParts);
        
        try {
            switch ($method) {
                case 'GET':
                    $this->handleGet($action);
                    break;
                case 'POST':
                    $this->handlePost($action);
                    break;
                default:
                    throw new Exception('Method not allowed');
            }
        } catch (Exception $e) {
            http_response_code(400);
            echo json_encode(['error' => $e->getMessage()]);
        }
    }

    /**
     * Handle GET requests
     */
    private function handleGet($action) {
        switch ($action) {
            case 'methods':
                $this->getPaymentMethods();
                break;
            case 'status':
                $this->getPaymentStatus();
                break;
            case 'history':
                $this->getPaymentHistory();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Handle POST requests
     */
    private function handlePost($action) {
        switch ($action) {
            case 'initialize':
                $this->initializePayment();
                break;
            case 'verify':
                $this->verifyPayment();
                break;
            case 'mobile-money':
                $this->processMobileMoneyPayment();
                break;
            case 'card':
                $this->processCardPayment();
                break;
            case 'webhook':
                $this->handleWebhook();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    /**
     * Get available payment methods
     */
    private function getPaymentMethods() {
        $methods = [
            'mobile_money' => [
                'name' => 'Mobile Money',
                'providers' => [
                    'mtn' => [
                        'name' => 'MTN Mobile Money',
                        'code' => 'mtn_momo',
                        'icon' => 'mtn-logo.png',
                        'enabled' => true
                    ],
                    'vodafone' => [
                        'name' => 'Vodafone Cash',
                        'code' => 'vodafone_cash',
                        'icon' => 'vodafone-logo.png',
                        'enabled' => true
                    ],
                    'airteltigo' => [
                        'name' => 'AirtelTigo Money',
                        'code' => 'airteltigo_money',
                        'icon' => 'airteltigo-logo.png',
                        'enabled' => true
                    ]
                ]
            ],
            'cards' => [
                'name' => 'Credit/Debit Cards',
                'providers' => [
                    'visa' => [
                        'name' => 'Visa',
                        'code' => 'visa',
                        'icon' => 'visa-logo.png',
                        'enabled' => true
                    ],
                    'mastercard' => [
                        'name' => 'Mastercard',
                        'code' => 'mastercard',
                        'icon' => 'mastercard-logo.png',
                        'enabled' => true
                    ]
                ]
            ],
            'bank_transfer' => [
                'name' => 'Bank Transfer',
                'enabled' => true,
                'processing_time' => '1-3 business days'
            ]
        ];

        echo json_encode([
            'success' => true,
            'data' => $methods
        ]);
    }

    /**
     * Initialize payment
     */
    private function initializePayment() {
        $user = AuthMiddleware::requireAuth();
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Validate required fields
        $required = ['amount', 'currency', 'payment_method', 'reference_type', 'reference_id'];
        foreach ($required as $field) {
            if (empty($input[$field])) {
                throw new Exception("$field is required");
            }
        }

        // Validate amount
        if ($input['amount'] <= 0) {
            throw new Exception('Amount must be greater than 0');
        }

        // Calculate fees and taxes
        $amount = floatval($input['amount']);
        $fees = $this->calculateFees($amount, $input['payment_method']);
        $taxAmount = $this->calculateTax($amount);
        $totalAmount = $amount + $fees + $taxAmount;

        // Create payment transaction record
        $transactionId = $this->createTransaction([
            'user_id' => $user['id'],
            'transaction_type' => $input['reference_type'],
            'reference_id' => $input['reference_id'],
            'amount' => $amount,
            'currency' => $input['currency'],
            'payment_method' => $input['payment_method'],
            'fees' => $fees,
            'tax_amount' => $taxAmount,
            'net_amount' => $totalAmount,
            'metadata' => json_encode($input['metadata'] ?? [])
        ]);

        // Initialize payment based on method
        switch ($input['payment_method']) {
            case 'mobile_money':
                $result = $this->initializeMobileMoneyPayment($transactionId, $input, $totalAmount);
                break;
            case 'visa':
            case 'mastercard':
                $result = $this->initializeCardPayment($transactionId, $input, $totalAmount);
                break;
            case 'bank_transfer':
                $result = $this->initializeBankTransfer($transactionId, $input, $totalAmount);
                break;
            default:
                throw new Exception('Unsupported payment method');
        }

        echo json_encode([
            'success' => true,
            'transaction_id' => $transactionId,
            'amount' => $amount,
            'fees' => $fees,
            'tax' => $taxAmount,
            'total_amount' => $totalAmount,
            'payment_data' => $result
        ]);
    }

    /**
     * Initialize Mobile Money payment
     */
    private function initializeMobileMoneyPayment($transactionId, $input, $amount) {
        $provider = $input['provider'] ?? 'mtn';
        $phoneNumber = $input['phone_number'] ?? '';
        
        if (empty($phoneNumber)) {
            throw new Exception('Phone number is required for mobile money payments');
        }

        // Format phone number for Ghana
        $phoneNumber = $this->formatGhanaPhoneNumber($phoneNumber);

        switch ($provider) {
            case 'mtn':
                return $this->initializeMTNMoMo($transactionId, $phoneNumber, $amount);
            case 'vodafone':
                return $this->initializeVodafoneCash($transactionId, $phoneNumber, $amount);
            case 'airteltigo':
                return $this->initializeAirtelTigoMoney($transactionId, $phoneNumber, $amount);
            default:
                throw new Exception('Unsupported mobile money provider');
        }
    }

    /**
     * Initialize MTN Mobile Money payment
     */
    private function initializeMTNMoMo($transactionId, $phoneNumber, $amount) {
        // MTN MoMo API integration
        $apiUrl = 'https://sandbox.momodeveloper.mtn.com/collection/v1_0/requesttopay';
        
        $requestData = [
            'amount' => strval($amount),
            'currency' => 'EUR', // MTN sandbox uses EUR
            'externalId' => $transactionId,
            'payer' => [
                'partyIdType' => 'MSISDN',
                'partyId' => $phoneNumber
            ],
            'payerMessage' => 'Payment for Ghana Events Platform',
            'payeeNote' => 'Event ticket payment'
        ];

        // Get access token
        $accessToken = $this->getMTNAccessToken();
        
        $headers = [
            'Authorization: Bearer ' . $accessToken,
            'X-Reference-Id: ' . $this->generateUUID(),
            'X-Target-Environment: sandbox',
            'Content-Type: application/json',
            'Ocp-Apim-Subscription-Key: ' . Config::MTN_MOMO_API_KEY
        ];

        $response = $this->makeHttpRequest($apiUrl, 'POST', $requestData, $headers);
        
        return [
            'provider' => 'mtn',
            'reference' => $transactionId,
            'status' => 'pending',
            'instructions' => 'Please check your phone for the payment prompt and enter your MTN Mobile Money PIN to complete the transaction.'
        ];
    }

    /**
     * Initialize Vodafone Cash payment
     */
    private function initializeVodafoneCash($transactionId, $phoneNumber, $amount) {
        // Vodafone Cash API integration (simulated)
        return [
            'provider' => 'vodafone',
            'reference' => $transactionId,
            'status' => 'pending',
            'instructions' => 'Please dial *110# and follow the prompts to complete your Vodafone Cash payment.'
        ];
    }

    /**
     * Initialize AirtelTigo Money payment
     */
    private function initializeAirtelTigoMoney($transactionId, $phoneNumber, $amount) {
        // AirtelTigo Money API integration (simulated)
        return [
            'provider' => 'airteltigo',
            'reference' => $transactionId,
            'status' => 'pending',
            'instructions' => 'Please dial *100# and follow the prompts to complete your AirtelTigo Money payment.'
        ];
    }

    /**
     * Initialize card payment using Paystack
     */
    private function initializeCardPayment($transactionId, $input, $amount) {
        $paystackUrl = 'https://api.paystack.co/transaction/initialize';
        
        $requestData = [
            'amount' => $amount * 100, // Paystack expects amount in pesewas
            'email' => $input['email'] ?? '',
            'reference' => $transactionId,
            'callback_url' => Config::APP_URL . '/payment-callback',
            'metadata' => [
                'transaction_id' => $transactionId,
                'platform' => 'Ghana Events Platform'
            ]
        ];

        $headers = [
            'Authorization: Bearer ' . Config::PAYSTACK_SECRET_KEY,
            'Content-Type: application/json'
        ];

        $response = $this->makeHttpRequest($paystackUrl, 'POST', $requestData, $headers);
        
        if ($response && isset($response['data']['authorization_url'])) {
            return [
                'provider' => 'paystack',
                'authorization_url' => $response['data']['authorization_url'],
                'access_code' => $response['data']['access_code'],
                'reference' => $response['data']['reference']
            ];
        }
        
        throw new Exception('Failed to initialize card payment');
    }

    /**
     * Initialize bank transfer
     */
    private function initializeBankTransfer($transactionId, $input, $amount) {
        return [
            'provider' => 'bank_transfer',
            'reference' => $transactionId,
            'bank_details' => [
                'account_name' => 'Ghana Events Platform',
                'account_number' => '1234567890',
                'bank_name' => 'Ghana Commercial Bank',
                'branch' => 'Accra Main Branch',
                'swift_code' => 'GHCBGHAC'
            ],
            'amount' => $amount,
            'instructions' => 'Please use the transaction reference as your payment description when making the transfer.'
        ];
    }

    /**
     * Verify payment status
     */
    private function verifyPayment() {
        $input = json_decode(file_get_contents('php://input'), true);
        $transactionId = $input['transaction_id'] ?? $_GET['transaction_id'] ?? null;
        
        if (!$transactionId) {
            throw new Exception('Transaction ID is required');
        }

        // Get transaction from database
        $transaction = $this->getTransaction($transactionId);
        if (!$transaction) {
            throw new Exception('Transaction not found');
        }

        // Verify with payment provider based on method
        switch ($transaction['payment_method']) {
            case 'mobile_money':
                $status = $this->verifyMobileMoneyPayment($transaction);
                break;
            case 'visa':
            case 'mastercard':
                $status = $this->verifyCardPayment($transaction);
                break;
            case 'bank_transfer':
                $status = $this->verifyBankTransfer($transaction);
                break;
            default:
                throw new Exception('Unsupported payment method');
        }

        // Update transaction status
        $this->updateTransactionStatus($transactionId, $status);

        // If payment is successful, complete the related action
        if ($status === 'completed') {
            $this->completePaymentAction($transaction);
        }

        echo json_encode([
            'success' => true,
            'transaction_id' => $transactionId,
            'status' => $status,
            'transaction' => $transaction
        ]);
    }

    /**
     * Get payment history for user
     */
    private function getPaymentHistory() {
        $user = AuthMiddleware::requireAuth();
        
        $page = $_GET['page'] ?? 1;
        $limit = min($_GET['limit'] ?? 20, 50);
        $offset = ($page - 1) * $limit;

        $query = "SELECT * FROM payment_transactions 
                  WHERE user_id = :user_id 
                  ORDER BY created_at DESC 
                  LIMIT :limit OFFSET :offset";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user['id']);
        $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
        $stmt->execute();

        $transactions = $stmt->fetchAll();

        echo json_encode([
            'success' => true,
            'data' => $transactions,
            'pagination' => [
                'page' => (int)$page,
                'limit' => (int)$limit
            ]
        ]);
    }

    /**
     * Handle payment webhooks
     */
    private function handleWebhook() {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
        
        // Verify webhook signature (implement based on provider)
        $provider = $_GET['provider'] ?? 'paystack';
        
        switch ($provider) {
            case 'paystack':
                $this->handlePaystackWebhook($data);
                break;
            case 'mtn':
                $this->handleMTNWebhook($data);
                break;
            default:
                throw new Exception('Unsupported webhook provider');
        }

        echo json_encode(['success' => true]);
    }

    /**
     * Helper methods
     */
    private function createTransaction($data) {
        $query = "INSERT INTO payment_transactions (
                    user_id, transaction_type, reference_id, amount, currency,
                    payment_method, fees, tax_amount, net_amount, metadata
                  ) VALUES (
                    :user_id, :transaction_type, :reference_id, :amount, :currency,
                    :payment_method, :fees, :tax_amount, :net_amount, :metadata
                  )";
        
        $stmt = $this->conn->prepare($query);
        foreach ($data as $key => $value) {
            $stmt->bindValue(":$key", $value);
        }
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        
        throw new Exception('Failed to create transaction');
    }

    private function getTransaction($transactionId) {
        $query = "SELECT * FROM payment_transactions WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $transactionId);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    private function updateTransactionStatus($transactionId, $status) {
        $query = "UPDATE payment_transactions SET status = :status";
        
        if ($status === 'completed') {
            $query .= ", completed_at = NOW()";
        }
        
        $query .= " WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':id', $transactionId);
        
        return $stmt->execute();
    }

    private function calculateFees($amount, $paymentMethod) {
        $feeRates = [
            'mobile_money' => 0.01, // 1%
            'visa' => 0.025, // 2.5%
            'mastercard' => 0.025, // 2.5%
            'bank_transfer' => 5.00 // Fixed fee
        ];
        
        $rate = $feeRates[$paymentMethod] ?? 0;
        
        if ($paymentMethod === 'bank_transfer') {
            return $rate; // Fixed fee
        }
        
        return $amount * $rate;
    }

    private function calculateTax($amount) {
        // Ghana VAT rate
        return $amount * 0.125; // 12.5%
    }

    private function formatGhanaPhoneNumber($phoneNumber) {
        // Remove all non-numeric characters
        $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
        
        // Add country code if not present
        if (strlen($phoneNumber) === 10 && substr($phoneNumber, 0, 1) === '0') {
            $phoneNumber = '233' . substr($phoneNumber, 1);
        } elseif (strlen($phoneNumber) === 9) {
            $phoneNumber = '233' . $phoneNumber;
        }
        
        return $phoneNumber;
    }

    private function getMTNAccessToken() {
        // Implementation for getting MTN MoMo access token
        // This would involve API calls to MTN's authentication endpoint
        return 'sample_access_token';
    }

    private function generateUUID() {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    private function makeHttpRequest($url, $method, $data = null, $headers = []) {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30
        ]);
        
        if ($data && in_array($method, ['POST', 'PUT', 'PATCH'])) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 200 && $httpCode < 300) {
            return json_decode($response, true);
        }
        
        return false;
    }

    private function verifyMobileMoneyPayment($transaction) {
        // Implementation for verifying mobile money payment status
        // This would involve checking with the respective provider's API
        return 'completed'; // Simulated
    }

    private function verifyCardPayment($transaction) {
        // Verify with Paystack
        $paystackUrl = "https://api.paystack.co/transaction/verify/{$transaction['id']}";
        
        $headers = [
            'Authorization: Bearer ' . Config::PAYSTACK_SECRET_KEY
        ];
        
        $response = $this->makeHttpRequest($paystackUrl, 'GET', null, $headers);
        
        if ($response && $response['data']['status'] === 'success') {
            return 'completed';
        }
        
        return 'pending';
    }

    private function verifyBankTransfer($transaction) {
        // Bank transfers require manual verification
        return $transaction['status']; // Keep current status
    }

    private function completePaymentAction($transaction) {
        // Complete the action based on transaction type
        switch ($transaction['transaction_type']) {
            case 'ticket_purchase':
                $this->completeTicketPurchase($transaction);
                break;
            case 'vendor_payment':
                $this->completeVendorPayment($transaction);
                break;
            case 'accommodation':
                $this->completeAccommodationBooking($transaction);
                break;
        }
    }

    private function completeTicketPurchase($transaction) {
        // Update ticket status to confirmed
        $query = "UPDATE event_tickets SET payment_status = 'completed' 
                  WHERE id = :reference_id AND user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':reference_id', $transaction['reference_id']);
        $stmt->bindParam(':user_id', $transaction['user_id']);
        $stmt->execute();
    }

    private function completeVendorPayment($transaction) {
        // Update vendor booking payment status
        $query = "UPDATE event_vendor_bookings SET payment_status = 'completed' 
                  WHERE id = :reference_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':reference_id', $transaction['reference_id']);
        $stmt->execute();
    }

    private function completeAccommodationBooking($transaction) {
        // Complete accommodation booking
        // Implementation would depend on accommodation booking structure
    }

    private function handlePaystackWebhook($data) {
        // Handle Paystack webhook events
        if ($data['event'] === 'charge.success') {
            $reference = $data['data']['reference'];
            $this->updateTransactionStatus($reference, 'completed');
        }
    }

    private function handleMTNWebhook($data) {
        // Handle MTN MoMo webhook events
        // Implementation based on MTN's webhook structure
    }
}

// Handle the request
$api = new PaymentAPI();
$api->handleRequest();
?>
