<?php
require_once '../config/database.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? '';

try {
    switch ($endpoint) {
        case 'report':
            if ($method === 'POST') {
                submitSafetyReport();
            }
            break;
        
        case 'reports':
            if ($method === 'GET') {
                getSafetyReports();
            }
            break;
            
        case 'emergency-alert':
            if ($method === 'POST') {
                sendEmergencyAlert();
            }
            break;
            
        case 'update-status':
            if ($method === 'PUT') {
                updateReportStatus();
            }
            break;
            
        case 'emergency-contacts':
            if ($method === 'GET') {
                getEmergencyContacts();
            }
            break;
            
        case 'incident-analytics':
            if ($method === 'GET') {
                getIncidentAnalytics();
            }
            break;
            
        case 'crowd-monitoring':
            if ($method === 'GET') {
                getCrowdMonitoring();
            } elseif ($method === 'POST') {
                updateCrowdStatus();
            }
            break;
            
        case 'emergency-response':
            if ($method === 'POST') {
                initiateEmergencyResponse();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function submitSafetyReport() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Allow anonymous reports
    $user_id = null;
    $token = getBearerToken();
    if ($token && verifyJWT($token)) {
        $user = getCurrentUser($token);
        $user_id = $user['id'] ?? null;
    }
    
    $required_fields = ['event_id', 'incident_type', 'description'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Generate unique report ID for tracking
        $report_id = 'SR' . date('Ymd') . sprintf('%06d', rand(1, 999999));
        
        // Insert safety report
        $stmt = $pdo->prepare("
            INSERT INTO safety_reports (
                report_id, event_id, user_id, incident_type, severity,
                description, location, latitude, longitude, 
                anonymous, contact_info, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'reported', NOW())
        ");
        
        $stmt->execute([
            $report_id,
            $input['event_id'],
            $user_id,
            $input['incident_type'],
            $input['severity'] ?? 'medium',
            $input['description'],
            $input['location'] ?? '',
            $input['latitude'] ?? null,
            $input['longitude'] ?? null,
            $input['anonymous'] ?? ($user_id ? 0 : 1),
            $input['contact_info'] ?? ''
        ]);
        
        $db_report_id = $pdo->lastInsertId();
        
        // Determine urgency and notify appropriate authorities
        $severity = $input['severity'] ?? 'medium';
        $incident_type = $input['incident_type'];
        
        // High priority incidents trigger immediate alerts
        if ($severity === 'high' || in_array($incident_type, ['violence', 'medical_emergency', 'fire', 'security_threat'])) {
            // Notify emergency services
            notifyEmergencyServices($db_report_id, $input);
            
            // Notify government officials
            $stmt = $pdo->prepare("
                INSERT INTO notifications (user_id, type, title, message, data, created_at)
                SELECT u.id, 'emergency_alert', 'URGENT: Safety Report', 
                       CONCAT('High priority safety report: ', ?), 
                       JSON_OBJECT('report_id', ?, 'incident_type', ?, 'severity', ?), NOW()
                FROM users u 
                JOIN user_roles ur ON u.id = ur.user_id 
                WHERE ur.role_key IN ('government_official', 'admin')
            ");
            $stmt->execute([$input['description'], $report_id, $incident_type, $severity]);
        } else {
            // Standard notification for non-urgent reports
            $stmt = $pdo->prepare("
                INSERT INTO notifications (user_id, type, title, message, data, created_at)
                SELECT u.id, 'safety_report', 'New Safety Report', 
                       CONCAT('New safety report: ', ?), 
                       JSON_OBJECT('report_id', ?, 'incident_type', ?), NOW()
                FROM users u 
                JOIN user_roles ur ON u.id = ur.user_id 
                WHERE ur.role_key = 'government_official'
            ");
            $stmt->execute([$input['description'], $report_id, $incident_type]);
        }
        
        // Update event safety status if needed
        if ($severity === 'high') {
            $stmt = $pdo->prepare("
                UPDATE events 
                SET safety_status = 'alert', updated_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$input['event_id']]);
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Safety report submitted successfully',
            'report_id' => $report_id,
            'tracking_number' => $report_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getSafetyReports() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'safety_monitoring')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $event_id = $_GET['event_id'] ?? '';
    $status = $_GET['status'] ?? '';
    $severity = $_GET['severity'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($event_id) {
        $where_conditions[] = 'sr.event_id = ?';
        $params[] = $event_id;
    }
    
    if ($status) {
        $where_conditions[] = 'sr.status = ?';
        $params[] = $status;
    }
    
    if ($severity) {
        $where_conditions[] = 'sr.severity = ?';
        $params[] = $severity;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM safety_reports sr WHERE $where_clause");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get reports
    $stmt = $pdo->prepare("
        SELECT sr.*, e.title as event_title, e.start_date as event_date,
               u.full_name as reporter_name,
               ru.full_name as responder_name
        FROM safety_reports sr
        LEFT JOIN events e ON sr.event_id = e.id
        LEFT JOIN users u ON sr.user_id = u.id
        LEFT JOIN users ru ON sr.responded_by = ru.id
        WHERE $where_clause
        ORDER BY sr.created_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $reports = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Hide reporter info for anonymous reports
    foreach ($reports as &$report) {
        if ($report['anonymous']) {
            $report['reporter_name'] = 'Anonymous';
            $report['contact_info'] = '[Hidden]';
        }
    }
    
    echo json_encode([
        'success' => true,
        'reports' => $reports,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function sendEmergencyAlert() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'emergency_management')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['alert_type', 'message', 'target_audience'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Create emergency alert
        $stmt = $pdo->prepare("
            INSERT INTO emergency_alerts (
                alert_type, title, message, severity, target_audience,
                event_id, location, radius_km, issued_by, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())
        ");
        
        $stmt->execute([
            $input['alert_type'],
            $input['title'] ?? 'Emergency Alert',
            $input['message'],
            $input['severity'] ?? 'high',
            $input['target_audience'],
            $input['event_id'] ?? null,
            $input['location'] ?? '',
            $input['radius_km'] ?? 5,
            $user['id']
        ]);
        
        $alert_id = $pdo->lastInsertId();
        
        // Determine notification targets based on audience
        $notification_query = "
            INSERT INTO notifications (user_id, type, title, message, data, created_at)
        ";
        
        switch ($input['target_audience']) {
            case 'all_users':
                $notification_query .= "
                    SELECT id, 'emergency_alert', ?, ?, 
                           JSON_OBJECT('alert_id', ?, 'alert_type', ?), NOW()
                    FROM users WHERE status = 'active'
                ";
                break;
                
            case 'event_attendees':
                if (!empty($input['event_id'])) {
                    $notification_query .= "
                        SELECT et.user_id, 'emergency_alert', ?, ?, 
                               JSON_OBJECT('alert_id', ?, 'alert_type', ?), NOW()
                        FROM event_tickets et
                        WHERE et.event_id = ? AND et.status = 'confirmed'
                    ";
                }
                break;
                
            case 'nearby_users':
                // For location-based alerts (would need geolocation implementation)
                $notification_query .= "
                    SELECT id, 'emergency_alert', ?, ?, 
                           JSON_OBJECT('alert_id', ?, 'alert_type', ?), NOW()
                    FROM users WHERE status = 'active'
                ";
                break;
        }
        
        $stmt = $pdo->prepare($notification_query);
        $params = [
            $input['title'] ?? 'Emergency Alert',
            $input['message'],
            $alert_id,
            $input['alert_type']
        ];
        
        if ($input['target_audience'] === 'event_attendees' && !empty($input['event_id'])) {
            $params[] = $input['event_id'];
        }
        
        $stmt->execute($params);
        
        // Send SMS/Email alerts for high severity
        if ($input['severity'] === 'critical') {
            // Queue SMS and email notifications
            queueEmergencyNotifications($alert_id, $input);
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Emergency alert sent successfully',
            'alert_id' => $alert_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function updateReportStatus() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'safety_monitoring')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $report_id = $input['report_id'] ?? '';
    $status = $input['status'] ?? '';
    $response_notes = $input['response_notes'] ?? '';
    
    if (!$report_id || !$status) {
        http_response_code(400);
        echo json_encode(['error' => 'Report ID and status required']);
        return;
    }
    
    $valid_statuses = ['reported', 'investigating', 'resolved', 'closed', 'escalated'];
    if (!in_array($status, $valid_statuses)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid status']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update report status
        $stmt = $pdo->prepare("
            UPDATE safety_reports 
            SET status = ?, responded_by = ?, response_notes = ?, responded_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$status, $user['id'], $response_notes, $report_id]);
        
        // Get report details for notification
        $stmt = $pdo->prepare("
            SELECT report_id, user_id, incident_type, anonymous 
            FROM safety_reports WHERE id = ?
        ");
        $stmt->execute([$report_id]);
        $report = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Notify reporter if not anonymous
        if ($report && $report['user_id'] && !$report['anonymous']) {
            $message = "Your safety report #{$report['report_id']} has been updated to: {$status}";
            if ($response_notes) {
                $message .= "\n\nResponse: {$response_notes}";
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO notifications (user_id, type, title, message, data, created_at)
                VALUES (?, 'safety_update', 'Safety Report Update', ?, 
                        JSON_OBJECT('report_id', ?, 'status', ?), NOW())
            ");
            $stmt->execute([
                $report['user_id'],
                $message,
                $report['report_id'],
                $status
            ]);
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Report status updated successfully'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getEmergencyContacts() {
    $contacts = [
        'police' => [
            'name' => 'Ghana Police Service',
            'number' => '191',
            'description' => 'Emergency police response'
        ],
        'fire' => [
            'name' => 'Ghana Fire Service',
            'number' => '192',
            'description' => 'Fire and rescue services'
        ],
        'ambulance' => [
            'name' => 'National Ambulance Service',
            'number' => '193',
            'description' => 'Medical emergency response'
        ],
        'disaster' => [
            'name' => 'National Disaster Management',
            'number' => '999',
            'description' => 'Natural disaster response'
        ],
        'platform' => [
            'name' => 'Event Platform Emergency',
            'number' => EMERGENCY_HOTLINE,
            'description' => 'Platform-specific emergency support'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'emergency_contacts' => $contacts
    ]);
}

function getIncidentAnalytics() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'analytics_access')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        return;
    }
    
    // Incident trends by type
    $stmt = $pdo->prepare("
        SELECT incident_type, COUNT(*) as count,
               AVG(CASE WHEN severity = 'low' THEN 1 
                        WHEN severity = 'medium' THEN 2 
                        WHEN severity = 'high' THEN 3 END) as avg_severity
        FROM safety_reports 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY incident_type
        ORDER BY count DESC
    ");
    $stmt->execute();
    $incident_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Monthly incident counts
    $stmt = $pdo->prepare("
        SELECT DATE_FORMAT(created_at, '%Y-%m') as month, 
               COUNT(*) as total_incidents,
               SUM(CASE WHEN severity = 'high' THEN 1 ELSE 0 END) as high_severity
        FROM safety_reports 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month
    ");
    $stmt->execute();
    $monthly_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Response time analytics
    $stmt = $pdo->prepare("
        SELECT AVG(TIMESTAMPDIFF(MINUTE, created_at, responded_at)) as avg_response_minutes,
               COUNT(CASE WHEN responded_at IS NOT NULL THEN 1 END) as responded_count,
               COUNT(*) as total_count
        FROM safety_reports 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 3 MONTH)
    ");
    $stmt->execute();
    $response_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'analytics' => [
            'incident_trends' => $incident_trends,
            'monthly_trends' => $monthly_trends,
            'response_stats' => $response_stats
        ]
    ]);
}

function notifyEmergencyServices($report_id, $incident_data) {
    // In a real implementation, this would integrate with emergency service APIs
    // For now, we'll log the emergency notification
    error_log("EMERGENCY NOTIFICATION: Report ID {$report_id} - {$incident_data['incident_type']} - {$incident_data['description']}");
    
    // Could integrate with:
    // - SMS gateway for emergency services
    // - Emergency service dispatch systems
    // - Government alert systems
}

function queueEmergencyNotifications($alert_id, $alert_data) {
    // Queue SMS and email notifications for critical alerts
    // This would integrate with SMS and email services
    error_log("CRITICAL ALERT QUEUED: Alert ID {$alert_id} - {$alert_data['message']}");
}
?>
