<?php
/**
 * Ghana National Event Management Platform
 * Ticketing API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/api-logger.php';

// Initialize database connection
$database = new Database();
$pdo = $database->getConnection();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

// Log API request
ApiLogger::logRequest($method, $action, $_GET);

try {
    switch ($action) {
        case 'create':
            if ($method === 'POST') {
                createTicket();
            }
            break;
        
        case 'list':
            if ($method === 'GET') {
                getTickets();
            }
            break;
            
        case 'details':
            if ($method === 'GET') {
                getTicketDetails();
            }
            break;
            
        case 'organizer-tickets':
            if ($method === 'GET') {
                getOrganizerTickets();
            }
            break;
            
        case 'validate':
            if ($method === 'POST') {
                validateTicket();
            }
            break;
            
        case 'sales-report':
            if ($method === 'GET') {
                getTicketSalesReport();
            }
            break;
            
        case 'user-tickets':
            if ($method === 'GET') {
                getUserTickets();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function createTicket() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['event_id', 'attendee_name', 'attendee_email'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Get event details
        $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ? AND approval_status = 'approved'");
        $stmt->execute([$input['event_id']]);
        $event = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$event) {
            http_response_code(404);
            echo json_encode(['error' => 'Event not found or not approved']);
            return;
        }
        
        // Check if event is full
        if ($event['max_attendees']) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM event_tickets WHERE event_id = ? AND payment_status = 'completed'");
            $stmt->execute([$input['event_id']]);
            $current_attendees = $stmt->fetchColumn();
            
            if ($current_attendees >= $event['max_attendees']) {
                http_response_code(400);
                echo json_encode(['error' => 'Event is fully booked']);
                return;
            }
        }
        
        // Generate ticket reference
        $ticket_reference = 'GH' . strtoupper(uniqid());
        
        // Create ticket
        $stmt = $pdo->prepare("
            INSERT INTO event_tickets (
                event_id, user_id, ticket_code, price_paid, 
                payment_method, payment_status, purchased_at
            ) VALUES (?, ?, ?, ?, 'card', 'pending', NOW())
        ");
        
        $stmt->execute([
            $input['event_id'],
            $input['user_id'] ?? 1, // Default user ID if not provided
            $ticket_reference,
            $event['ticket_price']
        ]);
        
        $ticket_id = $pdo->lastInsertId();
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Ticket created successfully',
            'ticket_id' => $ticket_id,
            'ticket_reference' => $ticket_reference,
            'amount' => $event['ticket_price']
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getOrganizerTickets() {
    global $pdo;
    
    $organizer_id = $_GET['organizer_id'] ?? 6; // Default to dorisanuson's ID
    $event_id = $_GET['event_id'] ?? '';
    $status = $_GET['status'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['e.organizer_id = ?'];
    $params = [$organizer_id];
    
    if ($event_id) {
        $where_conditions[] = 'et.event_id = ?';
        $params[] = $event_id;
    }
    
    if ($status) {
        $where_conditions[] = 'et.payment_status = ?';
        $params[] = $status;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("
        SELECT COUNT(*) 
        FROM event_tickets et
        JOIN events e ON et.event_id = e.id
        WHERE $where_clause
    ");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get tickets
    $stmt = $pdo->prepare("
        SELECT et.*, e.title as event_title, e.start_datetime, e.ticket_price,
               u.full_name as attendee_name, u.email as attendee_email, u.phone as attendee_phone
        FROM event_tickets et
        JOIN events e ON et.event_id = e.id
        JOIN users u ON et.user_id = u.id
        WHERE $where_clause
        ORDER BY et.purchased_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'data' => $tickets,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function getTicketSalesReport() {
    global $pdo;
    
    $organizer_id = $_GET['organizer_id'] ?? 6; // Default to dorisanuson's ID
    $event_id = $_GET['event_id'] ?? '';
    
    try {
        if ($event_id) {
            // Sales report for specific event
            $stmt = $pdo->prepare("
                SELECT 
                    COUNT(*) as total_tickets,
                    SUM(CASE WHEN payment_status = 'completed' THEN 1 ELSE 0 END) as sold_tickets,
                    SUM(CASE WHEN payment_status = 'pending' THEN 1 ELSE 0 END) as pending_tickets,
                    SUM(CASE WHEN payment_status = 'completed' THEN price_paid ELSE 0 END) as total_revenue,
                    AVG(CASE WHEN payment_status = 'completed' THEN price_paid ELSE NULL END) as avg_ticket_price
                FROM event_tickets et
                JOIN events e ON et.event_id = e.id
                WHERE e.organizer_id = ? AND et.event_id = ?
            ");
            $stmt->execute([$organizer_id, $event_id]);
        } else {
            // Overall sales report for organizer
            $stmt = $pdo->prepare("
                SELECT 
                    COUNT(*) as total_tickets,
                    SUM(CASE WHEN payment_status = 'completed' THEN 1 ELSE 0 END) as sold_tickets,
                    SUM(CASE WHEN payment_status = 'pending' THEN 1 ELSE 0 END) as pending_tickets,
                    SUM(CASE WHEN payment_status = 'completed' THEN price_paid ELSE 0 END) as total_revenue,
                    AVG(CASE WHEN payment_status = 'completed' THEN price_paid ELSE NULL END) as avg_ticket_price
                FROM event_tickets et
                JOIN events e ON et.event_id = e.id
                WHERE e.organizer_id = ?
            ");
            $stmt->execute([$organizer_id]);
        }
        
        $report = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get sales by date (last 30 days)
        $stmt = $pdo->prepare("
            SELECT 
                DATE(et.purchased_at) as sale_date,
                COUNT(*) as tickets_sold,
                SUM(et.price_paid) as daily_revenue
            FROM event_tickets et
            JOIN events e ON et.event_id = e.id
            WHERE e.organizer_id = ? 
                AND et.payment_status = 'completed'
                AND et.purchased_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY DATE(et.purchased_at)
            ORDER BY sale_date DESC
        ");
        $stmt->execute([$organizer_id]);
        $daily_sales = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => [
                'summary' => $report,
                'daily_sales' => $daily_sales
            ]
        ]);
        
    } catch (Exception $e) {
        throw $e;
    }
}

function validateTicket() {
    global $pdo;
    
    $input = json_decode(file_get_contents('php://input'), true);
    $ticket_reference = $input['ticket_reference'] ?? '';
    
    if (!$ticket_reference) {
        http_response_code(400);
        echo json_encode(['error' => 'Ticket reference required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT et.*, e.title as event_title, e.start_datetime, e.end_datetime,
                   v.name as venue_name, v.address as venue_address
            FROM event_tickets et
            JOIN events e ON et.event_id = e.id
            LEFT JOIN venues v ON e.venue_id = v.id
            WHERE et.ticket_reference = ?
        ");
        $stmt->execute([$ticket_reference]);
        $ticket = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$ticket) {
            http_response_code(404);
            echo json_encode(['error' => 'Ticket not found']);
            return;
        }
        
        $is_valid = $ticket['payment_status'] === 'completed';
        $event_started = strtotime($ticket['start_datetime']) <= time();
        $event_ended = strtotime($ticket['end_datetime']) <= time();
        
        echo json_encode([
            'success' => true,
            'ticket' => $ticket,
            'validation' => [
                'is_valid' => $is_valid,
                'is_paid' => $ticket['payment_status'] === 'completed',
                'event_started' => $event_started,
                'event_ended' => $event_ended,
                'can_enter' => $is_valid && !$event_ended
            ]
        ]);
        
    } catch (Exception $e) {
        throw $e;
    }
}

function getTickets() {
    global $pdo;
    
    $event_id = $_GET['event_id'] ?? '';
    $status = $_GET['status'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($event_id) {
        $where_conditions[] = 'et.event_id = ?';
        $params[] = $event_id;
    }
    
    if ($status) {
        $where_conditions[] = 'et.payment_status = ?';
        $params[] = $status;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM event_tickets et WHERE $where_clause");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get tickets
    $stmt = $pdo->prepare("
        SELECT et.*, e.title as event_title, e.start_datetime
        FROM event_tickets et
        JOIN events e ON et.event_id = e.id
        WHERE $where_clause
        ORDER BY et.booking_date DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'data' => $tickets,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function getTicketDetails() {
    global $pdo;
    
    $ticket_id = $_GET['id'] ?? $_GET['ticket_id'] ?? '';
    if (!$ticket_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Ticket ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT et.*, e.title as event_title, e.description as event_description,
               e.start_datetime, e.end_datetime, v.name as venue_name, 
               v.address as venue_address, v.city, v.region
        FROM event_tickets et
        JOIN events e ON et.event_id = e.id
        LEFT JOIN venues v ON e.venue_id = v.id
        WHERE et.id = ?
    ");
    $stmt->execute([$ticket_id]);
    $ticket = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$ticket) {
        http_response_code(404);
        echo json_encode(['error' => 'Ticket not found']);
        return;
    }
    
    echo json_encode([
        'success' => true,
        'data' => $ticket
    ]);
}

function getUserTickets() {
    global $pdo;
    
    $user_id = $_GET['user_id'] ?? '';
    if (!$user_id) {
        http_response_code(400);
        echo json_encode(['error' => 'User ID required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT et.*, e.title as event_title, e.description as event_description,
                   e.start_datetime, e.end_datetime, v.name as venue_name, 
                   v.address as venue_address, v.city, v.region
            FROM event_tickets et
            JOIN events e ON et.event_id = e.id
            LEFT JOIN venues v ON e.venue_id = v.id
            WHERE et.user_id = ?
            ORDER BY et.purchased_at DESC
        ");
        $stmt->execute([$user_id]);
        $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'tickets' => $tickets
        ]);
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to fetch user tickets: ' . $e->getMessage()]);
    }
}
?>
