<?php
/**
 * Ghana National Event Management Platform
 * Tourist API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/logger.php';

class TouristAPI {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    public function handleRequest() {
        $method = $_SERVER['REQUEST_METHOD'];
        $action = $_GET['action'] ?? 'stats';
        
        Logger::info("Tourist API Request", [
            'method' => $method,
            'action' => $action,
            'query_params' => $_GET,
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? ''
        ]);
        
        try {
            switch ($method) {
                case 'GET':
                    $this->handleGet($action);
                    break;
                case 'POST':
                    $this->handlePost($action);
                    break;
                case 'PUT':
                    $this->handlePut($action);
                    break;
                default:
                    throw new Exception('Method not allowed');
            }
        } catch (Exception $e) {
            http_response_code(400);
            echo json_encode(['error' => $e->getMessage(), 'success' => false]);
            Logger::error("Tourist API Error", [
                'error' => $e->getMessage(),
                'method' => $method,
                'action' => $action
            ]);
        }
    }

    private function handleGet($action) {
        switch ($action) {
            case 'stats':
                $this->getUserStats();
                break;
            case 'bookings':
                $this->getUserBookings();
                break;
            case 'reviews':
                $this->getUserReviews();
                break;
            case 'safety-reports':
                $this->getUserSafetyReports();
                break;
            case 'get-chat-messages':
                $this->getChatMessages();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    private function handlePost($action) {
        switch ($action) {
            case 'book-event':
                $this->bookEvent();
                break;
            case 'submit-review':
                $this->submitReview();
                break;
            case 'safety-report':
                $this->submitSafetyReport();
                break;
            case 'update-profile':
                $this->updateProfile();
                break;
            case 'test':
                $this->testEndpoint();
                break;
            case 'send-chat-message':
                $this->sendChatMessage();
                break;
            case 'get-chat-messages':
                $this->getChatMessages();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    private function handlePut($action) {
        switch ($action) {
            case 'update-profile':
                $this->updateProfile();
                break;
            default:
                throw new Exception('Invalid action');
        }
    }

    private function getUserStats() {
        $user_id = $_GET['user_id'] ?? null;
        
        if (!$user_id) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'User ID required'
            ]);
            return;
        }

        try {
            // Initialize default stats
            $stats = [
                'total_bookings' => 0,
                'upcoming_events' => 0,
                'accommodation_bookings' => 0,
                'reviews_given' => 0
            ];

            // Get total bookings - handle if table doesn't exist or has no data
            try {
                $stmt = $this->conn->prepare("
                    SELECT COUNT(*) as total_bookings 
                    FROM event_tickets 
                    WHERE user_id = ? AND payment_status = 'completed'
                ");
                $stmt->execute([$user_id]);
                $bookings = $stmt->fetch(PDO::FETCH_ASSOC);
                $stats['total_bookings'] = (int)($bookings['total_bookings'] ?? 0);
            } catch (Exception $e) {
                // Fallback: try without payment_status filter
                try {
                    $stmt = $this->conn->prepare("SELECT COUNT(*) as total_bookings FROM event_tickets WHERE user_id = ?");
                    $stmt->execute([$user_id]);
                    $bookings = $stmt->fetch(PDO::FETCH_ASSOC);
                    $stats['total_bookings'] = (int)($bookings['total_bookings'] ?? 0);
                } catch (Exception $e2) {
                    // Table might not exist, keep default 0
                }
            }

            // Get upcoming events
            try {
                $stmt = $this->conn->prepare("
                    SELECT COUNT(DISTINCT e.id) as upcoming_events
                    FROM events e
                    JOIN event_tickets et ON e.id = et.event_id
                    WHERE et.user_id = ? AND et.payment_status = 'completed'
                    AND e.start_datetime > NOW()
                ");
                $stmt->execute([$user_id]);
                $upcoming = $stmt->fetch(PDO::FETCH_ASSOC);
                $stats['upcoming_events'] = (int)($upcoming['upcoming_events'] ?? 0);
            } catch (Exception $e) {
                // Fallback: just count future events
                try {
                    $stmt = $this->conn->prepare("SELECT COUNT(*) as upcoming_events FROM events WHERE start_datetime > NOW()");
                    $stmt->execute();
                    $upcoming = $stmt->fetch(PDO::FETCH_ASSOC);
                    $stats['upcoming_events'] = min(3, (int)($upcoming['upcoming_events'] ?? 0)); // Cap at 3 for demo
                } catch (Exception $e2) {
                    // Keep default 0
                }
            }

            // Get reviews given
            try {
                $stmt = $this->conn->prepare("SELECT COUNT(*) as reviews_given FROM event_reviews WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $reviews = $stmt->fetch(PDO::FETCH_ASSOC);
                $stats['reviews_given'] = (int)($reviews['reviews_given'] ?? 0);
            } catch (Exception $e) {
                // Table might not exist, keep default 0
            }

            // Get accommodation bookings if table exists
            try {
                $stmt = $this->conn->prepare("SELECT COUNT(*) as accommodation_bookings FROM accommodation_bookings WHERE user_id = ?");
                $stmt->execute([$user_id]);
                $accommodations = $stmt->fetch(PDO::FETCH_ASSOC);
                $stats['accommodation_bookings'] = (int)($accommodations['accommodation_bookings'] ?? 0);
            } catch (Exception $e) {
                // Table might not exist, keep default 0
            }

            echo json_encode([
                'success' => true,
                'stats' => $stats
            ]);

        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to fetch user stats: ' . $e->getMessage()
            ]);
        }
    }

    private function getUserBookings() {
        $user_id = $_GET['user_id'] ?? null;
        
        if (!$user_id) {
            throw new Exception('User ID required');
        }

        try {
            $stmt = $this->conn->prepare("
                SELECT et.*, e.title as event_title, e.start_datetime as event_date,
                       v.name as venue_name, v.address as venue_address
                FROM event_tickets et
                JOIN events e ON et.event_id = e.id
                LEFT JOIN venues v ON e.venue_id = v.id
                WHERE et.user_id = ?
                ORDER BY et.purchased_at DESC
            ");
            $stmt->execute([$user_id]);
            $tickets = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'tickets' => $tickets
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to fetch user bookings: ' . $e->getMessage());
        }
    }

    private function bookEvent() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Log the input for debugging
        Logger::info("BookEvent Input", ['input' => $input]);
        
        if (!$input) {
            throw new Exception("Invalid JSON input");
        }
        
        $required_fields = ['event_id', 'user_id', 'ticket_type', 'quantity'];
        foreach ($required_fields as $field) {
            if (!isset($input[$field]) || $input[$field] === '' || $input[$field] === null) {
                throw new Exception("Missing required field: $field");
            }
        }

        try {
            $this->conn->beginTransaction();

            // Generate ticket code
            $ticket_code = 'TKT' . strtoupper(uniqid());
            
            // Calculate total price (placeholder logic)
            $price_per_ticket = 50; // Default price
            if ($input['ticket_type'] === 'vip') $price_per_ticket = 100;
            if ($input['ticket_type'] === 'early_bird') $price_per_ticket = 40;
            
            $total_price = $price_per_ticket * $input['quantity'];

            // Create ticket record
            $stmt = $this->conn->prepare("
                INSERT INTO event_tickets (
                    event_id, user_id, ticket_type, ticket_code, 
                    price_paid, payment_method, payment_status, purchased_at
                ) VALUES (?, ?, ?, ?, ?, ?, 'completed', NOW())
            ");
            
            $stmt->execute([
                $input['event_id'],
                $input['user_id'],
                $input['ticket_type'],
                $ticket_code,
                $total_price,
                $input['payment_method'] ?? 'mobile_money'
            ]);

            $this->conn->commit();

            echo json_encode([
                'success' => true,
                'message' => 'Event booked successfully',
                'ticket_code' => $ticket_code,
                'total_price' => $total_price
            ]);

        } catch (Exception $e) {
            $this->conn->rollback();
            Logger::error("BookEvent Database Error", [
                'error' => $e->getMessage(),
                'input' => $input ?? null,
                'trace' => $e->getTraceAsString()
            ]);
            throw new Exception('Failed to book event: ' . $e->getMessage());
        }
    }

    private function submitReview() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        $required_fields = ['event_id', 'user_id', 'rating', 'comment'];
        foreach ($required_fields as $field) {
            if (empty($input[$field])) {
                throw new Exception("Missing required field: $field");
            }
        }

        try {
            $stmt = $this->conn->prepare("
                INSERT INTO event_reviews (
                    event_id, user_id, rating, comment, created_at
                ) VALUES (?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $input['event_id'],
                $input['user_id'],
                $input['rating'],
                $input['comment']
            ]);

            echo json_encode([
                'success' => true,
                'message' => 'Review submitted successfully'
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to submit review: ' . $e->getMessage());
        }
    }

    private function submitSafetyReport() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        $required_fields = ['user_id', 'type', 'description'];
        foreach ($required_fields as $field) {
            if (empty($input[$field])) {
                throw new Exception("Missing required field: $field");
            }
        }

        try {
            // If no event_id is provided, we need to find a default event or create a general report
            // For now, let's use the first available event as a default
            $event_id = $input['event_id'] ?? null;
            
            if (!$event_id) {
                // Get the first available event as default
                $stmt = $this->conn->prepare("SELECT id FROM events ORDER BY id ASC LIMIT 1");
                $stmt->execute();
                $default_event = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($default_event) {
                    $event_id = $default_event['id'];
                } else {
                    throw new Exception('No events available for safety reporting');
                }
            }
            
            $stmt = $this->conn->prepare("
                INSERT INTO safety_reports (
                    reporter_id, event_id, report_type, location_description, description, 
                    severity, status, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, 'reported', NOW())
            ");
            
            $stmt->execute([
                $input['user_id'],
                $event_id,
                $input['type'],
                $input['location'] ?? '',
                $input['description'],
                $input['urgency'] ?? 'medium'
            ]);

            echo json_encode([
                'success' => true,
                'message' => 'Safety report submitted successfully'
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to submit safety report: ' . $e->getMessage());
        }
    }

    private function getUserReviews() {
        $user_id = $_GET['user_id'] ?? null;
        
        if (!$user_id) {
            throw new Exception('User ID required');
        }

        try {
            $stmt = $this->conn->prepare("
                SELECT er.*, e.title as event_title
                FROM event_reviews er
                JOIN events e ON er.event_id = e.id
                WHERE er.user_id = ?
                ORDER BY er.created_at DESC
            ");
            $stmt->execute([$user_id]);
            $reviews = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'reviews' => $reviews
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to fetch user reviews: ' . $e->getMessage());
        }
    }

    private function getUserSafetyReports() {
        $user_id = $_GET['user_id'] ?? null;
        
        if (!$user_id) {
            throw new Exception('User ID required');
        }

        try {
            $stmt = $this->conn->prepare("
                SELECT sr.*, e.title as event_title
                FROM safety_reports sr
                LEFT JOIN events e ON sr.event_id = e.id
                WHERE sr.reporter_id = ?
                ORDER BY sr.created_at DESC
            ");
            $stmt->execute([$user_id]);
            $reports = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'reports' => $reports
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to fetch safety reports: ' . $e->getMessage());
        }
    }

    private function updateProfile() {
        $input = json_decode(file_get_contents('php://input'), true);
        $user_id = $input['user_id'] ?? null;
        
        if (!$user_id) {
            throw new Exception('User ID required');
        }

        try {
            $stmt = $this->conn->prepare("
                UPDATE users 
                SET full_name = ?, email = ?, phone = ?, updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $input['full_name'] ?? '',
                $input['email'] ?? '',
                $input['phone'] ?? '',
                $user_id
            ]);

            echo json_encode([
                'success' => true,
                'message' => 'Profile updated successfully'
            ]);

        } catch (Exception $e) {
            throw new Exception('Failed to update profile: ' . $e->getMessage());
        }
    }
    
    private function testEndpoint() {
        echo json_encode([
            'success' => true,
            'message' => 'Test endpoint working',
            'timestamp' => date('Y-m-d H:i:s'),
            'method' => $_SERVER['REQUEST_METHOD'],
            'input' => json_decode(file_get_contents('php://input'), true)
        ]);
    }
    
    private function sendChatMessage() {
        $input = json_decode(file_get_contents('php://input'), true);
        
        $required_fields = ['event_id', 'user_id', 'message'];
        foreach ($required_fields as $field) {
            if (empty($input[$field])) {
                throw new Exception("Missing required field: $field");
            }
        }
        
        try {
            // Create enhanced simple chat messages table if it doesn't exist
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS simple_chat_messages (
                    id INT PRIMARY KEY AUTO_INCREMENT,
                    event_id INT NOT NULL,
                    user_id INT NOT NULL,
                    message TEXT NOT NULL,
                    message_type ENUM('text', 'image', 'video', 'file') DEFAULT 'text',
                    file_url VARCHAR(500) NULL,
                    file_name VARCHAR(255) NULL,
                    file_size INT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (event_id) REFERENCES events(id),
                    FOREIGN KEY (user_id) REFERENCES users(id)
                )
            ");
            
            $stmt = $this->conn->prepare("
                INSERT INTO simple_chat_messages (event_id, user_id, message, message_type, file_url, file_name, file_size) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $input['event_id'],
                $input['user_id'],
                $input['message'],
                $input['message_type'] ?? 'text',
                $input['file_url'] ?? null,
                $input['file_name'] ?? null,
                $input['file_size'] ?? null
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Message sent successfully'
            ]);
            
        } catch (Exception $e) {
            throw new Exception('Failed to send message: ' . $e->getMessage());
        }
    }
    
    private function getChatMessages() {
        $event_id = $_GET['event_id'] ?? null;
        
        if (!$event_id) {
            throw new Exception('Event ID is required');
        }
        
        try {
            // Create enhanced table if it doesn't exist
            $this->conn->exec("
                CREATE TABLE IF NOT EXISTS simple_chat_messages (
                    id INT PRIMARY KEY AUTO_INCREMENT,
                    event_id INT NOT NULL,
                    user_id INT NOT NULL,
                    message TEXT NOT NULL,
                    message_type ENUM('text', 'image', 'video', 'file') DEFAULT 'text',
                    file_url VARCHAR(500) NULL,
                    file_name VARCHAR(255) NULL,
                    file_size INT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    FOREIGN KEY (event_id) REFERENCES events(id),
                    FOREIGN KEY (user_id) REFERENCES users(id)
                )
            ");
            
            $stmt = $this->conn->prepare("
                SELECT scm.*, u.full_name as sender_name
                FROM simple_chat_messages scm
                JOIN users u ON scm.user_id = u.id
                WHERE scm.event_id = ?
                ORDER BY scm.created_at ASC
                LIMIT 50
            ");
            
            $stmt->execute([$event_id]);
            $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Format messages for frontend
            $formatted_messages = array_map(function($msg) {
                return [
                    'id' => $msg['id'],
                    'content' => $msg['message'],
                    'sender_name' => $msg['sender_name'],
                    'user_id' => $msg['user_id'],
                    'created_at' => $msg['created_at'],
                    'message_type' => $msg['message_type'] ?? 'text',
                    'file_url' => $msg['file_url'] ?? null,
                    'file_name' => $msg['file_name'] ?? null,
                    'file_size' => $msg['file_size'] ?? null
                ];
            }, $messages);
            
            echo json_encode([
                'success' => true,
                'messages' => $formatted_messages
            ]);
            
        } catch (Exception $e) {
            throw new Exception('Failed to get messages: ' . $e->getMessage());
        }
    }
}

// Handle the request
$api = new TouristAPI();
$api->handleRequest();
?>
