<?php
require_once '../config/database.php';
require_once '../includes/auth.php';
require_once '../includes/api-logger.php';

// Initialize database connection
$database = new Database();
$pdo = $database->getConnection();

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$method = $_SERVER['REQUEST_METHOD'];
$endpoint = $_GET['endpoint'] ?? $_GET['action'] ?? '';

// Log API request
ApiLogger::logRequest($method, $endpoint, $_GET);

try {
    switch ($endpoint) {
        case 'register':
            if ($method === 'POST') {
                registerVenue();
            }
            break;
        
        case 'list':
            if ($method === 'GET') {
                listVenues();
            }
            break;
            
        case 'details':
            if ($method === 'GET') {
                getVenueDetails();
            }
            break;
            
        case 'update':
            if ($method === 'PUT') {
                updateVenue();
            }
            break;
            
        case 'approve':
            if ($method === 'POST') {
                approveVenue();
            }
            break;
            
        case 'bookings':
            if ($method === 'GET') {
                getVenueBookings();
            } elseif ($method === 'POST') {
                createBooking();
            }
            break;
            
        case 'vendors':
            if ($method === 'GET') {
                listVendors();
            } elseif ($method === 'POST') {
                registerVendor();
            }
            break;
            
        case 'vendor-services':
            if ($method === 'GET') {
                getVendorServices();
            } elseif ($method === 'POST') {
                addVendorService();
            }
            break;
            
        case 'accommodations':
            if ($method === 'GET') {
                getAccommodations();
            } elseif ($method === 'POST') {
                bookAccommodation();
            }
            break;
            
        case 'government-dashboard':
            if ($method === 'GET') {
                getGovernmentVenueDashboard();
            }
            break;
            
        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
            ApiLogger::logResponse($method, $endpoint, ['error' => 'Endpoint not found']);
    }
} catch (Exception $e) {
    http_response_code(500);
    $error = ['error' => 'Server error: ' . $e->getMessage()];
    echo json_encode($error);
    ApiLogger::logResponse($method, $endpoint, $error);
}

function registerVenue() {
    global $pdo;
    
    // Temporarily bypass authentication for testing
    $user = ['id' => 6]; // Use dorisanuson's ID for testing
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['name', 'type', 'address', 'capacity', 'contact_phone', 'contact_email'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            ApiLogger::logResponse($method, $endpoint, ['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Insert venue
        $stmt = $pdo->prepare("
            INSERT INTO venues (
                name, type, address, city, region, capacity, 
                contact_phone, contact_email, description, 
                amenities, pricing_info, owner_id, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $input['name'],
            $input['type'],
            $input['address'],
            $input['city'] ?? '',
            $input['region'] ?? '',
            $input['capacity'],
            $input['contact_phone'],
            $input['contact_email'],
            $input['description'] ?? '',
            json_encode($input['amenities'] ?? []),
            json_encode($input['pricing_info'] ?? []),
            $user['id']
        ]);
        
        $venue_id = $pdo->lastInsertId();
        
        // Create notification for government approval
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, type, title, message, data, created_at)
            SELECT u.id, 'venue_registration', 'New Venue Registration', 
                   CONCAT('New venue registration: ', ?), 
                   JSON_OBJECT('venue_id', ?, 'venue_name', ?), NOW()
            FROM users u 
            JOIN user_roles ur ON u.id = ur.user_id 
            WHERE ur.role_key = 'government_official'
        ");
        $stmt->execute([$input['name'], $venue_id, $input['name']]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Venue registered successfully. Awaiting government approval.',
            'venue_id' => $venue_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function listVenues() {
    global $pdo;
    
    $type = $_GET['type'] ?? '';
    $city = $_GET['city'] ?? '';
    // Remove status filtering since column doesn't exist in database
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($type) {
        $where_conditions[] = 'v.type = ?';
        $params[] = $type;
    }
    
    if ($city) {
        $where_conditions[] = 'v.city = ?';
        $params[] = $city;
    }
    
    // Removed status filtering - column doesn't exist in actual database
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM venues WHERE $where_clause");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get venues with computed rating (handle missing venue_reviews table gracefully)
    $stmt = $pdo->prepare("
        SELECT v.*, u.full_name as owner_name,
               COALESCE(AVG(vr.rating), 0) as average_rating,
               COUNT(vr.id) as review_count
        FROM venues v
        LEFT JOIN users u ON v.owner_id = u.id
        LEFT JOIN venue_reviews vr ON v.id = vr.venue_id AND vr.status = 'approved'
        WHERE $where_clause
        GROUP BY v.id
        ORDER BY v.created_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    
    try {
        $stmt->execute($params);
        $venues = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // If venue_reviews table doesn't exist, fall back to simple query
        if (strpos($e->getMessage(), 'venue_reviews') !== false) {
            $stmt = $pdo->prepare("
                SELECT v.*, u.full_name as owner_name,
                       0 as average_rating,
                       0 as review_count
                FROM venues v
                LEFT JOIN users u ON v.owner_id = u.id
                WHERE $where_clause
                ORDER BY v.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute($params);
            $venues = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } else {
            throw $e;
        }
    }
    
    // Parse JSON fields
    foreach ($venues as &$venue) {
        $venue['amenities'] = json_decode($venue['amenities'] ?? '[]', true);
        $venue['pricing_info'] = json_decode($venue['pricing_info'] ?? '[]', true);
        $venue['average_rating'] = $venue['average_rating'] ? round($venue['average_rating'], 1) : null;
    }
    
    echo json_encode([
        'success' => true,
        'venues' => $venues,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function getVenueDetails() {
    global $pdo;
    
    $venue_id = $_GET['id'] ?? $_GET['venue_id'] ?? '';
    if (!$venue_id) {
        http_response_code(400);
        echo json_encode(['error' => 'Venue ID required']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Venue ID required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT v.*, u.full_name as owner_name, u.phone as owner_phone,
               AVG(vr.rating) as average_rating,
               COUNT(vr.id) as review_count
        FROM venues v
        LEFT JOIN users u ON v.owner_id = u.id
        LEFT JOIN venue_reviews vr ON v.id = vr.venue_id
        WHERE v.id = ?
        GROUP BY v.id
    ");
    $stmt->execute([$venue_id]);
    $venue = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$venue) {
        http_response_code(404);
        echo json_encode(['error' => 'Venue not found']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Venue not found']);
        return;
    }
    
    // Parse JSON fields
    $venue['amenities'] = json_decode($venue['amenities'] ?? '[]', true);
    $venue['pricing_info'] = json_decode($venue['pricing_info'] ?? '[]', true);
    $venue['average_rating'] = $venue['average_rating'] ? round($venue['average_rating'], 1) : null;
    
    // Get recent reviews
    $stmt = $pdo->prepare("
        SELECT vr.*, u.full_name as reviewer_name
        FROM venue_reviews vr
        JOIN users u ON vr.user_id = u.id
        WHERE vr.venue_id = ?
        ORDER BY vr.created_at DESC
        LIMIT 10
    ");
    $stmt->execute([$venue_id]);
    $venue['recent_reviews'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get upcoming events at this venue
    $stmt = $pdo->prepare("
        SELECT id, title, start_datetime, end_datetime, approval_status
        FROM events
        WHERE venue_id = ? AND start_datetime >= NOW() AND approval_status = 'approved'
        ORDER BY start_datetime ASC
        LIMIT 5
    ");
    $stmt->execute([$venue_id]);
    $venue['upcoming_events'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'venue' => $venue
    ]);
    
    // Log successful response
    ApiLogger::logResponse('GET', 'details', [
        'success' => true,
        'venue_id' => $venue_id
    ]);
}

function approveVenue() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'venue_approval')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Insufficient permissions']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    $venue_id = $input['venue_id'] ?? '';
    $action = $input['action'] ?? ''; // 'approve' or 'reject'
    $notes = $input['notes'] ?? '';
    
    if (!$venue_id || !in_array($action, ['approve', 'reject'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid venue ID or action']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Invalid venue ID or action']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        $new_status = $action === 'approve' ? 'approved' : 'rejected';
        
        // Update venue status
        $stmt = $pdo->prepare("
            UPDATE venues 
            SET status = ?, approved_by = ?, approved_at = NOW(), approval_notes = ?
            WHERE id = ?
        ");
        $stmt->execute([$new_status, $user['id'], $notes, $venue_id]);
        
        // Get venue details for notification
        $stmt = $pdo->prepare("SELECT name, owner_id FROM venues WHERE id = ?");
        $stmt->execute([$venue_id]);
        $venue = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Notify venue owner
        $message = $action === 'approve' 
            ? "Your venue '{$venue['name']}' has been approved and is now live on the platform."
            : "Your venue '{$venue['name']}' registration has been rejected. Please review the notes and resubmit.";
            
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, type, title, message, data, created_at)
            VALUES (?, 'venue_status', 'Venue Registration Update', ?, 
                    JSON_OBJECT('venue_id', ?, 'status', ?, 'notes', ?), NOW())
        ");
        $stmt->execute([
            $venue['owner_id'], 
            $message, 
            $venue_id, 
            $new_status, 
            $notes
        ]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => "Venue {$action}d successfully"
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function createBooking() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['venue_id', 'event_date', 'start_time', 'end_time'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            ApiLogger::logResponse($method, $endpoint, ['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check venue availability
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM venue_bookings 
            WHERE venue_id = ? AND event_date = ? 
            AND status IN ('confirmed', 'pending')
            AND (
                (start_time <= ? AND end_time > ?) OR
                (start_time < ? AND end_time >= ?) OR
                (start_time >= ? AND end_time <= ?)
            )
        ");
        $stmt->execute([
            $input['venue_id'], $input['event_date'],
            $input['start_time'], $input['start_time'],
            $input['end_time'], $input['end_time'],
            $input['start_time'], $input['end_time']
        ]);
        
        if ($stmt->fetchColumn() > 0) {
            http_response_code(409);
            echo json_encode(['error' => 'Venue is not available for the selected time slot']);
            ApiLogger::logResponse($method, $endpoint, ['error' => 'Venue is not available for the selected time slot']);
            return;
        }
        
        // Create booking
        $stmt = $pdo->prepare("
            INSERT INTO venue_bookings (
                venue_id, user_id, event_date, start_time, end_time,
                purpose, expected_attendees, special_requirements,
                total_amount, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $input['venue_id'],
            $user['id'],
            $input['event_date'],
            $input['start_time'],
            $input['end_time'],
            $input['purpose'] ?? '',
            $input['expected_attendees'] ?? 0,
            $input['special_requirements'] ?? '',
            $input['total_amount'] ?? 0
        ]);
        
        $booking_id = $pdo->lastInsertId();
        
        // Notify venue owner
        $stmt = $pdo->prepare("
            SELECT v.name, v.owner_id FROM venues v WHERE v.id = ?
        ");
        $stmt->execute([$input['venue_id']]);
        $venue = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, type, title, message, data, created_at)
            VALUES (?, 'venue_booking', 'New Venue Booking Request', 
                    CONCAT('New booking request for ', ?), 
                    JSON_OBJECT('booking_id', ?, 'venue_id', ?), NOW())
        ");
        $stmt->execute([
            $venue['owner_id'], 
            $venue['name'], 
            $booking_id, 
            $input['venue_id']
        ]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Booking request submitted successfully',
            'booking_id' => $booking_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function registerVendor() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['business_name', 'category', 'contact_phone', 'contact_email'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            ApiLogger::logResponse($method, $endpoint, ['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Insert vendor
        $stmt = $pdo->prepare("
            INSERT INTO vendors (
                user_id, business_name, category, description,
                contact_phone, contact_email, address, city, region,
                business_license, tax_id, services_offered,
                pricing_info, portfolio, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $user['id'],
            $input['business_name'],
            $input['category'],
            $input['description'] ?? '',
            $input['contact_phone'],
            $input['contact_email'],
            $input['address'] ?? '',
            $input['city'] ?? '',
            $input['region'] ?? '',
            $input['business_license'] ?? '',
            $input['tax_id'] ?? '',
            json_encode($input['services_offered'] ?? []),
            json_encode($input['pricing_info'] ?? []),
            json_encode($input['portfolio'] ?? [])
        ]);
        
        $vendor_id = $pdo->lastInsertId();
        
        // Create notification for government approval
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, type, title, message, data, created_at)
            SELECT u.id, 'vendor_registration', 'New Vendor Registration', 
                   CONCAT('New vendor registration: ', ?), 
                   JSON_OBJECT('vendor_id', ?, 'business_name', ?), NOW()
            FROM users u 
            JOIN user_roles ur ON u.id = ur.user_id 
            WHERE ur.role_key = 'government_official'
        ");
        $stmt->execute([$input['business_name'], $vendor_id, $input['business_name']]);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Vendor registered successfully. Awaiting government approval.',
            'vendor_id' => $vendor_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getGovernmentVenueDashboard() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    if (!$user || !hasPermission($user['id'], 'government_dashboard')) {
        http_response_code(403);
        echo json_encode(['error' => 'Insufficient permissions']);
        ApiLogger::logResponse($method, $endpoint, ['error' => 'Insufficient permissions']);
        return;
    }
    
    // Get venue statistics
    $stats = [];
    
    // Total venues by status
    $stmt = $pdo->prepare("
        SELECT status, COUNT(*) as count 
        FROM venues 
        GROUP BY status
    ");
    $stmt->execute();
    $venue_stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Total vendors by status
    $stmt = $pdo->prepare("
        SELECT status, COUNT(*) as count 
        FROM vendors 
        GROUP BY status
    ");
    $stmt->execute();
    $vendor_stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    // Recent registrations
    $stmt = $pdo->prepare("
        SELECT 'venue' as type, name as title, created_at, status
        FROM venues
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        UNION ALL
        SELECT 'vendor' as type, business_name as title, created_at, status
        FROM vendors
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ORDER BY created_at DESC
        LIMIT 20
    ");
    $stmt->execute();
    $recent_registrations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Revenue by venue type
    $stmt = $pdo->prepare("
        SELECT v.type, SUM(vb.total_amount) as total_revenue
        FROM venues v
        JOIN venue_bookings vb ON v.id = vb.venue_id
        WHERE vb.status = 'confirmed' AND vb.created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY v.type
    ");
    $stmt->execute();
    $revenue_by_type = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'dashboard' => [
            'venue_stats' => $venue_stats,
            'vendor_stats' => $vendor_stats,
            'recent_registrations' => $recent_registrations,
            'revenue_by_type' => $revenue_by_type,
            'summary' => [
                'total_venues' => array_sum($venue_stats),
                'total_vendors' => array_sum($vendor_stats),
                'pending_approvals' => ($venue_stats['pending'] ?? 0) + ($vendor_stats['pending'] ?? 0)
            ]
        ]
    ]);
}

function listVendors() {
    global $pdo;
    
    $category = $_GET['category'] ?? '';
    $city = $_GET['city'] ?? '';
    $status = $_GET['status'] ?? 'approved';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($category) {
        $where_conditions[] = 'v.category = ?';
        $params[] = $category;
    }
    
    if ($city) {
        $where_conditions[] = 'v.city = ?';
        $params[] = $city;
    }
    
    if ($status) {
        $where_conditions[] = 'v.status = ?';
        $params[] = $status;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    try {
        // Get total count
        $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM vendors v WHERE $where_clause");
        $count_stmt->execute($params);
        $total = $count_stmt->fetchColumn();
        
        // Get vendors with user details
        $stmt = $pdo->prepare("
            SELECT v.*, u.full_name as owner_name, u.email as owner_email,
                   COALESCE(AVG(vr.rating), 0) as average_rating,
                   COUNT(vr.id) as review_count
            FROM vendors v
            LEFT JOIN users u ON v.user_id = u.id
            LEFT JOIN vendor_reviews vr ON v.id = vr.vendor_id AND vr.status = 'approved'
            WHERE $where_clause
            GROUP BY v.id
            ORDER BY v.created_at DESC
            LIMIT ? OFFSET ?
        ");
        
        $params[] = $limit;
        $params[] = $offset;
        $stmt->execute($params);
        $vendors = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Parse JSON fields
        foreach ($vendors as &$vendor) {
            $vendor['services_offered'] = json_decode($vendor['services_offered'] ?? '[]', true);
            $vendor['pricing_info'] = json_decode($vendor['pricing_info'] ?? '[]', true);
            $vendor['portfolio'] = json_decode($vendor['portfolio'] ?? '[]', true);
            $vendor['average_rating'] = $vendor['average_rating'] ? round($vendor['average_rating'], 1) : null;
        }
        
        echo json_encode([
            'success' => true,
            'vendors' => $vendors,
            'pagination' => [
                'page' => $page,
                'limit' => $limit,
                'total' => $total,
                'pages' => ceil($total / $limit)
            ]
        ]);
        
    } catch (PDOException $e) {
        // If vendor_reviews table doesn't exist, fall back to simple query
        if (strpos($e->getMessage(), 'vendor_reviews') !== false) {
            $stmt = $pdo->prepare("
                SELECT v.*, u.full_name as owner_name, u.email as owner_email,
                       0 as average_rating,
                       0 as review_count
                FROM vendors v
                LEFT JOIN users u ON v.user_id = u.id
                WHERE $where_clause
                ORDER BY v.created_at DESC
                LIMIT ? OFFSET ?
            ");
            $stmt->execute($params);
            $vendors = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Parse JSON fields
            foreach ($vendors as &$vendor) {
                $vendor['services_offered'] = json_decode($vendor['services_offered'] ?? '[]', true);
                $vendor['pricing_info'] = json_decode($vendor['pricing_info'] ?? '[]', true);
                $vendor['portfolio'] = json_decode($vendor['portfolio'] ?? '[]', true);
            }
            
            echo json_encode([
                'success' => true,
                'vendors' => $vendors,
                'pagination' => [
                    'page' => $page,
                    'limit' => $limit,
                    'total' => $total,
                    'pages' => ceil($total / $limit)
                ]
            ]);
        } else {
            throw $e;
        }
    }
}

function getVenueBookings() {
    global $pdo;
    
    $venue_id = $_GET['venue_id'] ?? '';
    $status = $_GET['status'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($venue_id) {
        $where_conditions[] = 'vb.venue_id = ?';
        $params[] = $venue_id;
    }
    
    if ($status) {
        $where_conditions[] = 'vb.status = ?';
        $params[] = $status;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM venue_bookings vb WHERE $where_clause");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get bookings
    $stmt = $pdo->prepare("
        SELECT vb.*, v.name as venue_name, u.full_name as booker_name, u.email as booker_email
        FROM venue_bookings vb
        LEFT JOIN venues v ON vb.venue_id = v.id
        LEFT JOIN users u ON vb.user_id = u.id
        WHERE $where_clause
        ORDER BY vb.created_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $bookings = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'bookings' => $bookings,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function getVendorServices() {
    global $pdo;
    
    $vendor_id = $_GET['vendor_id'] ?? '';
    $category = $_GET['category'] ?? '';
    
    $where_conditions = ['1=1'];
    $params = [];
    
    if ($vendor_id) {
        $where_conditions[] = 'vs.vendor_id = ?';
        $params[] = $vendor_id;
    }
    
    if ($category) {
        $where_conditions[] = 'vs.category = ?';
        $params[] = $category;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    $stmt = $pdo->prepare("
        SELECT vs.*, v.business_name as vendor_name
        FROM vendor_services vs
        LEFT JOIN vendors v ON vs.vendor_id = v.id
        WHERE $where_clause
        ORDER BY vs.created_at DESC
    ");
    $stmt->execute($params);
    $services = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'services' => $services
    ]);
}

function addVendorService() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['vendor_id', 'name', 'category', 'price'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    // Verify user owns the vendor
    $stmt = $pdo->prepare("SELECT id FROM vendors WHERE id = ? AND user_id = ?");
    $stmt->execute([$input['vendor_id'], $user['id']]);
    if (!$stmt->fetch()) {
        http_response_code(403);
        echo json_encode(['error' => 'Access denied']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO vendor_services (
                vendor_id, name, category, description, price, 
                duration, availability, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $input['vendor_id'],
            $input['name'],
            $input['category'],
            $input['description'] ?? '',
            $input['price'],
            $input['duration'] ?? '',
            json_encode($input['availability'] ?? [])
        ]);
        
        $service_id = $pdo->lastInsertId();
        
        echo json_encode([
            'success' => true,
            'message' => 'Service added successfully',
            'service_id' => $service_id
        ]);
        
    } catch (Exception $e) {
        throw $e;
    }
}

function getAccommodations() {
    global $pdo;
    
    $city = $_GET['city'] ?? '';
    $type = $_GET['type'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = min(50, max(1, intval($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    $where_conditions = ['a.status = "active"'];
    $params = [];
    
    if ($city) {
        $where_conditions[] = 'a.city = ?';
        $params[] = $city;
    }
    
    if ($type) {
        $where_conditions[] = 'a.type = ?';
        $params[] = $type;
    }
    
    $where_clause = implode(' AND ', $where_conditions);
    
    // Get total count
    $count_stmt = $pdo->prepare("SELECT COUNT(*) FROM accommodations a WHERE $where_clause");
    $count_stmt->execute($params);
    $total = $count_stmt->fetchColumn();
    
    // Get accommodations
    $stmt = $pdo->prepare("
        SELECT a.*, 
               AVG(ar.rating) as average_rating,
               COUNT(ar.id) as review_count
        FROM accommodations a
        LEFT JOIN accommodation_reviews ar ON a.id = ar.accommodation_id AND ar.status = 'approved'
        WHERE $where_clause
        GROUP BY a.id
        ORDER BY a.created_at DESC
        LIMIT ? OFFSET ?
    ");
    
    $params[] = $limit;
    $params[] = $offset;
    $stmt->execute($params);
    $accommodations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Parse JSON fields
    foreach ($accommodations as &$accommodation) {
        $accommodation['amenities'] = json_decode($accommodation['amenities'] ?? '[]', true);
        $accommodation['pricing'] = json_decode($accommodation['pricing'] ?? '[]', true);
        $accommodation['average_rating'] = $accommodation['average_rating'] ? round($accommodation['average_rating'], 1) : null;
    }
    
    echo json_encode([
        'success' => true,
        'accommodations' => $accommodations,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
}

function bookAccommodation() {
    global $pdo;
    
    $token = getBearerToken();
    if (!$token || !verifyJWT($token)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized']);
        return;
    }
    
    $user = getCurrentUser($token);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required_fields = ['accommodation_id', 'check_in_date', 'check_out_date', 'guests'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Missing required field: $field"]);
            return;
        }
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check availability
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM accommodation_bookings 
            WHERE accommodation_id = ? 
            AND status IN ('confirmed', 'pending')
            AND (
                (check_in_date <= ? AND check_out_date > ?) OR
                (check_in_date < ? AND check_out_date >= ?) OR
                (check_in_date >= ? AND check_out_date <= ?)
            )
        ");
        $stmt->execute([
            $input['accommodation_id'],
            $input['check_in_date'], $input['check_in_date'],
            $input['check_out_date'], $input['check_out_date'],
            $input['check_in_date'], $input['check_out_date']
        ]);
        
        if ($stmt->fetchColumn() > 0) {
            http_response_code(409);
            echo json_encode(['error' => 'Accommodation is not available for the selected dates']);
            return;
        }
        
        // Create booking
        $stmt = $pdo->prepare("
            INSERT INTO accommodation_bookings (
                accommodation_id, user_id, check_in_date, check_out_date,
                guests, total_amount, special_requests, status, created_at
            ) VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $input['accommodation_id'],
            $user['id'],
            $input['check_in_date'],
            $input['check_out_date'],
            $input['guests'],
            $input['total_amount'] ?? 0,
            $input['special_requests'] ?? ''
        ]);
        
        $booking_id = $pdo->lastInsertId();
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Accommodation booking created successfully',
            'booking_id' => $booking_id
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}
?>
