<?php
/**
 * Ghana National Event Management Platform
 * Database Configuration
 */

// Include logger for debugging
require_once __DIR__ . '/../includes/logger.php';

class Database {
    private $host = 'localhost';
    private $db_name = 'gh_tourist_app';
    private $username = 'root';
    private $password = 'pass';
    private $charset = 'utf8mb4';
    private $conn;

    public function getConnection() {
        $this->conn = null;
        
        try {
            Logger::database("Attempting database connection", [
                'host' => $this->host,
                'database' => $this->db_name,
                'username' => $this->username
            ]);
            
            $dsn = "mysql:host=" . $this->host . ";dbname=" . $this->db_name . ";charset=" . $this->charset;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4"
            ];
            
            $this->conn = new PDO($dsn, $this->username, $this->password, $options);
            Logger::database("Database connection successful");
            
        } catch(PDOException $exception) {
            Logger::logDatabaseError($exception->getMessage(), "Connection attempt");
            Logger::error("Database connection failed", [
                'host' => $this->host,
                'database' => $this->db_name,
                'error' => $exception->getMessage()
            ]);
            throw new Exception("Database connection failed: " . $exception->getMessage());
        }

        return $this->conn;
    }

    public function closeConnection() {
        $this->conn = null;
    }
}

// Environment-specific configurations
class Config {
    // Application settings
    const APP_NAME = 'Ghana National Event Management Platform';
    const APP_VERSION = '1.0.0';
    const APP_URL = 'http://localhost/GHTouristApp';
    
    // Security settings
    const JWT_SECRET = 'your-super-secret-jwt-key-change-in-production';
    const JWT_EXPIRY = 86400; // 24 hours
    const PASSWORD_MIN_LENGTH = 8;
    const MAX_LOGIN_ATTEMPTS = 5;
    const LOGIN_LOCKOUT_TIME = 900; // 15 minutes
    
    // File upload settings
    const MAX_FILE_SIZE = 10485760; // 10MB
    const ALLOWED_IMAGE_TYPES = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    const ALLOWED_DOCUMENT_TYPES = ['pdf', 'doc', 'docx'];
    const UPLOAD_PATH = 'uploads/';
    
    // Payment gateway settings
    const PAYSTACK_PUBLIC_KEY = 'pk_test_your_paystack_public_key';
    const PAYSTACK_SECRET_KEY = 'sk_test_your_paystack_secret_key';
    const FLUTTERWAVE_PUBLIC_KEY = 'FLWPUBK_TEST-your_flutterwave_public_key';
    const FLUTTERWAVE_SECRET_KEY = 'FLWSECK_TEST-your_flutterwave_secret_key';
    
    // Mobile Money settings
    const MTN_MOMO_API_KEY = 'your_mtn_momo_api_key';
    const VODAFONE_CASH_API_KEY = 'your_vodafone_cash_api_key';
    const AIRTEL_MONEY_API_KEY = 'your_airtel_money_api_key';
    
    // Email settings
    const SMTP_HOST = 'smtp.gmail.com';
    const SMTP_PORT = 587;
    const SMTP_USERNAME = 'your_email@gmail.com';
    const SMTP_PASSWORD = 'your_app_password';
    const FROM_EMAIL = 'noreply@ghtouristapp.gov.gh';
    const FROM_NAME = 'Ghana Tourist App';
    
    // SMS settings
    const SMS_API_KEY = 'your_sms_api_key';
    const SMS_SENDER_ID = 'GH-TOURIST';
    
    // Government API settings
    const GHANA_CARD_API_URL = 'https://api.nia.gov.gh/verify';
    const GHANA_CARD_API_KEY = 'your_ghana_card_api_key';
    
    // Emergency services
    const EMERGENCY_CONTACTS = [
        'police' => '191',
        'fire' => '192',
        'ambulance' => '193',
        'national_disaster' => '999'
    ];
    
    // Timezone
    const TIMEZONE = 'Africa/Accra';
    
    // Pagination
    const DEFAULT_PAGE_SIZE = 20;
    const MAX_PAGE_SIZE = 100;
    
    // Cache settings
    const CACHE_ENABLED = true;
    const CACHE_DURATION = 3600; // 1 hour
    
    // Rate limiting
    const RATE_LIMIT_REQUESTS = 100;
    const RATE_LIMIT_WINDOW = 3600; // 1 hour
    
    // Logging
    const LOG_LEVEL = 'INFO'; // DEBUG, INFO, WARNING, ERROR
    const LOG_FILE = 'logs/app.log';
    
    public static function get($key, $default = null) {
        return defined("self::$key") ? constant("self::$key") : $default;
    }
}

// Set default timezone
date_default_timezone_set(Config::TIMEZONE);

// Error reporting (disable in production)
if (Config::get('APP_ENV', 'development') === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
}
?>
