<?php
/**
 * Create Missing Directories - Ghana National Event Management Platform
 * This script creates the required directories that are missing and causing Internal Server Error
 */

echo "<!DOCTYPE html>
<html>
<head>
    <title>Create Directories - Ghana Tourist App</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; background: #f5f5f5; }
        .container { max-width: 600px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; background: #d4edda; padding: 15px; border-radius: 4px; margin: 10px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 15px; border-radius: 4px; margin: 10px 0; }
        .info { color: #17a2b8; background: #d1ecf1; padding: 15px; border-radius: 4px; margin: 10px 0; }
        h1 { color: #333; text-align: center; }
        .btn { display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 4px; margin: 10px 5px; }
    </style>
</head>
<body>
    <div class='container'>
        <h1>🗂️ Creating Required Directories</h1>";

$directories = [
    'logs' => 'Application logs and error tracking',
    'uploads' => 'File uploads and media storage',
    'uploads/events' => 'Event images and documents',
    'uploads/venues' => 'Venue images and documents',
    'uploads/users' => 'User profile images',
    'uploads/vendors' => 'Vendor portfolio images',
    'cache' => 'Application cache files',
    'temp' => 'Temporary files'
];

$created = 0;
$errors = 0;

foreach ($directories as $dir => $description) {
    $fullPath = __DIR__ . '/' . $dir;
    
    echo "<div class='info'><strong>Creating:</strong> $dir - $description</div>";
    
    try {
        if (!is_dir($fullPath)) {
            if (mkdir($fullPath, 0755, true)) {
                echo "<div class='success'>✅ Created directory: $dir</div>";
                $created++;
                
                // Create .htaccess for uploads directory to prevent direct access to sensitive files
                if ($dir === 'uploads') {
                    $htaccessContent = "# Prevent direct access to uploaded files\n";
                    $htaccessContent .= "Options -Indexes\n";
                    $htaccessContent .= "<Files ~ \"\\.(php|phtml|php3|php4|php5|pl|py|jsp|asp|sh|cgi)$\">\n";
                    $htaccessContent .= "    deny from all\n";
                    $htaccessContent .= "</Files>\n";
                    
                    file_put_contents($fullPath . '/.htaccess', $htaccessContent);
                    echo "<div class='success'>✅ Created security .htaccess for uploads</div>";
                }
                
                // Create index.php to prevent directory listing
                $indexContent = "<?php\n// Prevent directory access\nheader('HTTP/1.0 403 Forbidden');\nexit('Access denied');\n?>";
                file_put_contents($fullPath . '/index.php', $indexContent);
                
            } else {
                echo "<div class='error'>❌ Failed to create directory: $dir</div>";
                $errors++;
            }
        } else {
            echo "<div class='info'>ℹ️ Directory already exists: $dir</div>";
        }
    } catch (Exception $e) {
        echo "<div class='error'>❌ Error creating $dir: " . htmlspecialchars($e->getMessage()) . "</div>";
        $errors++;
    }
}

// Set proper permissions
echo "<div class='info'><strong>Setting Permissions...</strong></div>";
try {
    chmod(__DIR__ . '/logs', 0755);
    chmod(__DIR__ . '/uploads', 0755);
    chmod(__DIR__ . '/cache', 0755);
    chmod(__DIR__ . '/temp', 0755);
    echo "<div class='success'>✅ Permissions set successfully</div>";
} catch (Exception $e) {
    echo "<div class='error'>❌ Error setting permissions: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Summary
echo "<div style='margin-top: 30px; padding: 20px; background: #f8f9fa; border-radius: 4px;'>";
echo "<h3>📊 Summary</h3>";
echo "<p><strong>Directories created:</strong> $created</p>";
echo "<p><strong>Errors:</strong> $errors</p>";

if ($errors === 0) {
    echo "<div class='success'>
        <h3>🎉 All directories created successfully!</h3>
        <p>Your application should now work without Internal Server Errors.</p>
        <a href='index.html' class='btn'>🏠 Go to Homepage</a>
        <a href='debug.php' class='btn'>🔧 Test Debug Page</a>
    </div>";
} else {
    echo "<div class='error'>
        <h3>⚠️ Some directories could not be created</h3>
        <p>Please check file permissions and try again, or create the directories manually.</p>
    </div>";
}

echo "</div>";
echo "    </div>
</body>
</html>";
?>
