-- Ghana National Event Management Platform
-- Production Database Migration Script
-- Version: 1.0
-- Date: September 17, 2025

-- =====================================================
-- PRODUCTION DATABASE SETUP
-- =====================================================

-- Create production database if not exists
CREATE DATABASE IF NOT EXISTS gh_tourist_app_prod 
CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE gh_tourist_app_prod;

-- =====================================================
-- TOURIST DASHBOARD SPECIFIC TABLES
-- =====================================================

-- Create accommodation bookings table if not exists
CREATE TABLE IF NOT EXISTS accommodation_bookings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    accommodation_id INT NOT NULL,
    user_id INT NOT NULL,
    booking_reference VARCHAR(50) UNIQUE NOT NULL,
    check_in_date DATE NOT NULL,
    check_out_date DATE NOT NULL,
    guests INT NOT NULL DEFAULT 1,
    nights INT NOT NULL,
    room_type ENUM('standard', 'deluxe', 'suite', 'family') DEFAULT 'standard',
    special_requests TEXT,
    total_price DECIMAL(10,2) NOT NULL,
    booking_status ENUM('pending', 'confirmed', 'cancelled', 'completed') DEFAULT 'pending',
    payment_status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    INDEX idx_accommodation_id (accommodation_id),
    INDEX idx_booking_reference (booking_reference),
    INDEX idx_booking_status (booking_status),
    INDEX idx_check_in_date (check_in_date)
) ENGINE=InnoDB;

-- Create event reviews table if not exists
CREATE TABLE IF NOT EXISTS event_reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL,
    user_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    comment TEXT,
    is_verified BOOLEAN DEFAULT FALSE,
    helpful_votes INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_event_id (event_id),
    INDEX idx_user_id (user_id),
    INDEX idx_rating (rating),
    INDEX idx_created_at (created_at),
    UNIQUE KEY unique_user_event_review (user_id, event_id)
) ENGINE=InnoDB;

-- =====================================================
-- SAMPLE DATA FOR ACCOMMODATIONS
-- =====================================================

-- Insert sample accommodations for tourist dashboard
INSERT IGNORE INTO accommodations (
    id, name, type, location, description, price_per_night, 
    rating, amenities, contact_phone, contact_email, 
    available_rooms, image_url, status, created_at
) VALUES 
(1, 'Labadi Beach Hotel', 'hotel', 'Labadi, Accra', 
 'Luxury beachfront hotel with stunning ocean views and world-class amenities', 
 250.00, 4.5, 'WiFi, Pool, Restaurant, Spa, Beach Access, Gym, Conference Rooms', 
 '+233-30-277-2501', 'info@labadibeach.com', 15, 
 'https://via.placeholder.com/300x200?text=Labadi+Beach+Hotel', 'active', NOW()),

(2, 'Kempinski Hotel Gold Coast City', 'hotel', 'Airport City, Accra', 
 'Five-star luxury hotel in the heart of Accra with premium services', 
 400.00, 5.0, 'WiFi, Pool, Gym, Restaurant, Bar, Spa, Business Center, Concierge', 
 '+233-30-258-0000', 'reservations@kempinski-accra.com', 8, 
 'https://via.placeholder.com/300x200?text=Kempinski+Hotel', 'active', NOW()),

(3, 'Movenpick Ambassador Hotel', 'hotel', 'Ridge, Accra', 
 'Business hotel in the diplomatic area with excellent conference facilities', 
 180.00, 4.2, 'WiFi, Restaurant, Bar, Conference Rooms, Business Center', 
 '+233-30-269-4546', 'hotel.accra@movenpick.com', 12, 
 'https://via.placeholder.com/300x200?text=Movenpick+Hotel', 'active', NOW()),

(4, 'Golden Tulip Accra', 'hotel', 'Airport Residential Area, Accra', 
 'Modern hotel near Kotoka International Airport with excellent connectivity', 
 220.00, 4.3, 'WiFi, Pool, Restaurant, Gym, Business Center, Airport Shuttle', 
 '+233-30-275-8900', 'info@goldentulipaccra.com', 20, 
 'https://via.placeholder.com/300x200?text=Golden+Tulip', 'active', NOW()),

(5, 'Coconut Grove Beach Resort', 'resort', 'Cape Coast', 
 'Beachfront resort perfect for weekend getaways with cultural attractions nearby', 
 150.00, 4.0, 'Beach Access, WiFi, Restaurant, Bar, Cultural Tours', 
 '+233-33-213-2100', 'reservations@coconutgrove.com', 25, 
 'https://via.placeholder.com/300x200?text=Coconut+Grove', 'active', NOW()),

(6, 'Kumasi Guest House', 'guesthouse', 'Kumasi', 
 'Comfortable accommodation in the cultural capital with local hospitality', 
 80.00, 3.8, 'WiFi, Restaurant, Parking, Cultural Tours', 
 '+233-32-202-4567', 'info@kumasiguesthouse.com', 18, 
 'https://via.placeholder.com/300x200?text=Kumasi+Guest+House', 'active', NOW());

-- =====================================================
-- SAMPLE EVENT REVIEWS
-- =====================================================

-- Insert sample event reviews
INSERT IGNORE INTO event_reviews (
    event_id, user_id, rating, comment, is_verified, created_at
) VALUES 
(1, 3, 5, 'Amazing event! The organization was top-notch and the performances were incredible. Definitely attending next year!', TRUE, '2025-09-10 14:30:00'),
(1, 4, 4, 'Great music festival with good variety of artists. The venue was perfect and security was excellent.', TRUE, '2025-09-11 09:15:00'),
(2, 3, 5, 'Wonderful cultural experience! Learned so much about Ghanaian traditions and the food was fantastic.', TRUE, '2025-09-12 16:45:00'),
(3, 4, 4, 'Informative tech summit with great networking opportunities. Well organized and relevant content.', TRUE, '2025-09-13 11:20:00'),
(4, 3, 5, 'Best food festival ever! So many delicious local dishes and great atmosphere.', TRUE, '2025-09-14 18:30:00');

-- =====================================================
-- SAMPLE SAFETY REPORTS
-- =====================================================

-- Insert sample safety reports for testing
INSERT IGNORE INTO safety_reports (
    user_id, event_id, report_type, location, description, 
    urgency_level, status, created_at
) VALUES 
(3, 1, 'security', 'Main entrance gate', 'Reported suspicious individual loitering near entrance for extended period', 'medium', 'resolved', '2025-09-15 10:30:00'),
(4, 2, 'infrastructure', 'Parking area', 'Pothole in main parking area causing vehicle damage risk', 'low', 'pending', '2025-09-16 14:15:00'),
(3, NULL, 'medical', 'Labadi Beach Hotel lobby', 'Guest slipped on wet floor, minor injury sustained', 'medium', 'investigating', '2025-09-17 09:45:00');

-- =====================================================
-- SAMPLE ACCOMMODATION BOOKINGS
-- =====================================================

-- Insert sample accommodation bookings
INSERT IGNORE INTO accommodation_bookings (
    accommodation_id, user_id, booking_reference, check_in_date, check_out_date, 
    guests, nights, room_type, total_price, booking_status, payment_status, created_at
) VALUES 
(1, 3, 'ACC001LBH2025', '2025-12-20', '2025-12-23', 2, 3, 'deluxe', 750.00, 'confirmed', 'completed', '2025-09-15 11:30:00'),
(2, 4, 'ACC002KMP2025', '2025-11-15', '2025-11-17', 1, 2, 'suite', 800.00, 'confirmed', 'completed', '2025-09-16 15:45:00'),
(3, 3, 'ACC003MOV2025', '2025-10-10', '2025-10-12', 2, 2, 'standard', 360.00, 'pending', 'pending', '2025-09-17 08:20:00');

-- =====================================================
-- INDEXES FOR PERFORMANCE
-- =====================================================

-- Additional indexes for tourist dashboard queries
CREATE INDEX IF NOT EXISTS idx_events_start_datetime ON events(start_datetime);
CREATE INDEX IF NOT EXISTS idx_events_status ON events(status);
CREATE INDEX IF NOT EXISTS idx_event_tickets_user_payment ON event_tickets(user_id, payment_status);
CREATE INDEX IF NOT EXISTS idx_safety_reports_user_status ON safety_reports(user_id, status);
CREATE INDEX IF NOT EXISTS idx_accommodations_location_status ON accommodations(location, status);

-- =====================================================
-- VIEWS FOR TOURIST DASHBOARD
-- =====================================================

-- Create view for user statistics
CREATE OR REPLACE VIEW user_dashboard_stats AS
SELECT 
    u.id as user_id,
    u.full_name,
    COUNT(DISTINCT et.id) as total_bookings,
    COUNT(DISTINCT CASE WHEN e.start_datetime > NOW() THEN et.id END) as upcoming_events,
    COUNT(DISTINCT ab.id) as accommodation_bookings,
    COUNT(DISTINCT er.id) as reviews_given,
    COUNT(DISTINCT sr.id) as safety_reports_submitted
FROM users u
LEFT JOIN event_tickets et ON u.id = et.user_id AND et.payment_status = 'completed'
LEFT JOIN events e ON et.event_id = e.id
LEFT JOIN accommodation_bookings ab ON u.id = ab.user_id AND ab.booking_status = 'confirmed'
LEFT JOIN event_reviews er ON u.id = er.user_id
LEFT JOIN safety_reports sr ON u.id = sr.user_id
GROUP BY u.id, u.full_name;

-- Create view for popular events
CREATE OR REPLACE VIEW popular_events AS
SELECT 
    e.*,
    v.name as venue_name,
    v.address as venue_address,
    ec.name as category_name,
    COUNT(DISTINCT et.id) as total_bookings,
    AVG(er.rating) as average_rating,
    COUNT(DISTINCT er.id) as review_count
FROM events e
LEFT JOIN venues v ON e.venue_id = v.id
LEFT JOIN event_categories ec ON e.category_id = ec.id
LEFT JOIN event_tickets et ON e.id = et.event_id AND et.payment_status = 'completed'
LEFT JOIN event_reviews er ON e.id = er.event_id
WHERE e.status = 'approved'
GROUP BY e.id
ORDER BY total_bookings DESC, average_rating DESC;

-- =====================================================
-- STORED PROCEDURES FOR TOURIST DASHBOARD
-- =====================================================

DELIMITER //

-- Procedure to get user booking summary
CREATE PROCEDURE IF NOT EXISTS GetUserBookingSummary(IN p_user_id INT)
BEGIN
    SELECT 
        'event_tickets' as booking_type,
        et.id as booking_id,
        e.title as item_name,
        et.ticket_type as booking_details,
        et.price_paid as amount,
        et.payment_status,
        e.start_datetime as event_date,
        et.purchased_at as booking_date
    FROM event_tickets et
    JOIN events e ON et.event_id = e.id
    WHERE et.user_id = p_user_id
    
    UNION ALL
    
    SELECT 
        'accommodation_bookings' as booking_type,
        ab.id as booking_id,
        a.name as item_name,
        CONCAT(ab.room_type, ' - ', ab.nights, ' nights') as booking_details,
        ab.total_price as amount,
        ab.payment_status,
        ab.check_in_date as event_date,
        ab.created_at as booking_date
    FROM accommodation_bookings ab
    JOIN accommodations a ON ab.accommodation_id = a.id
    WHERE ab.user_id = p_user_id
    
    ORDER BY booking_date DESC;
END //

-- Procedure to update user statistics
CREATE PROCEDURE IF NOT EXISTS UpdateUserStats(IN p_user_id INT)
BEGIN
    -- This procedure can be called to refresh cached user statistics
    -- Implementation depends on caching strategy
    SELECT 'User stats updated' as message;
END //

DELIMITER ;

-- =====================================================
-- TRIGGERS FOR DATA INTEGRITY
-- =====================================================

DELIMITER //

-- Trigger to update accommodation availability after booking
CREATE TRIGGER IF NOT EXISTS after_accommodation_booking_insert
AFTER INSERT ON accommodation_bookings
FOR EACH ROW
BEGIN
    UPDATE accommodations 
    SET available_rooms = available_rooms - 1 
    WHERE id = NEW.accommodation_id AND available_rooms > 0;
END //

-- Trigger to restore accommodation availability after booking cancellation
CREATE TRIGGER IF NOT EXISTS after_accommodation_booking_update
AFTER UPDATE ON accommodation_bookings
FOR EACH ROW
BEGIN
    IF OLD.booking_status != 'cancelled' AND NEW.booking_status = 'cancelled' THEN
        UPDATE accommodations 
        SET available_rooms = available_rooms + 1 
        WHERE id = NEW.accommodation_id;
    END IF;
END //

DELIMITER ;

-- =====================================================
-- SECURITY AND PERMISSIONS
-- =====================================================

-- Create application user with limited permissions
CREATE USER IF NOT EXISTS 'gh_tourist_app'@'localhost' IDENTIFIED BY 'secure_app_password_2025!';

-- Grant necessary permissions
GRANT SELECT, INSERT, UPDATE ON gh_tourist_app_prod.* TO 'gh_tourist_app'@'localhost';
GRANT DELETE ON gh_tourist_app_prod.accommodation_bookings TO 'gh_tourist_app'@'localhost';
GRANT DELETE ON gh_tourist_app_prod.event_tickets TO 'gh_tourist_app'@'localhost';
GRANT EXECUTE ON gh_tourist_app_prod.* TO 'gh_tourist_app'@'localhost';

-- Flush privileges
FLUSH PRIVILEGES;

-- =====================================================
-- FINAL VERIFICATION
-- =====================================================

-- Verify all tables exist
SELECT 
    TABLE_NAME,
    TABLE_ROWS,
    CREATE_TIME
FROM INFORMATION_SCHEMA.TABLES 
WHERE TABLE_SCHEMA = 'gh_tourist_app_prod'
ORDER BY TABLE_NAME;

-- Verify sample data
SELECT 'Accommodations' as data_type, COUNT(*) as count FROM accommodations
UNION ALL
SELECT 'Event Reviews' as data_type, COUNT(*) as count FROM event_reviews
UNION ALL
SELECT 'Safety Reports' as data_type, COUNT(*) as count FROM safety_reports
UNION ALL
SELECT 'Accommodation Bookings' as data_type, COUNT(*) as count FROM accommodation_bookings;

-- =====================================================
-- MIGRATION COMPLETE
-- =====================================================

SELECT 'Tourist Dashboard Production Migration Completed Successfully!' as status;
