-- Ghana National Event Management Platform Database Schema
-- Created for comprehensive event management, government oversight, and tourism enhancement

-- Users and Authentication
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),
    profile_image VARCHAR(255),
    status ENUM('active', 'inactive', 'suspended', 'pending_verification') DEFAULT 'pending_verification',
    email_verified BOOLEAN DEFAULT FALSE,
    phone_verified BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    password_changed BOOLEAN DEFAULT FALSE
);

-- User Roles and Permissions
CREATE TABLE user_roles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    role_type ENUM('admin', 'government_official', 'event_organizer', 'venue_owner', 'tourist', 'vendor', 'emergency_responder') NOT NULL,
    role_level ENUM('national', 'regional', 'district', 'local') DEFAULT 'local',
    department VARCHAR(100), -- Ministry of Tourism, Police Service, etc.
    permissions JSON, -- Flexible permissions structure
    assigned_by INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_by) REFERENCES users(id)
);

-- Government Agencies and Officials
CREATE TABLE government_agencies (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    type ENUM('ministry', 'police', 'security', 'health', 'fire_service', 'customs') NOT NULL,
    level ENUM('national', 'regional', 'district') NOT NULL,
    contact_email VARCHAR(100),
    contact_phone VARCHAR(20),
    address TEXT,
    head_official_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (head_official_id) REFERENCES users(id)
);

-- Event Categories and Types
CREATE TABLE event_categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    icon VARCHAR(100),
    color VARCHAR(7), -- Hex color code
    requires_special_permit BOOLEAN DEFAULT FALSE,
    min_security_level ENUM('low', 'medium', 'high', 'maximum') DEFAULT 'low',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Venues and Locations
CREATE TABLE venues (
    id INT PRIMARY KEY AUTO_INCREMENT,
    owner_id INT NOT NULL,
    name VARCHAR(200) NOT NULL,
    type ENUM('event_center', 'beach', 'hotel', 'restaurant', 'pub', 'club', 'outdoor_space', 'cultural_site', 'sports_facility') NOT NULL,
    description TEXT,
    address TEXT NOT NULL,
    city VARCHAR(100) NOT NULL,
    region VARCHAR(100) NOT NULL,
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    capacity INT,
    amenities JSON, -- Parking, WiFi, AC, etc.
    contact_email VARCHAR(100),
    contact_phone VARCHAR(20),
    website VARCHAR(255),
    images JSON, -- Array of image URLs
    license_number VARCHAR(100),
    license_expiry DATE,
    safety_certification VARCHAR(100),
    government_verified BOOLEAN DEFAULT FALSE,
    verification_date TIMESTAMP NULL,
    rating DECIMAL(3,2) DEFAULT 0.00,
    total_reviews INT DEFAULT 0,
    status ENUM('active', 'inactive', 'suspended', 'pending_approval') DEFAULT 'pending_approval',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id)
);

-- Events
CREATE TABLE events (
    id INT PRIMARY KEY AUTO_INCREMENT,
    organizer_id INT NOT NULL,
    venue_id INT,
    category_id INT NOT NULL,
    title VARCHAR(200) NOT NULL,
    description TEXT,
    event_type ENUM('public', 'private', 'government', 'cultural', 'sports', 'business', 'entertainment') NOT NULL,
    start_datetime DATETIME NOT NULL,
    end_datetime DATETIME NOT NULL,
    registration_deadline DATETIME,
    max_attendees INT,
    current_attendees INT DEFAULT 0,
    ticket_price DECIMAL(10,2) DEFAULT 0.00,
    currency VARCHAR(3) DEFAULT 'GHS',
    is_free BOOLEAN DEFAULT FALSE,
    requires_approval BOOLEAN DEFAULT TRUE,
    approval_status ENUM('pending', 'approved', 'rejected', 'cancelled') DEFAULT 'pending',
    approved_by INT,
    approval_date TIMESTAMP NULL,
    government_verified BOOLEAN DEFAULT FALSE,
    security_level ENUM('low', 'medium', 'high', 'maximum') DEFAULT 'medium',
    security_plan TEXT,
    emergency_contacts JSON,
    expected_attendance INT,
    age_restriction ENUM('all_ages', '13+', '16+', '18+', '21+') DEFAULT 'all_ages',
    dress_code VARCHAR(200),
    special_requirements TEXT,
    images JSON,
    promotional_video VARCHAR(255),
    tags JSON, -- Array of tags for search
    social_media JSON, -- Facebook, Instagram, etc.
    livestream_url VARCHAR(255),
    chat_room_enabled BOOLEAN DEFAULT TRUE,
    feedback_enabled BOOLEAN DEFAULT TRUE,
    safety_reporting_enabled BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (organizer_id) REFERENCES users(id),
    FOREIGN KEY (venue_id) REFERENCES venues(id),
    FOREIGN KEY (category_id) REFERENCES event_categories(id),
    FOREIGN KEY (approved_by) REFERENCES users(id)
);

-- Event Permits and Licenses
CREATE TABLE event_permits (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    permit_type ENUM('event_permit', 'liquor_license', 'music_license', 'food_permit', 'security_clearance') NOT NULL,
    permit_number VARCHAR(100),
    issuing_agency_id INT,
    issue_date DATE,
    expiry_date DATE,
    status ENUM('pending', 'approved', 'rejected', 'expired') DEFAULT 'pending',
    fee_amount DECIMAL(10,2),
    fee_paid BOOLEAN DEFAULT FALSE,
    documents JSON, -- Array of document URLs
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (issuing_agency_id) REFERENCES government_agencies(id)
);

-- Event Tickets and Registration
CREATE TABLE event_tickets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    user_id INT NOT NULL,
    ticket_type ENUM('regular', 'vip', 'early_bird', 'group', 'complimentary') DEFAULT 'regular',
    ticket_code VARCHAR(100) UNIQUE NOT NULL,
    qr_code TEXT, -- Base64 encoded QR code
    blockchain_hash VARCHAR(255), -- For blockchain verification
    price_paid DECIMAL(10,2),
    payment_method ENUM('mobile_money', 'card', 'bank_transfer', 'cash', 'complimentary') NOT NULL,
    payment_reference VARCHAR(100),
    payment_status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    check_in_status BOOLEAN DEFAULT FALSE,
    check_in_time TIMESTAMP NULL,
    special_access JSON, -- VIP areas, backstage, etc.
    purchased_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Event Chat Rooms and Messages
CREATE TABLE event_chat_rooms (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    is_moderated BOOLEAN DEFAULT TRUE,
    moderator_id INT,
    max_participants INT DEFAULT 1000,
    current_participants INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (moderator_id) REFERENCES users(id)
);

CREATE TABLE chat_messages (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chat_room_id INT NOT NULL,
    user_id INT NOT NULL,
    message_type ENUM('text', 'image', 'video', 'file', 'system') DEFAULT 'text',
    content TEXT NOT NULL,
    media_url VARCHAR(255),
    is_flagged BOOLEAN DEFAULT FALSE,
    flagged_by INT,
    flag_reason VARCHAR(200),
    is_deleted BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (chat_room_id) REFERENCES event_chat_rooms(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (flagged_by) REFERENCES users(id)
);

-- Safety and Emergency Reporting
CREATE TABLE safety_reports (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    reporter_id INT,
    report_type ENUM('emergency', 'security_concern', 'medical', 'harassment', 'disturbance', 'facility_issue') NOT NULL,
    severity ENUM('low', 'medium', 'high', 'critical') NOT NULL,
    location_description TEXT,
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    description TEXT NOT NULL,
    images JSON,
    status ENUM('reported', 'acknowledged', 'in_progress', 'resolved', 'false_alarm') DEFAULT 'reported',
    assigned_to INT,
    response_time TIMESTAMP NULL,
    resolution_time TIMESTAMP NULL,
    resolution_notes TEXT,
    is_anonymous BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id),
    FOREIGN KEY (reporter_id) REFERENCES users(id),
    FOREIGN KEY (assigned_to) REFERENCES users(id)
);

-- Event Feedback and Reviews
CREATE TABLE event_feedback (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    user_id INT NOT NULL,
    rating INT CHECK (rating >= 1 AND rating <= 5),
    feedback_type ENUM('overall', 'venue', 'organization', 'security', 'amenities') DEFAULT 'overall',
    comment TEXT,
    is_public BOOLEAN DEFAULT TRUE,
    helpful_votes INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Vendors and Service Providers
CREATE TABLE vendors (
    id INT PRIMARY KEY AUTO_INCREMENT,
    owner_id INT NOT NULL,
    business_name VARCHAR(200) NOT NULL,
    business_type ENUM('catering', 'decoration', 'sound_equipment', 'photography', 'security', 'transport', 'accommodation') NOT NULL,
    description TEXT,
    services_offered JSON,
    contact_email VARCHAR(100),
    contact_phone VARCHAR(20),
    address TEXT,
    city VARCHAR(100),
    region VARCHAR(100),
    license_number VARCHAR(100),
    tax_id VARCHAR(100),
    portfolio_images JSON,
    pricing_info JSON,
    availability_calendar JSON,
    rating DECIMAL(3,2) DEFAULT 0.00,
    total_reviews INT DEFAULT 0,
    government_verified BOOLEAN DEFAULT FALSE,
    status ENUM('active', 'inactive', 'suspended', 'pending_approval') DEFAULT 'pending_approval',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id)
);

-- Event-Vendor Bookings
CREATE TABLE event_vendor_bookings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_id INT NOT NULL,
    vendor_id INT NOT NULL,
    service_type VARCHAR(100) NOT NULL,
    booking_status ENUM('requested', 'confirmed', 'cancelled', 'completed') DEFAULT 'requested',
    service_date DATE NOT NULL,
    service_duration INT, -- Hours
    agreed_price DECIMAL(10,2),
    payment_status ENUM('pending', 'partial', 'completed') DEFAULT 'pending',
    special_requirements TEXT,
    contract_document VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES events(id),
    FOREIGN KEY (vendor_id) REFERENCES vendors(id)
);

-- Tourism and Accommodation
CREATE TABLE accommodations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    venue_id INT NOT NULL,
    room_type VARCHAR(100) NOT NULL,
    capacity INT NOT NULL,
    price_per_night DECIMAL(10,2) NOT NULL,
    amenities JSON,
    images JSON,
    availability_calendar JSON,
    booking_policies TEXT,
    cancellation_policy TEXT,
    is_available BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (venue_id) REFERENCES venues(id)
);

-- Payment Transactions
CREATE TABLE payment_transactions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    transaction_type ENUM('ticket_purchase', 'vendor_payment', 'accommodation', 'permit_fee', 'levy_payment') NOT NULL,
    reference_id INT, -- ID of related record (event_id, booking_id, etc.)
    amount DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'GHS',
    payment_method ENUM('mobile_money', 'visa', 'mastercard', 'bank_transfer', 'cash') NOT NULL,
    payment_provider VARCHAR(50), -- MTN, Vodafone, Visa, etc.
    provider_reference VARCHAR(100),
    status ENUM('pending', 'processing', 'completed', 'failed', 'cancelled', 'refunded') DEFAULT 'pending',
    fees DECIMAL(10,2) DEFAULT 0.00,
    tax_amount DECIMAL(10,2) DEFAULT 0.00,
    net_amount DECIMAL(10,2) NOT NULL,
    metadata JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Government Analytics and Reporting
CREATE TABLE analytics_events (
    id INT PRIMARY KEY AUTO_INCREMENT,
    event_type VARCHAR(100) NOT NULL,
    entity_type ENUM('event', 'user', 'venue', 'vendor', 'transaction') NOT NULL,
    entity_id INT NOT NULL,
    data JSON NOT NULL,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- System Notifications
CREATE TABLE notifications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    type ENUM('event_approval', 'payment_confirmation', 'safety_alert', 'system_update', 'emergency') NOT NULL,
    title VARCHAR(200) NOT NULL,
    message TEXT NOT NULL,
    data JSON,
    is_read BOOLEAN DEFAULT FALSE,
    priority ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
    expires_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Create indexes for better performance
CREATE INDEX idx_events_start_date ON events(start_datetime);
CREATE INDEX idx_events_status ON events(approval_status);
CREATE INDEX idx_events_organizer ON events(organizer_id);
CREATE INDEX idx_events_venue ON events(venue_id);
CREATE INDEX idx_tickets_event ON event_tickets(event_id);
CREATE INDEX idx_tickets_user ON event_tickets(user_id);
CREATE INDEX idx_safety_reports_event ON safety_reports(event_id);
CREATE INDEX idx_safety_reports_status ON safety_reports(status);
CREATE INDEX idx_venues_location ON venues(city, region);
CREATE INDEX idx_venues_type ON venues(type);
CREATE INDEX idx_transactions_user ON payment_transactions(user_id);
CREATE INDEX idx_transactions_status ON payment_transactions(status);

-- Insert default event categories
INSERT INTO event_categories (name, description, icon, color, requires_special_permit, min_security_level) VALUES
('Music & Concerts', 'Live music performances and concerts', 'music', '#FF6B6B', TRUE, 'medium'),
('Cultural Festivals', 'Traditional and cultural celebrations', 'culture', '#4ECDC4', TRUE, 'medium'),
('Sports Events', 'Sporting competitions and tournaments', 'sports', '#45B7D1', TRUE, 'high'),
('Business & Corporate', 'Conferences, seminars, and corporate events', 'business', '#96CEB4', FALSE, 'low'),
('Food & Dining', 'Food festivals and culinary events', 'food', '#FFEAA7', FALSE, 'low'),
('Arts & Entertainment', 'Theater, comedy, and entertainment shows', 'arts', '#DDA0DD', FALSE, 'medium'),
('Religious & Spiritual', 'Religious gatherings and spiritual events', 'religion', '#98D8C8', FALSE, 'low'),
('Community & Social', 'Community gatherings and social events', 'community', '#F7DC6F', FALSE, 'low'),
('Educational', 'Workshops, seminars, and educational programs', 'education', '#AED6F1', FALSE, 'low'),
('Government & Official', 'Official government events and ceremonies', 'government', '#F1948A', TRUE, 'maximum');
