-- Schema Updates for Ghana National Event Management Platform
-- Missing tables required by notification system and AI recommendations

-- Update existing notifications table
ALTER TABLE notifications 
ADD COLUMN sender_id INT AFTER user_id,
ADD COLUMN read_at TIMESTAMP NULL AFTER data,
DROP COLUMN is_read,
MODIFY COLUMN type ENUM('event_approval', 'payment_confirmation', 'safety_alert', 'system_update', 'emergency', 'chat_message', 'venue_booking', 'vendor_request', 'government_notice', 'event_reminder') NOT NULL,
ADD FOREIGN KEY (sender_id) REFERENCES users(id);

-- User Notification Preferences
CREATE TABLE user_notification_preferences (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL UNIQUE,
    email_notifications BOOLEAN DEFAULT TRUE,
    sms_notifications BOOLEAN DEFAULT TRUE,
    push_notifications BOOLEAN DEFAULT TRUE,
    event_reminders BOOLEAN DEFAULT TRUE,
    safety_alerts BOOLEAN DEFAULT TRUE,
    payment_confirmations BOOLEAN DEFAULT TRUE,
    chat_messages BOOLEAN DEFAULT FALSE,
    government_notices BOOLEAN DEFAULT TRUE,
    marketing_notifications BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Push Notification Subscriptions
CREATE TABLE push_subscriptions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    endpoint TEXT NOT NULL,
    p256dh_key TEXT,
    auth_key TEXT,
    user_agent TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_endpoint (user_id, endpoint(255))
);

-- Notification Queue for Processing
CREATE TABLE notification_queue (
    id INT PRIMARY KEY AUTO_INCREMENT,
    notification_id INT NOT NULL,
    delivery_method ENUM('email', 'sms', 'push') NOT NULL,
    recipient_info JSON NOT NULL,
    status ENUM('pending', 'processing', 'sent', 'failed', 'cancelled') DEFAULT 'pending',
    attempts INT DEFAULT 0,
    max_attempts INT DEFAULT 3,
    scheduled_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    processed_at TIMESTAMP NULL,
    error_message TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (notification_id) REFERENCES notifications(id) ON DELETE CASCADE
);

-- Notification Delivery Log
CREATE TABLE notification_delivery_log (
    id INT PRIMARY KEY AUTO_INCREMENT,
    notification_id INT NOT NULL,
    user_id INT NOT NULL,
    delivery_method ENUM('in_app', 'email', 'sms', 'push') NOT NULL,
    status ENUM('sent', 'delivered', 'failed', 'bounced') NOT NULL,
    provider_response JSON,
    delivered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (notification_id) REFERENCES notifications(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Venue Bookings (missing from original schema)
CREATE TABLE venue_bookings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    venue_id INT NOT NULL,
    user_id INT NOT NULL,
    event_id INT,
    booking_type ENUM('event', 'accommodation', 'meeting', 'private') DEFAULT 'event',
    start_datetime DATETIME NOT NULL,
    end_datetime DATETIME NOT NULL,
    guest_count INT,
    special_requirements TEXT,
    total_amount DECIMAL(10,2),
    booking_status ENUM('pending', 'confirmed', 'cancelled', 'completed') DEFAULT 'pending',
    payment_status ENUM('pending', 'partial', 'completed', 'refunded') DEFAULT 'pending',
    confirmation_code VARCHAR(50) UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (venue_id) REFERENCES venues(id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (event_id) REFERENCES events(id)
);

-- AI Recommendation Data Tables
CREATE TABLE user_preferences (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL UNIQUE,
    preferred_event_categories JSON,
    preferred_venues JSON,
    preferred_regions JSON,
    budget_range JSON,
    accessibility_needs JSON,
    dietary_restrictions JSON,
    language_preferences JSON,
    notification_frequency ENUM('immediate', 'daily', 'weekly') DEFAULT 'daily',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE user_activity_log (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    activity_type ENUM('event_view', 'event_register', 'venue_view', 'search', 'booking', 'review', 'chat_join') NOT NULL,
    entity_type ENUM('event', 'venue', 'vendor', 'user') NOT NULL,
    entity_id INT NOT NULL,
    metadata JSON,
    session_id VARCHAR(100),
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE recommendation_cache (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    recommendation_type ENUM('events', 'venues', 'vendors', 'similar_users') NOT NULL,
    recommendations JSON NOT NULL,
    score DECIMAL(5,4),
    expires_at TIMESTAMP NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- Chat Room Participants (missing from original schema)
CREATE TABLE chat_room_participants (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chat_room_id INT NOT NULL,
    user_id INT NOT NULL,
    role ENUM('participant', 'moderator', 'admin') DEFAULT 'participant',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    left_at TIMESTAMP NULL,
    is_banned BOOLEAN DEFAULT FALSE,
    ban_reason TEXT,
    banned_by INT,
    banned_at TIMESTAMP NULL,
    FOREIGN KEY (chat_room_id) REFERENCES event_chat_rooms(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (banned_by) REFERENCES users(id),
    UNIQUE KEY unique_room_user (chat_room_id, user_id)
);

-- Emergency Contacts (referenced in safety API)
CREATE TABLE emergency_contacts (
    id INT PRIMARY KEY AUTO_INCREMENT,
    contact_type ENUM('police', 'fire', 'medical', 'security', 'event_coordinator') NOT NULL,
    name VARCHAR(100) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    email VARCHAR(100),
    region VARCHAR(100),
    city VARCHAR(100),
    is_active BOOLEAN DEFAULT TRUE,
    response_time_minutes INT DEFAULT 15,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Add missing venue_reviews table for venues API
CREATE TABLE venue_reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    venue_id INT NOT NULL,
    user_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    review_text TEXT,
    visit_date DATE,
    is_verified BOOLEAN DEFAULT FALSE,
    helpful_votes INT DEFAULT 0,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (venue_id) REFERENCES venues(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_venue_reviews_venue (venue_id),
    INDEX idx_venue_reviews_user (user_id),
    INDEX idx_venue_reviews_rating (rating)
);

-- Update venues table to use computed rating from reviews
-- Remove the static rating and total_reviews columns since they'll be computed
ALTER TABLE venues 
DROP COLUMN rating,
DROP COLUMN total_reviews;

-- Add missing indexes
CREATE INDEX idx_notifications_user ON notifications(user_id);
CREATE INDEX idx_notifications_read ON notifications(read_at);
CREATE INDEX idx_notifications_type ON notifications(type);
CREATE INDEX idx_notifications_sender ON notifications(sender_id);
CREATE INDEX idx_notification_queue_status ON notification_queue(status);
CREATE INDEX idx_notification_queue_scheduled ON notification_queue(scheduled_at);
CREATE INDEX idx_venue_bookings_venue ON venue_bookings(venue_id);
CREATE INDEX idx_venue_bookings_user ON venue_bookings(user_id);
CREATE INDEX idx_venue_bookings_dates ON venue_bookings(start_datetime, end_datetime);
CREATE INDEX idx_user_activity_user ON user_activity_log(user_id);
CREATE INDEX idx_user_activity_type ON user_activity_log(activity_type);
CREATE INDEX idx_recommendation_cache_user ON recommendation_cache(user_id);
CREATE INDEX idx_recommendation_cache_expires ON recommendation_cache(expires_at);
CREATE INDEX idx_chat_participants_room ON chat_room_participants(chat_room_id);
CREATE INDEX idx_chat_participants_user ON chat_room_participants(user_id);

-- Insert default emergency contacts
INSERT INTO emergency_contacts (contact_type, name, phone, email, region) VALUES
('police', 'Ghana Police Service - Emergency', '191', 'emergency@police.gov.gh', 'National'),
('fire', 'Ghana Fire Service - Emergency', '192', 'emergency@fireservice.gov.gh', 'National'),
('medical', 'National Ambulance Service', '193', 'emergency@ambulance.gov.gh', 'National'),
('medical', 'Korle-Bu Teaching Hospital', '0302-674858', 'emergency@kbth.gov.gh', 'Greater Accra'),
('medical', 'Komfo Anokye Teaching Hospital', '0322-022308', 'emergency@kath.gov.gh', 'Ashanti');
