<?php
/**
 * Debug Page - Ghana National Event Management Platform
 * Use this to test database connection and view error logs
 */

// Include configuration and logger
require_once 'config/database.php';
require_once 'includes/logger.php';

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Debug - Ghana Tourist App</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; }
        .card { background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        pre { background: #f8f9fa; padding: 15px; border-radius: 4px; overflow-x: auto; }
        .log-entry { margin: 5px 0; padding: 8px; border-left: 3px solid #ddd; }
        .log-error { border-left-color: #dc3545; background: #f8d7da; }
        .log-warning { border-left-color: #ffc107; background: #fff3cd; }
        .log-info { border-left-color: #17a2b8; background: #d1ecf1; }
        .log-debug { border-left-color: #6c757d; background: #f8f9fa; }
        .log-database { border-left-color: #28a745; background: #d4edda; }
        .log-api { border-left-color: #6f42c1; background: #e2d9f3; }
        h2 { color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }
        .status-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; }
        .status-item { padding: 15px; border-radius: 6px; }
        .status-ok { background: #d4edda; border: 1px solid #c3e6cb; }
        .status-error { background: #f8d7da; border: 1px solid #f5c6cb; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; }
        .btn:hover { background: #0056b3; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Ghana Tourist App - Debug Dashboard</h1>
        
        <div class="card">
            <h2>System Status</h2>
            <div class="status-grid">
                <?php
                // Test PHP version
                $phpVersion = phpversion();
                $phpOk = version_compare($phpVersion, '7.4', '>=');
                ?>
                <div class="status-item <?= $phpOk ? 'status-ok' : 'status-error' ?>">
                    <strong>PHP Version:</strong> <?= $phpVersion ?><br>
                    <small><?= $phpOk ? '✅ Compatible' : '❌ Requires PHP 7.4+' ?></small>
                </div>
                
                <?php
                // Test PDO MySQL
                $pdoOk = extension_loaded('pdo_mysql');
                ?>
                <div class="status-item <?= $pdoOk ? 'status-ok' : 'status-error' ?>">
                    <strong>PDO MySQL:</strong><br>
                    <small><?= $pdoOk ? '✅ Available' : '❌ Not installed' ?></small>
                </div>
                
                <?php
                // Test logs directory
                $logsDir = __DIR__ . '/logs';
                $logsWritable = is_dir($logsDir) && is_writable($logsDir);
                ?>
                <div class="status-item <?= $logsWritable ? 'status-ok' : 'status-error' ?>">
                    <strong>Logs Directory:</strong><br>
                    <small><?= $logsWritable ? '✅ Writable' : '❌ Not writable' ?></small>
                </div>
                
                <?php
                // Test uploads directory
                $uploadsDir = __DIR__ . '/uploads';
                $uploadsWritable = is_dir($uploadsDir) && is_writable($uploadsDir);
                if (!is_dir($uploadsDir)) {
                    mkdir($uploadsDir, 0755, true);
                    $uploadsWritable = is_writable($uploadsDir);
                }
                ?>
                <div class="status-item <?= $uploadsWritable ? 'status-ok' : 'status-error' ?>">
                    <strong>Uploads Directory:</strong><br>
                    <small><?= $uploadsWritable ? '✅ Writable' : '❌ Not writable' ?></small>
                </div>
            </div>
        </div>

        <div class="card">
            <h2>Database Connection Test</h2>
            <?php
            try {
                Logger::info("Debug page - Testing database connection");
                $database = new Database();
                $conn = $database->getConnection();
                
                if ($conn) {
                    echo '<div class="success">✅ <strong>Database connection successful!</strong></div>';
                    
                    // Test if tables exist
                    $stmt = $conn->query("SHOW TABLES");
                    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
                    
                    if (count($tables) > 0) {
                        echo '<div class="info">📊 <strong>Found ' . count($tables) . ' tables:</strong></div>';
                        echo '<pre>' . implode(', ', $tables) . '</pre>';
                    } else {
                        echo '<div class="warning">⚠️ <strong>Database connected but no tables found!</strong><br>';
                        echo 'Please import schema.sql and schema_updates.sql</div>';
                    }
                }
            } catch (Exception $e) {
                echo '<div class="error">❌ <strong>Database connection failed:</strong><br>';
                echo htmlspecialchars($e->getMessage()) . '</div>';
                
                // Common solutions
                echo '<div class="card" style="background: #fff3cd; border: 1px solid #ffeaa7;">';
                echo '<h3>💡 Common Solutions:</h3>';
                echo '<ul>';
                echo '<li>Make sure XAMPP MySQL service is running</li>';
                echo '<li>Create database "gh_tourist_app" in phpMyAdmin</li>';
                echo '<li>Import schema.sql and schema_updates.sql files</li>';
                echo '<li>Check database credentials in config/database.php</li>';
                echo '</ul>';
                echo '</div>';
            }
            ?>
        </div>

        <div class="card">
            <h2>Recent Error Logs</h2>
            <?php
            $logs = Logger::getRecentLogs(30);
            if (empty($logs)) {
                echo '<div class="info">No recent logs found.</div>';
            } else {
                echo '<div style="max-height: 400px; overflow-y: auto;">';
                foreach (array_reverse($logs) as $log) {
                    $log = trim($log);
                    if (empty($log)) continue;
                    
                    $class = 'log-debug';
                    if (strpos($log, '[ERROR]') !== false) $class = 'log-error';
                    elseif (strpos($log, '[WARNING]') !== false) $class = 'log-warning';
                    elseif (strpos($log, '[INFO]') !== false) $class = 'log-info';
                    elseif (strpos($log, '[DATABASE]') !== false) $class = 'log-database';
                    elseif (strpos($log, '[API]') !== false) $class = 'log-api';
                    
                    echo '<div class="log-entry ' . $class . '">' . htmlspecialchars($log) . '</div>';
                }
                echo '</div>';
            }
            ?>
        </div>

        <div class="card">
            <h2>Quick Actions</h2>
            <p>
                <a href="index.html" class="btn">🏠 Go to Homepage</a>
                <a href="events.html" class="btn">📅 Events Page</a>
                <a href="venues.html" class="btn">🏢 Venues Page</a>
                <a href="dashboard.html" class="btn">📊 Dashboard</a>
            </p>
            <p>
                <a href="http://localhost/phpmyadmin" target="_blank" class="btn">🗄️ phpMyAdmin</a>
                <button onclick="location.reload()" class="btn">🔄 Refresh Debug</button>
            </p>
        </div>

        <div class="card">
            <h2>Setup Instructions</h2>
            <ol>
                <li><strong>Start XAMPP:</strong> Make sure Apache and MySQL are running</li>
                <li><strong>Create Database:</strong> Go to phpMyAdmin → Create database "gh_tourist_app"</li>
                <li><strong>Import Schema:</strong> Import schema.sql then schema_updates.sql</li>
                <li><strong>Test Connection:</strong> Refresh this page to verify database connection</li>
                <li><strong>Access App:</strong> Visit <a href="index.html">index.html</a> to start using the app</li>
            </ol>
        </div>
    </div>
</body>
</html>
