<?php
/**
 * API Log Viewer - Debug API calls and responses
 */

// Handle AJAX requests FIRST - before any output
if (isset($_GET['action'])) {
    require_once __DIR__ . '/includes/api-logger.php';
    
    // Set JSON header and prevent any HTML output
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    switch ($_GET['action']) {
        case 'get_logs':
            try {
                $logs = ApiLogger::getRecentLogs(100);
                echo json_encode($logs);
            } catch (Exception $e) {
                echo json_encode(['error' => $e->getMessage()]);
            }
            exit;
            
        case 'clear_logs':
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                try {
                    $logFile = __DIR__ . '/logs/api_calls.log';
                    if (file_exists($logFile)) {
                        file_put_contents($logFile, '');
                    }
                    echo json_encode(['success' => true]);
                } catch (Exception $e) {
                    echo json_encode(['success' => false, 'error' => $e->getMessage()]);
                }
            } else {
                echo json_encode(['success' => false, 'error' => 'POST method required']);
            }
            exit;
            
        default:
            echo json_encode(['error' => 'Invalid action']);
            exit;
    }
}

// Only load ApiLogger for HTML page (not for AJAX requests)
require_once __DIR__ . '/includes/api-logger.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>API Log Viewer - Debug Tool</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Consolas', 'Monaco', monospace;
            background: #1a1a1a;
            color: #e0e0e0;
            padding: 20px;
        }
        
        .header {
            background: #2d2d2d;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid #4CAF50;
        }
        
        .header h1 {
            color: #4CAF50;
            margin-bottom: 10px;
        }
        
        .controls {
            background: #2d2d2d;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .controls button {
            background: #4CAF50;
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 4px;
            cursor: pointer;
            font-family: inherit;
        }
        
        .controls button:hover {
            background: #45a049;
        }
        
        .controls select {
            background: #3d3d3d;
            color: #e0e0e0;
            border: 1px solid #555;
            padding: 8px;
            border-radius: 4px;
        }
        
        .log-entry {
            background: #2d2d2d;
            margin-bottom: 15px;
            border-radius: 8px;
            overflow: hidden;
            border-left: 4px solid #666;
        }
        
        .log-entry.request {
            border-left-color: #2196F3;
        }
        
        .log-entry.response {
            border-left-color: #4CAF50;
        }
        
        .log-entry.js-call {
            border-left-color: #FF9800;
        }
        
        .log-entry.error {
            border-left-color: #f44336;
        }
        
        .log-header {
            background: #3d3d3d;
            padding: 12px 15px;
            cursor: pointer;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .log-header:hover {
            background: #4d4d4d;
        }
        
        .log-type {
            font-weight: bold;
            padding: 4px 8px;
            border-radius: 4px;
            font-size: 12px;
        }
        
        .log-type.request {
            background: #2196F3;
            color: white;
        }
        
        .log-type.response {
            background: #4CAF50;
            color: white;
        }
        
        .log-type.js-call {
            background: #FF9800;
            color: white;
        }
        
        .log-details {
            padding: 15px;
            display: none;
            border-top: 1px solid #555;
        }
        
        .log-details.expanded {
            display: block;
        }
        
        .detail-row {
            margin-bottom: 10px;
            display: flex;
            align-items: flex-start;
        }
        
        .detail-label {
            font-weight: bold;
            color: #4CAF50;
            min-width: 120px;
            margin-right: 10px;
        }
        
        .detail-value {
            flex: 1;
            word-break: break-all;
        }
        
        .json-data {
            background: #1a1a1a;
            padding: 10px;
            border-radius: 4px;
            border: 1px solid #555;
            white-space: pre-wrap;
            font-size: 12px;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .status-code {
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
            font-size: 12px;
        }
        
        .status-200 { background: #4CAF50; color: white; }
        .status-400 { background: #FF9800; color: white; }
        .status-404 { background: #f44336; color: white; }
        .status-500 { background: #9C27B0; color: white; }
        
        .no-logs {
            text-align: center;
            padding: 40px;
            color: #888;
            background: #2d2d2d;
            border-radius: 8px;
        }
        
        .auto-refresh {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .refresh-indicator {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background: #666;
        }
        
        .refresh-indicator.active {
            background: #4CAF50;
            animation: pulse 1s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.5; }
            100% { opacity: 1; }
        }
        
        .loading {
            text-align: center;
            padding: 20px;
            color: #4CAF50;
        }
        
        .error {
            background: #f44336;
            color: white;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>🔍 API Log Viewer</h1>
        <p>Real-time debugging tool for API calls and responses</p>
    </div>
    
    <div class="controls">
        <button onclick="refreshLogs()">🔄 Refresh</button>
        <button onclick="clearLogs()">🗑️ Clear Logs</button>
        
        <select id="filterType" onchange="filterLogs()">
            <option value="all">All Types</option>
            <option value="REQUEST">Requests</option>
            <option value="RESPONSE">Responses</option>
            <option value="JS_CALL">JS Calls</option>
        </select>
        
        <div class="auto-refresh">
            <input type="checkbox" id="autoRefresh" onchange="toggleAutoRefresh()">
            <label for="autoRefresh">Auto-refresh</label>
            <div class="refresh-indicator" id="refreshIndicator"></div>
        </div>
    </div>
    
    <div id="logsContainer">
        <div class="loading">Loading logs...</div>
    </div>

    <script>
        let autoRefreshInterval = null;
        let allLogs = [];
        
        // Load logs on page load
        document.addEventListener('DOMContentLoaded', function() {
            refreshLogs();
        });
        
        async function refreshLogs() {
            try {
                console.log('Fetching logs from: api/logs.php');
                const response = await fetch('api/logs.php?action=get_logs');
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                const contentType = response.headers.get('content-type');
                console.log('Response content-type:', contentType);
                
                const data = await response.json();
                console.log('API Response:', data);
                
                if (data.success) {
                    allLogs = data.data || [];
                    displayLogs(allLogs);
                    console.log('Loaded', allLogs.length, 'log entries');
                } else {
                    throw new Error(data.error || 'Unknown API error');
                }
            } catch (error) {
                console.error('Error loading logs:', error);
                document.getElementById('logsContainer').innerHTML = 
                    `<div class="error">Error loading logs: ${error.message}</div>`;
            }
        }
        
        function displayLogs(logs) {
            const container = document.getElementById('logsContainer');
            
            if (!logs || logs.length === 0) {
                container.innerHTML = '<div class="no-logs">No API logs found. Make some API calls to see them here.</div>';
                return;
            }
            
            container.innerHTML = logs.map((log, index) => {
                const typeClass = log.type.toLowerCase().replace('_', '-');
                const statusClass = log.status_code ? `status-${log.status_code}` : '';
                
                return `
                    <div class="log-entry ${typeClass} ${log.error ? 'error' : ''}" data-type="${log.type}">
                        <div class="log-header" onclick="toggleLogDetails(${index})">
                            <div>
                                <span class="log-type ${typeClass}">${log.type}</span>
                                <strong>${log.endpoint || 'Unknown'}</strong>
                                ${log.status_code ? `<span class="status-code ${statusClass}">${log.status_code}</span>` : ''}
                                ${log.error ? '<span style="color: #f44336;">❌ ERROR</span>' : ''}
                            </div>
                            <div>
                                <small>${log.timestamp}</small>
                                <span style="margin-left: 10px;">▼</span>
                            </div>
                        </div>
                        <div class="log-details" id="details-${index}">
                            ${generateLogDetails(log)}
                        </div>
                    </div>
                `;
            }).join('');
        }
        
        function generateLogDetails(log) {
            let details = `
                <div class="detail-row">
                    <div class="detail-label">Timestamp:</div>
                    <div class="detail-value">${log.timestamp}</div>
                </div>
                <div class="detail-row">
                    <div class="detail-label">Type:</div>
                    <div class="detail-value">${log.type}</div>
                </div>
            `;
            
            if (log.endpoint) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Endpoint:</div>
                        <div class="detail-value">${log.endpoint}</div>
                    </div>
                `;
            }
            
            if (log.method) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Method:</div>
                        <div class="detail-value">${log.method}</div>
                    </div>
                `;
            }
            
            if (log.status_code) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Status Code:</div>
                        <div class="detail-value"><span class="status-code status-${log.status_code}">${log.status_code}</span></div>
                    </div>
                `;
            }
            
            if (log.ip) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">IP Address:</div>
                        <div class="detail-value">${log.ip}</div>
                    </div>
                `;
            }
            
            if (log.params && Object.keys(log.params).length > 0) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Parameters:</div>
                        <div class="detail-value">
                            <div class="json-data">${JSON.stringify(log.params, null, 2)}</div>
                        </div>
                    </div>
                `;
            }
            
            if (log.headers && Object.keys(log.headers).length > 0) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Headers:</div>
                        <div class="detail-value">
                            <div class="json-data">${JSON.stringify(log.headers, null, 2)}</div>
                        </div>
                    </div>
                `;
            }
            
            if (log.response_preview) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Response Preview:</div>
                        <div class="detail-value">
                            <div class="json-data">${JSON.stringify(log.response_preview, null, 2)}</div>
                        </div>
                    </div>
                `;
            }
            
            if (log.response_size) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Response Size:</div>
                        <div class="detail-value">${log.response_size} bytes</div>
                    </div>
                `;
            }
            
            if (log.error) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">Error:</div>
                        <div class="detail-value" style="color: #f44336;">${log.error}</div>
                    </div>
                `;
            }
            
            if (log.user_agent) {
                details += `
                    <div class="detail-row">
                        <div class="detail-label">User Agent:</div>
                        <div class="detail-value">${log.user_agent}</div>
                    </div>
                `;
            }
            
            return details;
        }
        
        function toggleLogDetails(index) {
            const details = document.getElementById(`details-${index}`);
            details.classList.toggle('expanded');
        }
        
        function filterLogs() {
            const filterType = document.getElementById('filterType').value;
            const filteredLogs = filterType === 'all' ? allLogs : allLogs.filter(log => log.type === filterType);
            displayLogs(filteredLogs);
        }
        
        function clearLogs() {
            if (confirm('Are you sure you want to clear all logs?')) {
                fetch('?action=clear_logs', { method: 'POST' })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            refreshLogs();
                        } else {
                            alert('Failed to clear logs: ' + (data.error || 'Unknown error'));
                        }
                    })
                    .catch(error => {
                        console.error('Error clearing logs:', error);
                        alert('Error clearing logs: ' + error.message);
                    });
            }
        }
        
        function toggleAutoRefresh() {
            const checkbox = document.getElementById('autoRefresh');
            const indicator = document.getElementById('refreshIndicator');
            
            if (checkbox.checked) {
                autoRefreshInterval = setInterval(refreshLogs, 3000);
                indicator.classList.add('active');
            } else {
                if (autoRefreshInterval) {
                    clearInterval(autoRefreshInterval);
                    autoRefreshInterval = null;
                }
                indicator.classList.remove('active');
            }
        }
    </script>
</body>
</html>
