<?php
/**
 * Debug script to test events API and database
 */

header('Content-Type: text/plain');
require_once 'config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "=== DATABASE CONNECTION TEST ===\n";
    echo "Database connection: SUCCESS\n\n";
    
    // Check if events table exists
    echo "=== TABLE STRUCTURE TEST ===\n";
    $tables = $conn->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    echo "Available tables: " . implode(', ', $tables) . "\n\n";
    
    if (in_array('events', $tables)) {
        echo "Events table: EXISTS\n";
        
        // Check events count
        $countStmt = $conn->query("SELECT COUNT(*) as count FROM events");
        $count = $countStmt->fetch()['count'];
        echo "Total events in database: $count\n\n";
        
        if ($count > 0) {
            echo "=== SAMPLE EVENTS ===\n";
            $sampleStmt = $conn->query("SELECT id, title, approval_status, start_datetime FROM events LIMIT 3");
            $samples = $sampleStmt->fetchAll();
            foreach ($samples as $event) {
                echo "ID: {$event['id']}, Title: {$event['title']}, Status: {$event['approval_status']}, Date: {$event['start_datetime']}\n";
            }
            echo "\n";
        }
        
        // Test the actual API query
        echo "=== API QUERY TEST ===\n";
        $query = "SELECT e.*, ec.name as category_name, ec.color as category_color,
                         v.name as venue_name, v.address as venue_address, v.city, v.region,
                         u.full_name as organizer_name,
                         COUNT(et.id) as registered_count
                  FROM events e
                  LEFT JOIN event_categories ec ON e.category_id = ec.id
                  LEFT JOIN venues v ON e.venue_id = v.id
                  LEFT JOIN users u ON e.organizer_id = u.id
                  LEFT JOIN event_tickets et ON e.id = et.event_id AND et.payment_status = 'completed'
                  WHERE e.approval_status = 'approved'
                  GROUP BY e.id
                  ORDER BY e.start_datetime ASC
                  LIMIT 5";
        
        try {
            $stmt = $conn->prepare($query);
            $stmt->execute();
            $events = $stmt->fetchAll();
            
            echo "API query executed successfully\n";
            echo "Events returned: " . count($events) . "\n\n";
            
            if (count($events) > 0) {
                echo "=== SAMPLE EVENT DATA ===\n";
                print_r($events[0]);
                echo "\n";
                
                // Test the exact API response format
                echo "=== API RESPONSE FORMAT TEST ===\n";
                $response = [
                    'success' => true,
                    'data' => $events,
                    'pagination' => [
                        'page' => 1,
                        'limit' => 20,
                        'total' => count($events),
                        'pages' => 1
                    ]
                ];
                echo "API Response Structure:\n";
                echo json_encode($response, JSON_PRETTY_PRINT);
            }
            
        } catch (Exception $e) {
            echo "API query failed: " . $e->getMessage() . "\n";
        }
        
    } else {
        echo "Events table: NOT FOUND\n";
        echo "You need to run the database schema setup first.\n";
    }
    
} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
}
?>
