<?php
require_once 'config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "<h2>Tourist API Debug</h2>";
    
    // Test the stats query with user_id 7
    $user_id = 7;
    echo "<h3>Testing stats queries for user_id: $user_id</h3>";
    
    // Test total bookings query
    echo "<h4>1. Total Bookings Query:</h4>";
    try {
        $stmt = $conn->prepare("
            SELECT COUNT(*) as total_bookings 
            FROM event_tickets 
            WHERE user_id = ? AND payment_status = 'completed'
        ");
        $stmt->execute([$user_id]);
        $bookings = $stmt->fetch(PDO::FETCH_ASSOC);
        echo "<p>Result: " . json_encode($bookings) . "</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        
        // Try alternative query
        echo "<p>Trying alternative query without payment_status...</p>";
        try {
            $stmt = $conn->prepare("SELECT COUNT(*) as total_bookings FROM event_tickets WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $bookings = $stmt->fetch(PDO::FETCH_ASSOC);
            echo "<p>Alternative result: " . json_encode($bookings) . "</p>";
        } catch (Exception $e2) {
            echo "<p style='color: red;'>Alternative error: " . $e2->getMessage() . "</p>";
        }
    }
    
    // Test upcoming events query
    echo "<h4>2. Upcoming Events Query:</h4>";
    try {
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT e.id) as upcoming_events
            FROM events e
            JOIN event_tickets et ON e.id = et.event_id
            WHERE et.user_id = ? AND et.payment_status = 'completed'
            AND e.start_datetime > NOW()
        ");
        $stmt->execute([$user_id]);
        $upcoming = $stmt->fetch(PDO::FETCH_ASSOC);
        echo "<p>Result: " . json_encode($upcoming) . "</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        
        // Try simpler query
        echo "<p>Trying simpler query...</p>";
        try {
            $stmt = $conn->prepare("SELECT COUNT(*) as upcoming_events FROM events WHERE start_datetime > NOW()");
            $stmt->execute();
            $upcoming = $stmt->fetch(PDO::FETCH_ASSOC);
            echo "<p>Simple result: " . json_encode($upcoming) . "</p>";
        } catch (Exception $e2) {
            echo "<p style='color: red;'>Simple error: " . $e2->getMessage() . "</p>";
        }
    }
    
    // Test reviews query
    echo "<h4>3. Reviews Query:</h4>";
    try {
        $stmt = $conn->prepare("SELECT COUNT(*) as reviews_given FROM event_reviews WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $reviews = $stmt->fetch(PDO::FETCH_ASSOC);
        echo "<p>Result: " . json_encode($reviews) . "</p>";
    } catch (Exception $e) {
        echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
    }
    
    // Show table structures
    echo "<h3>Table Structure Analysis:</h3>";
    
    $tables = ['event_tickets', 'events', 'event_reviews'];
    foreach ($tables as $table) {
        echo "<h4>Table: $table</h4>";
        try {
            $stmt = $conn->prepare("DESCRIBE $table");
            $stmt->execute();
            $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo "<table border='1' style='border-collapse: collapse;'>";
            echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
            foreach ($columns as $col) {
                echo "<tr>";
                echo "<td>{$col['Field']}</td>";
                echo "<td>{$col['Type']}</td>";
                echo "<td>{$col['Null']}</td>";
                echo "<td>{$col['Key']}</td>";
                echo "<td>{$col['Default']}</td>";
                echo "</tr>";
            }
            echo "</table><br>";
            
            // Show sample data
            $stmt = $conn->prepare("SELECT * FROM $table LIMIT 3");
            $stmt->execute();
            $samples = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if ($samples) {
                echo "<p><strong>Sample data:</strong></p>";
                echo "<pre>" . json_encode($samples, JSON_PRETTY_PRINT) . "</pre>";
            } else {
                echo "<p><em>No data in table</em></p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error accessing $table: " . $e->getMessage() . "</p>";
        }
    }
    
} catch (Exception $e) {
    echo "<p style='color: red;'>Database connection error: " . $e->getMessage() . "</p>";
}
?>
