<?php
/**
 * API Logger - Comprehensive logging for API calls and responses
 */

class ApiLogger {
    private static $logFile = null;
    
    /**
     * Get the log file path
     */
    private static function getLogFile() {
        if (self::$logFile === null) {
            self::$logFile = __DIR__ . '/../logs/api_calls.log';
        }
        return self::$logFile;
    }
    
    /**
     * Log API request
     */
    public static function logRequest($endpoint, $method, $params = [], $headers = []) {
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'REQUEST',
            'endpoint' => $endpoint,
            'method' => $method,
            'params' => $params,
            'headers' => self::sanitizeHeaders($headers),
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ];
        
        self::writeLog($logData);
    }
    
    /**
     * Log API response
     */
    public static function logResponse($endpoint, $statusCode, $response = null, $error = null) {
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'RESPONSE',
            'endpoint' => $endpoint,
            'status_code' => $statusCode,
            'response_size' => $response ? strlen(json_encode($response)) : 0,
            'response_preview' => $response ? self::getResponsePreview($response) : null,
            'error' => $error,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ];
        
        self::writeLog($logData);
    }
    
    /**
     * Log JavaScript API call
     */
    public static function logJsCall($endpoint, $params = [], $response = null, $error = null) {
        $logData = [
            'timestamp' => date('Y-m-d H:i:s'),
            'type' => 'JS_CALL',
            'endpoint' => $endpoint,
            'params' => $params,
            'response_received' => $response !== null,
            'error' => $error,
            'referer' => $_SERVER['HTTP_REFERER'] ?? 'unknown'
        ];
        
        self::writeLog($logData);
    }
    
    /**
     * Write log entry to file
     */
    private static function writeLog($data) {
        $logFile = self::getLogFile();
        $logEntry = json_encode($data) . "\n";
        
        // Ensure logs directory exists
        $logDir = dirname($logFile);
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        // Write log entry
        $result = file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
        
        // Debug: If writing fails, try to understand why
        if ($result === false) {
            error_log("ApiLogger: Failed to write to log file: $logFile");
            error_log("ApiLogger: Directory exists: " . (is_dir($logDir) ? 'yes' : 'no'));
            error_log("ApiLogger: Directory writable: " . (is_writable($logDir) ? 'yes' : 'no'));
        }
    }
    
    /**
     * Sanitize headers for logging
     */
    private static function sanitizeHeaders($headers) {
        $sanitized = [];
        foreach ($headers as $key => $value) {
            if (stripos($key, 'authorization') !== false) {
                $sanitized[$key] = 'Bearer ***';
            } else {
                $sanitized[$key] = $value;
            }
        }
        return $sanitized;
    }
    
    /**
     * Get response preview for logging
     */
    private static function getResponsePreview($response) {
        if (is_array($response)) {
            return [
                'success' => $response['success'] ?? null,
                'data_count' => isset($response['data']) ? (is_array($response['data']) ? count($response['data']) : 'not_array') : null,
                'message' => $response['message'] ?? null,
                'error' => $response['error'] ?? null
            ];
        }
        return substr(json_encode($response), 0, 200);
    }
    
    /**
     * Get recent API logs
     */
    public static function getRecentLogs($limit = 50) {
        $logFile = self::getLogFile();
        if (!file_exists($logFile)) {
            return [];
        }
        
        $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $logs = [];
        
        foreach (array_slice($lines, -$limit) as $line) {
            $logs[] = json_decode($line, true);
        }
        
        return array_reverse($logs);
    }
}
?>
