<?php
/**
 * Ghana National Event Management Platform
 * Authentication and Authorization System
 */

require_once __DIR__ . '/../config/database.php';

class Auth {
    private $db;
    private $conn;

    public function __construct() {
        $this->db = new Database();
        $this->conn = $this->db->getConnection();
    }

    /**
     * Register a new user
     */
    public function register($userData) {
        try {
            // Validate input
            $this->validateRegistrationData($userData);
            
            // Check if user already exists
            if ($this->userExists($userData['email'], $userData['username'])) {
                throw new Exception('User with this email or username already exists');
            }

            // Hash password
            $hashedPassword = password_hash($userData['password'], PASSWORD_DEFAULT);

            // Insert user
            $query = "INSERT INTO users (username, email, password_hash, full_name, phone, status) 
                     VALUES (:username, :email, :password_hash, :full_name, :phone, 'pending_verification')";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':username', $userData['username']);
            $stmt->bindParam(':email', $userData['email']);
            $stmt->bindParam(':password_hash', $hashedPassword);
            $stmt->bindParam(':full_name', $userData['full_name']);
            $stmt->bindParam(':phone', $userData['phone']);
            
            if ($stmt->execute()) {
                $userId = $this->conn->lastInsertId();
                
                // Map frontend account_type to backend role_type
                $roleTypes = [
                    'admin' => 'admin',
                    'government_official' => 'government_official',
                    'event_organizer' => 'event_organizer',
                    'venue_owner' => 'venue_owner',
                    'vendor' => 'vendor',
                    'tourist' => 'tourist',
                    'emergency_responder' => 'emergency_responder'
                ];
                $roleType = $roleTypes[$userData['role_type']] ?? 'tourist';

                // Assign default role
                $this->assignRole($userId, $roleType);
                
                // Send verification email
                $this->sendVerificationEmail($userData['email'], $userId);
                
                return [
                    'success' => true,
                    'message' => 'Registration successful. Please check your email for verification.',
                    'user_id' => $userId
                ];
            }
            
            throw new Exception('Registration failed');
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Login user
     */
    public function login($email, $password, $rememberMe = false) {
        try {
            // Check rate limiting
            if ($this->isRateLimited($email)) {
                throw new Exception('Too many login attempts. Please try again later.');
            }

            // Get user by email or username
            $user = $this->getUserByEmailOrUsername($email);
            if (!$user) {
                $this->recordFailedLogin($email);
                throw new Exception('Invalid credentials');
            }

            // Verify password
            if (!password_verify($password, $user['password_hash'])) {
                $this->recordFailedLogin($email);
                throw new Exception('Invalid credentials');
            }

            // Check user status
            if ($user['status'] !== 'active') {
                throw new Exception('Account is not active. Please verify your email or contact support.');
            }

            // Generate JWT token
            $token = $this->generateJWT($user);
            
            // Update last login
            $this->updateLastLogin($user['id']);
            
            // Clear failed login attempts
            $this->clearFailedLogins($email);

            return [
                'success' => true,
                'message' => 'Login successful',
                'token' => $token,
                'user' => $this->sanitizeUserData($user),
                'expires_in' => Config::JWT_EXPIRY
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Verify JWT token
     */
    public function verifyToken($token) {
        try {
            $parts = explode('.', $token);
            if (count($parts) !== 3) {
                throw new Exception('Invalid token format');
            }

            $header = json_decode(base64_decode($parts[0]), true);
            $payload = json_decode(base64_decode($parts[1]), true);
            $signature = $parts[2];

            // Verify signature
            $expectedSignature = base64_encode(hash_hmac('sha256', $parts[0] . '.' . $parts[1], Config::JWT_SECRET, true));
            if (!hash_equals($expectedSignature, $signature)) {
                throw new Exception('Invalid token signature');
            }

            // Check expiration
            if ($payload['exp'] < time()) {
                throw new Exception('Token has expired');
            }

            // Get current user data
            $user = $this->getUserById($payload['user_id']);
            if (!$user || $user['status'] !== 'active') {
                throw new Exception('User not found or inactive');
            }

            return [
                'success' => true,
                'user' => $this->sanitizeUserData($user),
                'payload' => $payload
            ];

        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Verify JWT token
     */
    public function verifyJWT($token) {
        try {
            $parts = explode('.', $token);
            if (count($parts) !== 3) {
                return false;
            }

            $header = json_decode(base64_decode($parts[0]), true);
            $payload = json_decode(base64_decode($parts[1]), true);
            $signature = $parts[2];

            // Verify signature
            $expectedSignature = base64_encode(hash_hmac('sha256', $parts[0] . '.' . $parts[1], Config::JWT_SECRET, true));
            $expectedSignature = rtrim(strtr($expectedSignature, '+/', '-_'), '=');

            if ($signature !== $expectedSignature) {
                return false;
            }

            // Check expiration
            if (isset($payload['exp']) && $payload['exp'] < time()) {
                return false;
            }

            return $payload;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Check if user has permission
     */
    public function hasPermission($userId, $permission) {
        try {
            $query = "SELECT ur.role_type, ur.permissions, ur.role_level 
                     FROM user_roles ur 
                     WHERE ur.user_id = :user_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->execute();
            
            $role = $stmt->fetch();
            if (!$role) {
                return false;
            }

            // Admin has all permissions
            if ($role['role_type'] === 'admin') {
                return true;
            }

            // Check specific permissions
            $permissions = json_decode($role['permissions'], true) ?? [];
            return in_array($permission, $permissions) || $this->hasRolePermission($role['role_type'], $permission);

        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Assign role to user
     */
    public function assignRole($userId, $roleType, $assignedBy = null, $department = null, $roleLevel = 'local') {
        try {
            // Remove existing roles
            $deleteQuery = "DELETE FROM user_roles WHERE user_id = :user_id";
            $deleteStmt = $this->conn->prepare($deleteQuery);
            $deleteStmt->bindParam(':user_id', $userId);
            $deleteStmt->execute();

            // Insert new role
            $query = "INSERT INTO user_roles (user_id, role_type, role_level, department, permissions, assigned_by) 
                     VALUES (:user_id, :role_type, :role_level, :department, :permissions, :assigned_by)";
            
            $permissions = $this->getDefaultPermissions($roleType);
            $permissionsJson = json_encode($permissions); // Fix: Store in variable first
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':role_type', $roleType);
            $stmt->bindParam(':role_level', $roleLevel);
            $stmt->bindParam(':department', $department);
            $stmt->bindParam(':permissions', $permissionsJson); // Use variable reference
            $stmt->bindParam(':assigned_by', $assignedBy);
            
            return $stmt->execute();

        } catch (Exception $e) {
            error_log("Role assignment error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get default permissions for role
     */
    private function getDefaultPermissions($roleType) {
        $permissions = [
            'admin' => ['*'], // All permissions
            'government_official' => [
                'view_all_events', 'approve_events', 'view_analytics', 'manage_permits',
                'emergency_response', 'view_safety_reports', 'manage_venues'
            ],
            'event_organizer' => [
                'create_events', 'manage_own_events', 'view_analytics', 'manage_tickets',
                'chat_moderation', 'view_feedback'
            ],
            'venue_owner' => [
                'manage_venues', 'view_bookings', 'manage_accommodations', 'view_reviews'
            ],
            'vendor' => [
                'manage_services', 'view_bookings', 'manage_portfolio', 'view_reviews'
            ],
            'tourist' => [
                'book_events', 'leave_feedback', 'join_chats', 'report_safety', 'book_accommodation'
            ],
            'emergency_responder' => [
                'view_safety_reports', 'emergency_response', 'view_event_locations'
            ]
        ];

        return $permissions[$roleType] ?? ['basic_access'];
    }

    /**
     * Check role-based permissions
     */
    private function hasRolePermission($roleType, $permission) {
        $rolePermissions = [
            'admin' => true, // Admin has all permissions
            'government_official' => [
                'view_all_events', 'approve_events', 'view_analytics', 'manage_permits',
                'emergency_response', 'view_safety_reports'
            ],
            'event_organizer' => [
                'create_events', 'manage_own_events', 'view_analytics', 'manage_tickets'
            ],
            'venue_owner' => [
                'manage_venues', 'view_bookings', 'manage_accommodations'
            ],
            'vendor' => [
                'manage_services', 'view_bookings', 'manage_portfolio'
            ],
            'tourist' => [
                'book_events', 'leave_feedback', 'join_chats', 'report_safety'
            ],
            'emergency_responder' => [
                'view_safety_reports', 'emergency_response', 'view_event_locations'
            ]
        ];

        if ($roleType === 'admin') {
            return true;
        }

        return isset($rolePermissions[$roleType]) && in_array($permission, $rolePermissions[$roleType]);
    }

    /**
     * Generate JWT token
     */
    private function generateJWT($user) {
        $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
        $payload = json_encode([
            'user_id' => $user['id'],
            'email' => $user['email'],
            'role' => $this->getUserRole($user['id']),
            'iat' => time(),
            'exp' => time() + Config::JWT_EXPIRY
        ]);

        $base64Header = base64_encode($header);
        $base64Payload = base64_encode($payload);
        $signature = base64_encode(hash_hmac('sha256', $base64Header . '.' . $base64Payload, Config::JWT_SECRET, true));

        return $base64Header . '.' . $base64Payload . '.' . $signature;
    }

    /**
     * Get user role
     */
    private function getUserRole($userId) {
        $query = "SELECT role_type, role_level, department FROM user_roles WHERE user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Validate registration data
     */
    private function validateRegistrationData($data) {
        $required = ['username', 'email', 'password', 'full_name'];
        
        foreach ($required as $field) {
            if (empty($data[$field])) {
                throw new Exception("$field is required");
            }
        }

        if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Invalid email format');
        }

        if (strlen($data['password']) < Config::PASSWORD_MIN_LENGTH) {
            throw new Exception('Password must be at least ' . Config::PASSWORD_MIN_LENGTH . ' characters long');
        }

        if (strlen($data['username']) < 3) {
            throw new Exception('Username must be at least 3 characters long');
        }
    }

    /**
     * Check if user exists
     */
    private function userExists($email, $username) {
        $query = "SELECT id FROM users WHERE email = :email OR username = :username";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':username', $username);
        $stmt->execute();
        
        return $stmt->fetch() !== false;
    }

    /**
     * Get user by email or username
     */
    private function getUserByEmailOrUsername($emailOrUsername) {
        try {
            $query = "SELECT u.*, ur.role_type 
                     FROM users u 
                     LEFT JOIN user_roles ur ON u.id = ur.user_id 
                     WHERE u.email = :email OR u.username = :username 
                     LIMIT 1";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':email', $emailOrUsername);
            $stmt->bindParam(':username', $emailOrUsername);
            $stmt->execute();
            
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Debug logging to see what we get from database
            error_log("getUserByEmailOrUsername DEBUG - Input: " . $emailOrUsername);
            error_log("getUserByEmailOrUsername DEBUG - User data: " . json_encode($user));
            error_log("getUserByEmailOrUsername DEBUG - Role type: " . ($user['role_type'] ?? 'NULL'));
            
            return $user;
        } catch (Exception $e) {
            error_log("Database error in getUserByEmailOrUsername: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get user by ID
     */
    private function getUserById($id) {
        $query = "SELECT * FROM users WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        return $stmt->fetch();
    }

    /**
     * Update last login
     */
    private function updateLastLogin($userId) {
        $query = "UPDATE users SET last_login = NOW() WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $userId);
        $stmt->execute();
    }

    /**
     * Rate limiting functions
     */
    private function isRateLimited($email) {
        // Implementation for rate limiting
        return false; // Simplified for now
    }

    private function recordFailedLogin($email) {
        // Implementation for recording failed login attempts
    }

    private function clearFailedLogins($email) {
        // Implementation for clearing failed login attempts
    }

    /**
     * Send verification email
     */
    private function sendVerificationEmail($email, $userId) {
        // Implementation for sending verification email
        // This would integrate with your email service
    }

    /**
     * Sanitize user data for response
     */
    private function sanitizeUserData($user) {
        unset($user['password_hash']);
        return $user;
    }
}

/**
 * Middleware for protecting routes
 */
class AuthMiddleware {
    public static function requireAuth() {
        $headers = getallheaders();
        $token = null;

        if (isset($headers['Authorization'])) {
            $authHeader = $headers['Authorization'];
            if (preg_match('/Bearer\s+(.*)$/i', $authHeader, $matches)) {
                $token = $matches[1];
            }
        }

        if (!$token) {
            http_response_code(401);
            echo json_encode(['error' => 'Authorization token required']);
            exit;
        }

        $auth = new Auth();
        $result = $auth->verifyToken($token);

        if (!$result['success']) {
            http_response_code(401);
            echo json_encode(['error' => $result['message']]);
            exit;
        }

        return $result['user'];
    }

    public static function requirePermission($permission) {
        $user = self::requireAuth();
        
        $auth = new Auth();
        if (!$auth->hasPermission($user['id'], $permission)) {
            http_response_code(403);
            echo json_encode(['error' => 'Insufficient permissions']);
            exit;
        }

        return $user;
    }

    public static function requireRole($roles) {
        $user = self::requireAuth();
        
        if (!is_array($roles)) {
            $roles = [$roles];
        }

        $auth = new Auth();
        $userRole = $auth->getUserRole($user['id']);
        
        if (!in_array($userRole['role_type'], $roles)) {
            http_response_code(403);
            echo json_encode(['error' => 'Access denied']);
            exit;
        }

        return $user;
    }
}
?>
