<?php
/**
 * Ghana National Event Management Platform
 * Error Logging System
 */

class Logger {
    private static $logFile;
    private static $initialized = false;
    
    public static function init() {
        if (self::$initialized) return;
        
        // Create logs directory if it doesn't exist
        $logDir = dirname(__DIR__) . '/logs';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        self::$logFile = $logDir . '/app_' . date('Y-m-d') . '.log';
        self::$initialized = true;
        
        // Log application startup
        self::info("Application started", ['timestamp' => date('Y-m-d H:i:s'), 'ip' => $_SERVER['REMOTE_ADDR'] ?? 'CLI']);
    }
    
    public static function error($message, $context = []) {
        self::log('ERROR', $message, $context);
    }
    
    public static function warning($message, $context = []) {
        self::log('WARNING', $message, $context);
    }
    
    public static function info($message, $context = []) {
        self::log('INFO', $message, $context);
    }
    
    public static function debug($message, $context = []) {
        self::log('DEBUG', $message, $context);
    }
    
    public static function database($message, $context = []) {
        self::log('DATABASE', $message, $context);
    }
    
    public static function api($message, $context = []) {
        self::log('API', $message, $context);
    }
    
    public static function log($level, $message, $context = [], $source = null) {
        if (!self::$initialized) {
            self::init();
        }
        
        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' | Context: ' . json_encode($context) : '';
        $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 3);
        $caller = $source ?: (isset($backtrace[2]) ? basename($backtrace[2]['file']) . ':' . $backtrace[2]['line'] : 'unknown');
        
        $logEntry = "[$timestamp] [$level] [$caller] $message$contextStr" . PHP_EOL;
        
        // Write to file
        file_put_contents(self::$logFile, $logEntry, FILE_APPEND | LOCK_EX);
        
        // Also output to error log for critical errors
        if ($level === 'ERROR') {
            error_log("GH Tourist App - $message");
        }
    }
    
    public static function logException($exception, $context = []) {
        $message = "Exception: " . $exception->getMessage();
        $context['file'] = $exception->getFile();
        $context['line'] = $exception->getLine();
        $context['trace'] = $exception->getTraceAsString();
        
        self::error($message, $context);
    }
    
    public static function logDatabaseError($error, $query = null) {
        $context = ['error' => $error];
        if ($query) {
            $context['query'] = $query;
        }
        self::database("Database error occurred", $context);
    }
    
    public static function logApiRequest($endpoint, $method, $params = [], $response = null) {
        $context = [
            'endpoint' => $endpoint,
            'method' => $method,
            'params' => $params,
            'response_code' => http_response_code(),
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
        ];
        
        if ($response) {
            $context['response'] = $response;
        }
        
        self::api("API request processed", $context);
    }
    
    public static function getLogFile() {
        return self::$logFile;
    }
    
    public static function getRecentLogs($lines = 50) {
        if (!file_exists(self::$logFile)) {
            return [];
        }
        
        $file = file(self::$logFile);
        return array_slice($file, -$lines);
    }
}

// Initialize logger
Logger::init();

// Set custom error handler
set_error_handler(function($severity, $message, $file, $line) {
    $context = [
        'severity' => $severity,
        'file' => $file,
        'line' => $line
    ];
    
    switch ($severity) {
        case E_ERROR:
        case E_CORE_ERROR:
        case E_COMPILE_ERROR:
        case E_USER_ERROR:
            Logger::error("PHP Error: $message", $context);
            break;
        case E_WARNING:
        case E_CORE_WARNING:
        case E_COMPILE_WARNING:
        case E_USER_WARNING:
            Logger::warning("PHP Warning: $message", $context);
            break;
        default:
            Logger::debug("PHP Notice: $message", $context);
            break;
    }
    
    return false; // Let PHP handle the error normally
});

// Set exception handler
set_exception_handler(function($exception) {
    Logger::logException($exception);
    
    // Show user-friendly error page in production
    if (Config::get('APP_ENV', 'development') !== 'development') {
        http_response_code(500);
        include 'error_pages/500.html';
        exit;
    }
});
?>
