// GH Tourist App - Admin Dashboard JavaScript
// Clean, unified implementation

class AdminManager {
    constructor() {
        this.apiBase = 'api/admin.php';
        this.token = localStorage.getItem('auth_token') || 'dummy_token';
        this.user = JSON.parse(localStorage.getItem('user_data') || '{"id": 1, "username": "admin"}');
        this.currentSection = 'dashboard';
        this.init();
    }

    init() {
        console.log('🚀 AdminManager initialized');
        this.setupEventListeners();
        this.loadDashboardStats();
    }

    setupEventListeners() {
        // Sidebar navigation
        document.querySelectorAll('.sidebar-item').forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const section = item.getAttribute('data-section') || item.getAttribute('onclick')?.match(/showSection\('([^']+)'/)?.[1];
                if (section) {
                    this.showSection(section, item);
                }
            });
        });
    }

    showSection(sectionId, activeItem = null) {
        console.log(`🔄 Switching to section: ${sectionId}`);
        
        // Remove active class from all sidebar items
        document.querySelectorAll('.sidebar-item').forEach(item => {
            item.classList.remove('active');
        });
        
        // Add active class to clicked item
        if (activeItem) {
            activeItem.classList.add('active');
        }
        
        // Hide all content sections
        document.querySelectorAll('.content-section').forEach(section => {
            section.classList.remove('active');
            section.style.display = 'none';
        });
        
        // Show selected section
        const targetSection = document.getElementById(sectionId);
        if (targetSection) {
            targetSection.classList.add('active');
            targetSection.style.display = 'block';
        }
        
        // Map section IDs to data loading methods
        const sectionMap = {
            'dashboard-section': 'dashboard',
            'pending-users-section': 'pending-users',
            'users-section': 'users',
            'events-section': 'events',
            'venues-section': 'venues',
            'categories-section': 'categories',
            'vendors-section': 'vendors',
            'government-section': 'government-agencies',
            'safety-section': 'safety-reports',
            'payments-section': 'payment-transactions',
            'permits-section': 'event-permits',
            'analytics-section': 'analytics',
            'roles-section': 'roles',
            'logs-section': 'system-logs'
        };
        
        const section = sectionMap[sectionId];
        if (section) {
            this.currentSection = section;
            this.loadSectionData(section);
        }
    }

    async loadSectionData(section) {
        console.log(`📊 Loading data for section: ${section}`);
        
        try {
            switch(section) {
                case 'dashboard':
                    await this.loadDashboardStats();
                    break;
                case 'pending-users':
                    await this.loadPendingUsers();
                    break;
                case 'users':
                    await this.loadUsers();
                    break;
                case 'events':
                    await this.loadEvents();
                    break;
                case 'venues':
                    await this.loadVenues();
                    break;
                case 'categories':
                    await this.loadCategories();
                    break;
                case 'vendors':
                    await this.loadVendors();
                    break;
                case 'government-agencies':
                    await this.loadGovernmentAgencies();
                    break;
                case 'safety-reports':
                    await this.loadSafetyReports();
                    break;
                case 'payment-transactions':
                    await this.loadPaymentTransactions();
                    break;
                case 'event-permits':
                    await this.loadEventPermits();
                    break;
                case 'roles':
                    await this.loadRoles();
                    break;
                case 'system-logs':
                    await this.loadSystemLogs();
                    break;
                case 'analytics':
                    await this.loadAnalytics();
                    break;
            }
        } catch (error) {
            console.error(`❌ Error loading ${section} data:`, error);
            this.showNotification(`Error loading ${section} data: ${error.message}`, 'error');
        }
    }

    async loadDashboardStats() {
        console.log('📈 Loading dashboard statistics...');
        
        try {
            const response = await fetch(`${this.apiBase}/stats`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayDashboardStats(result.data);
                } else {
                    throw new Error(result.error || 'Failed to load dashboard stats');
                }
            } else {
                throw new Error('Failed to fetch dashboard stats');
            }
        } catch (error) {
            console.error('❌ Error loading dashboard stats:', error);
            // Set default values if API fails
            this.displayDashboardStats({
                total_users: 0,
                total_events: 0,
                total_venues: 0,
                pending_approvals: 0
            });
        }
    }

    displayDashboardStats(data) {
        const elements = {
            'totalUsers': data.total_users || 0,
            'totalEvents': data.total_events || 0,
            'totalVenues': data.total_venues || 0,
            'pendingApprovals': data.pending_approvals || 0
        };

        Object.entries(elements).forEach(([id, value]) => {
            const element = document.getElementById(id);
            if (element) {
                element.textContent = value;
            }
        });

        console.log('✅ Dashboard stats updated:', elements);
    }

    async loadPendingUsers() {
        console.log('🔍 Loading pending users...');
        console.log('API Base:', this.apiBase);
        console.log('Token:', this.token ? 'Present' : 'Missing');
        
        const tbody = document.getElementById('pendingUsersTableBody');
        if (!tbody) {
            console.error('❌ pendingUsersTableBody element not found');
            return;
        }

        // Show loading state
        tbody.innerHTML = '<tr><td colspan="5" class="text-center">Loading pending users...</td></tr>';
        
        try {
            const url = `${this.apiBase}/users/pending`;
            console.log('📡 Making API call to:', url);
            
            const response = await fetch(url, {
                headers: { 
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });
            
            console.log('📥 Response status:', response.status);
            console.log('📥 Response ok:', response.ok);
            
            if (response.ok) {
                const result = await response.json();
                console.log('📄 API Response:', result);
                
                if (result.success) {
                    console.log('✅ Success! Data received:', result.data);
                    this.displayPendingUsers(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load pending users');
                }
            } else {
                const errorText = await response.text();
                console.error('❌ HTTP Error:', response.status, errorText);
                throw new Error(`HTTP ${response.status}: ${errorText}`);
            }
        } catch (error) {
            console.error('💥 Error loading pending users:', error);
            tbody.innerHTML = `<tr><td colspan="5" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            this.showNotification(`Error loading pending users: ${error.message}`, 'error');
        }
    }

    displayPendingUsers(users) {
        const tbody = document.getElementById('pendingUsersTableBody');
        if (!tbody) return;

        console.log('🎨 Displaying pending users:', users);

        if (!users || users.length === 0) {
            tbody.innerHTML = '<tr><td colspan="5" class="text-center">No pending users found</td></tr>';
            return;
        }

        tbody.innerHTML = users.map(user => `
            <tr>
                <td>${user.full_name || user.username || 'N/A'}</td>
                <td>${user.email || 'N/A'}</td>
                <td>${user.phone || 'N/A'}</td>
                <td>${user.created_at ? new Date(user.created_at).toLocaleDateString() : 'N/A'}</td>
                <td>
                    <button class="btn btn-success btn-sm" onclick="approveUser(${user.id})">
                        <i class="fas fa-check"></i> Approve
                    </button>
                    <button class="btn btn-danger btn-sm" onclick="rejectUser(${user.id})">
                        <i class="fas fa-times"></i> Reject
                    </button>
                </td>
            </tr>
        `).join('');

        console.log('✅ Pending users table updated');
    }

    async loadUsers() {
        console.log('👥 Loading users...');
        
        try {
            const response = await fetch(`${this.apiBase}/users`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayUsers(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load users');
                }
            } else {
                throw new Error('Failed to fetch users');
            }
        } catch (error) {
            console.error('❌ Error loading users:', error);
            this.showNotification('Error loading users: ' + error.message, 'error');
        }
    }

    displayUsers(users) {
        const tbody = document.getElementById('usersTableBody');
        if (!tbody) return;

        if (!users || users.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">No users found</td></tr>';
            return;
        }

        tbody.innerHTML = users.map(user => `
            <tr>
                <td>${user.full_name || user.username || 'N/A'}</td>
                <td>${user.email || 'N/A'}</td>
                <td>${user.role_type || 'No Role'}</td>
                <td><span class="status-badge status-${user.status}">${user.status}</span></td>
                <td>${user.created_at ? new Date(user.created_at).toLocaleDateString() : 'N/A'}</td>
                <td>
                    <button class="btn btn-primary btn-sm">Edit</button>
                    <button class="btn btn-danger btn-sm">Delete</button>
                </td>
            </tr>
        `).join('');
    }

    async loadEvents() {
        console.log('🎉 Loading events...');
        
        try {
            const response = await fetch(`${this.apiBase}/events`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayEvents(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load events');
                }
            } else {
                throw new Error('Failed to fetch events');
            }
        } catch (error) {
            console.error('❌ Error loading events:', error);
            this.showNotification('Error loading events: ' + error.message, 'error');
        }
    }

    displayEvents(events) {
        const tbody = document.getElementById('eventsTableBody');
        if (!tbody) return;

        if (!events || events.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No events found</td></tr>';
            return;
        }

        tbody.innerHTML = events.map(event => `
            <tr>
                <td>${event.title || 'N/A'}</td>
                <td>${event.organizer_name || 'Unknown'}</td>
                <td>${event.venue_name || 'TBD'}</td>
                <td>${event.start_datetime ? new Date(event.start_datetime).toLocaleDateString() : 'N/A'}</td>
                <td><span class="status-badge status-${event.approval_status}">${event.approval_status}</span></td>
                <td>${event.current_attendees || 0}/${event.max_attendees || '∞'}</td>
                <td>
                    <button class="btn btn-primary btn-sm">View</button>
                    <button class="btn btn-success btn-sm">Approve</button>
                    <button class="btn btn-danger btn-sm">Reject</button>
                </td>
            </tr>
        `).join('');
    }

    async loadVenues() {
        console.log('🏢 Loading venues...');
        
        try {
            const response = await fetch(`${this.apiBase}/venues`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayVenues(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load venues');
                }
            } else {
                throw new Error('Failed to fetch venues');
            }
        } catch (error) {
            console.error('❌ Error loading venues:', error);
            this.showNotification('Error loading venues: ' + error.message, 'error');
        }
    }

    displayVenues(venues) {
        const tbody = document.getElementById('venuesTableBody');
        if (!tbody) return;

        if (!venues || venues.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No venues found</td></tr>';
            return;
        }

        tbody.innerHTML = venues.map(venue => `
            <tr>
                <td>${venue.name || 'N/A'}</td>
                <td>${venue.type || 'N/A'}</td>
                <td>${venue.city || 'N/A'}, ${venue.region || 'N/A'}</td>
                <td>${venue.capacity || 'N/A'}</td>
                <td><span class="status-badge status-${venue.status}">${venue.status}</span></td>
                <td>${venue.event_count || 0}</td>
                <td>
                    <button class="btn btn-primary btn-sm" onclick="viewVenue(${venue.id})">View</button>
                    <button class="btn btn-success btn-sm" onclick="verifyVenue(${venue.id})">Verify</button>
                    <button class="btn btn-danger btn-sm" onclick="suspendVenue(${venue.id})">Suspend</button>
                </td>
            </tr>
        `).join('');
    }

    async loadCategories(page = 1, search = '') {
        console.log('📂 Loading categories...');
        
        try {
            let url = `${this.apiBase}/categories?page=${page}`;
            if (search) {
                url += `&search=${encodeURIComponent(search)}`;
            }
            
            console.log('📡 Making API call to:', url);

            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📥 Categories response status:', response.status);

            const data = await response.json();
            console.log('📄 Categories API Response:', data);
            
            if (data.success) {
                console.log('✅ Categories loaded successfully:', data.data);
                this.renderCategories(data.data || []);
            } else {
                console.error('❌ Categories API returned error:', data.error);
                throw new Error(data.error || 'Failed to load categories');
            }
        } catch (error) {
            console.error('💥 Error loading categories:', error);
            const tbody = document.getElementById('categoriesTableBody');
            if (tbody) {
                tbody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            }
            this.showNotification(`Error loading categories: ${error.message}`, 'error');
        }
    }

    renderCategories(categories) {
        const tbody = document.getElementById('categoriesTableBody');
        if (!tbody) return;

        if (!categories || categories.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">No categories found</td></tr>';
            return;
        }

        tbody.innerHTML = categories.map(category => `
            <tr>
                <td>${category.id}</td>
                <td>${this.escapeHtml(category.name)}</td>
                <td>
                    <span class="badge" style="background-color: ${category.color}; color: white;">
                        ${category.color}
                    </span>
                </td>
                <td>${this.escapeHtml(category.description || 'No description')}</td>
                <td>${category.event_count || 0}</td>
                <td>
                    <button class="btn btn-sm btn-primary me-1" onclick="editCategory(${category.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteCategory(${category.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.toString().replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    async loadVendors(page = 1, search = '') {
        console.log('🏪 Loading vendors...');
        
        try {
            let url = `${this.apiBase}/vendors?page=${page}`;
            if (search) {
                url += `&search=${encodeURIComponent(search)}`;
            }
            
            console.log('📡 Making API call to:', url);

            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📥 Vendors response status:', response.status);

            const data = await response.json();
            console.log('📄 Vendors API Response:', data);
            
            if (data.success) {
                console.log('✅ Vendors loaded successfully:', data.data);
                this.renderVendors(data.data || []);
            } else {
                console.error('❌ Vendors API returned error:', data.error);
                throw new Error(data.error || 'Failed to load vendors');
            }
        } catch (error) {
            console.error('💥 Error loading vendors:', error);
            const tbody = document.getElementById('vendorsTableBody');
            if (tbody) {
                tbody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            }
            this.showNotification(`Error loading vendors: ${error.message}`, 'error');
        }
    }

    renderVendors(vendors) {
        const tbody = document.getElementById('vendorsTableBody');
        if (!tbody) return;

        if (!vendors || vendors.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">No vendors found</td></tr>';
            return;
        }

        tbody.innerHTML = vendors.map(vendor => `
            <tr>
                <td>${vendor.id}</td>
                <td>${this.escapeHtml(vendor.business_name)}</td>
                <td>${this.escapeHtml(vendor.contact_email)}</td>
                <td>${this.escapeHtml(vendor.contact_phone)}</td>
                <td>${this.escapeHtml(vendor.address)}</td>
                <td><span class="status-badge status-${vendor.status}">${vendor.status}</span></td>
                <td>
                    <button class="btn btn-sm btn-primary me-1" onclick="viewVendor(${vendor.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-success me-1" onclick="approveVendor(${vendor.id})">
                        <i class="fas fa-check"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="suspendVendor(${vendor.id})">
                        <i class="fas fa-ban"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    async loadGovernmentAgencies(page = 1, search = '') {
        console.log('🏛️ Loading government agencies...');
        
        try {
            let url = `${this.apiBase}/government-agencies?page=${page}`;
            if (search) {
                url += `&search=${encodeURIComponent(search)}`;
            }
            
            console.log('📡 Making API call to:', url);

            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📥 Government agencies response status:', response.status);

            const data = await response.json();
            console.log('📄 Government agencies API Response:', data);
            
            if (data.success) {
                console.log('✅ Government agencies loaded successfully:', data.data);
                // Store the data for use in view/edit functions
                if (!this.currentData) this.currentData = {};
                this.currentData.agencies = data.data || [];
                this.renderGovernmentAgencies(data.data || []);
            } else {
                console.error('❌ Government agencies API returned error:', data.error);
                throw new Error(data.error || 'Failed to load government agencies');
            }
        } catch (error) {
            console.error('💥 Error loading government agencies:', error);
            const tbody = document.getElementById('governmentAgenciesTableBody');
            if (tbody) {
                tbody.innerHTML = `<tr><td colspan="6" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            }
            this.showNotification(`Error loading government agencies: ${error.message}`, 'error');
        }
    }

    renderGovernmentAgencies(agencies) {
        const tbody = document.getElementById('governmentAgenciesTableBody');
        if (!tbody) return;

        if (!agencies || agencies.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center">No government agencies found</td></tr>';
            return;
        }

        tbody.innerHTML = agencies.map(agency => `
            <tr>
                <td>${agency.id}</td>
                <td>${this.escapeHtml(agency.name)}</td>
                <td>${this.escapeHtml(agency.contact_email)}</td>
                <td>${this.escapeHtml(agency.contact_phone)}</td>
                <td>${this.escapeHtml(agency.address)}</td>
                <td><span class="status-badge status-active">Active</span></td>
                <td>
                    <button class="btn btn-sm btn-primary me-1" onclick="adminManager.viewGovernmentAgency(${agency.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-warning me-1" onclick="adminManager.editGovernmentAgency(${agency.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="adminManager.deleteGovernmentAgency(${agency.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    async viewGovernmentAgency(id) {
        console.log('👁️ Viewing government agency:', id);
        
        try {
            // First try to find in cached data
            const agencies = this.currentData?.agencies || [];
            let agency = agencies.find(a => a.id == id);
            
            // If not found in cache, fetch from API
            if (!agency) {
                const response = await fetch(`${this.apiBase}/government-agencies/${id}`, {
                    headers: { 'Authorization': `Bearer ${this.token}` }
                });
                
                const data = await response.json();
                if (data.success) {
                    agency = data.data;
                } else {
                    throw new Error(data.error || 'Agency not found');
                }
            }

            // Create and show view modal
            this.showGovernmentAgencyModal(agency, 'view');
            
        } catch (error) {
            console.error('❌ Error viewing government agency:', error);
            this.showNotification(`Error loading agency: ${error.message}`, 'error');
        }
    }

    async editGovernmentAgency(id) {
        console.log('✏️ Editing government agency:', id);
        
        try {
            // First try to find in cached data
            const agencies = this.currentData?.agencies || [];
            let agency = agencies.find(a => a.id == id);
            
            // If not found in cache, fetch from API
            if (!agency) {
                const response = await fetch(`${this.apiBase}/government-agencies/${id}`, {
                    headers: { 'Authorization': `Bearer ${this.token}` }
                });
                
                const data = await response.json();
                if (data.success) {
                    agency = data.data;
                } else {
                    throw new Error(data.error || 'Agency not found');
                }
            }

            // Create and show edit modal
            this.showGovernmentAgencyModal(agency, 'edit');
            
        } catch (error) {
            console.error('❌ Error editing government agency:', error);
            this.showNotification(`Error loading agency: ${error.message}`, 'error');
        }
    }

    async deleteGovernmentAgency(id) {
        console.log('🗑️ Deleting government agency:', id);
        
        if (!confirm('Are you sure you want to delete this government agency? This action cannot be undone.')) {
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/government-agencies/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            const data = await response.json();
            
            if (data.success) {
                this.showNotification('Government agency deleted successfully', 'success');
                await this.loadGovernmentAgencies(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to delete government agency');
            }
        } catch (error) {
            console.error('Error deleting government agency:', error);
            this.showNotification(`Error deleting government agency: ${error.message}`, 'error');
        }
    }

    showGovernmentAgencyModal(agency = null, mode = 'add') {
        const isEdit = mode === 'edit';
        const isView = mode === 'view';
        const title = isView ? 'View Government Agency' : (isEdit ? 'Edit Government Agency' : 'Add New Government Agency');
        
        // Create modal overlay
        const modalOverlay = document.createElement('div');
        modalOverlay.className = 'modal-overlay';
        modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        `;
        
        modalOverlay.innerHTML = `
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 600px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    color: white;
                    padding: 20px;
                    border-radius: 8px 8px 0 0;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; font-size: 18px;">${title}</h5>
                    <button type="button" class="close-btn" style="
                        background: rgba(255,255,255,0.2);
                        border: none;
                        color: white;
                        font-size: 20px;
                        cursor: pointer;
                        padding: 5px 10px;
                        border-radius: 4px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='rgba(255,255,255,0.3)'" onmouseout="this.style.background='rgba(255,255,255,0.2)'">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addGovernmentAgencyForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="agencyName" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Agency Name *</label>
                                <input type="text" id="agencyName" name="name" required placeholder="Enter agency name" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${agency ? this.escapeHtml(agency.name) : ''}" ${isView ? 'readonly' : ''}>
                            </div>
                            <div>
                                <label for="agencyType" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Agency Type *</label>
                                <select id="agencyType" name="type" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select agency type</option>
                                    <option value="ministry" ${agency?.type === 'ministry' ? 'selected' : ''}>Ministry</option>
                                    <option value="police" ${agency?.type === 'police' ? 'selected' : ''}>Police</option>
                                    <option value="security" ${agency?.type === 'security' ? 'selected' : ''}>Security</option>
                                    <option value="health" ${agency?.type === 'health' ? 'selected' : ''}>Health Department</option>
                                    <option value="fire_service" ${agency?.type === 'fire_service' ? 'selected' : ''}>Fire Service</option>
                                    <option value="customs" ${agency?.type === 'customs' ? 'selected' : ''}>Customs</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="agencyEmail" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Contact Email *</label>
                                <input type="email" id="agencyEmail" name="email" required placeholder="Enter email address" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${agency ? this.escapeHtml(agency.contact_email) : ''}" ${isView ? 'readonly' : ''}>
                            </div>
                            <div>
                                <label for="agencyPhone" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Contact Phone *</label>
                                <input type="tel" id="agencyPhone" name="phone" required placeholder="Enter phone number" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${agency ? this.escapeHtml(agency.contact_phone) : ''}" ${isView ? 'readonly' : ''}>
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="agencyAddress" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Address *</label>
                            <textarea id="agencyAddress" name="address" required rows="3" placeholder="Enter agency address" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                                box-sizing: border-box;
                            " ${isView ? 'readonly' : ''}>${agency ? this.escapeHtml(agency.address) : ''}</textarea>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="agencyLevel" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Level *</label>
                            <select id="agencyLevel" name="level" required style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                box-sizing: border-box;
                            " ${isView ? 'disabled' : ''}>
                                <option value="">Select level</option>
                                <option value="national" ${agency?.level === 'national' ? 'selected' : ''}>National</option>
                                <option value="regional" ${agency?.level === 'regional' ? 'selected' : ''}>Regional</option>
                                <option value="district" ${agency?.level === 'district' ? 'selected' : ''}>District</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                    background: #f8f9fa;
                    border-radius: 0 0 8px 8px;
                ">
                    ${!isView ? `
                    <button type="button" style="
                        background: #28a745;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#218838'" onmouseout="this.style.background='#28a745'" onclick="adminManager.${isEdit ? 'updateGovernmentAgency' : 'createGovernmentAgency'}(${agency ? agency.id : 'null'})">
                        ${isEdit ? '✓ Update Agency' : '+ Create Agency'}
                    </button>
                    ` : ''}
                    <button type="button" style="
                        background: #6c757d;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#5a6268'" onmouseout="this.style.background='#6c757d'" onclick="this.closest('.modal-overlay').remove()">
                        ${isView ? 'Close' : 'Cancel'}
                    </button>
                </div>
            </div>
        `;
        
        // Add click outside to close
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) {
                modalOverlay.remove();
            }
        });
        
        // Add close button functionality
        modalOverlay.querySelector('.close-btn').addEventListener('click', () => {
            modalOverlay.remove();
        });
        
        document.body.appendChild(modalOverlay);
    }

    async createGovernmentAgency() {
        const form = document.getElementById('addGovernmentAgencyForm');
        const formData = new FormData(form);
        
        const agencyData = {
            name: formData.get('name'),
            type: formData.get('type'),
            level: formData.get('level'),
            contact_email: formData.get('email'),
            contact_phone: formData.get('phone'),
            address: formData.get('address')
        };
        
        // Validation
        if (!agencyData.name?.trim() || !agencyData.type?.trim() || !agencyData.level?.trim() || !agencyData.contact_email?.trim() || !agencyData.contact_phone?.trim() || !agencyData.address?.trim()) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Creating government agency:', agencyData);
            
            const response = await fetch(`${this.apiBase}/government-agencies`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(agencyData)
            });
            
            const data = await response.json();
            console.log('📥 Create agency response:', data);
            
            if (data.success) {
                this.showNotification('Government agency created successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadGovernmentAgencies(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to create government agency');
            }
        } catch (error) {
            console.error('❌ Error creating government agency:', error);
            this.showNotification(`Error creating government agency: ${error.message}`, 'error');
        }
    }

    async updateGovernmentAgency(id) {
        const form = document.getElementById('addGovernmentAgencyForm');
        const formData = new FormData(form);
        
        const agencyData = {
            name: formData.get('name'),
            type: formData.get('type'),
            level: formData.get('level'),
            contact_email: formData.get('email'),
            contact_phone: formData.get('phone'),
            address: formData.get('address')
        };
        
        // Validation
        if (!agencyData.name || !agencyData.type || !agencyData.level || !agencyData.contact_email || !agencyData.contact_phone || !agencyData.address) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Updating government agency:', id, agencyData);
            
            const response = await fetch(`${this.apiBase}/government-agencies/${id}`, {
                method: 'PUT',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(agencyData)
            });
            
            const data = await response.json();
            console.log('📥 Update agency response:', data);
            
            if (data.success) {
                this.showNotification('Government agency updated successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadGovernmentAgencies(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to update government agency');
            }
        } catch (error) {
            console.error('❌ Error updating government agency:', error);
            this.showNotification(`Error updating government agency: ${error.message}`, 'error');
        }
    }

    async loadSafetyReports(page = 1, search = '') {
        console.log('🛡️ Loading safety reports...');
        
        try {
            let url = `${this.apiBase}/safety-reports?page=${page}`;
            if (search) {
                url += `&search=${encodeURIComponent(search)}`;
            }
            
            console.log('📡 Making API call to:', url);

            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📥 Safety reports response status:', response.status);

            const data = await response.json();
            console.log('📄 Safety reports API Response:', data);
            
            if (data.success) {
                console.log('✅ Safety reports loaded successfully:', data.data);
                // Store the data for use in view/edit functions
                if (!this.currentData) this.currentData = {};
                this.currentData.reports = data.data || [];
                this.renderSafetyReports(data.data || []);
            } else {
                console.error('❌ Safety reports API returned error:', data.error);
                throw new Error(data.error || 'Failed to load safety reports');
            }
        } catch (error) {
            console.error('💥 Error loading safety reports:', error);
            const tbody = document.getElementById('safetyReportsTableBody');
            if (tbody) {
                tbody.innerHTML = `<tr><td colspan="8" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            }
            this.showNotification(`Error loading safety reports: ${error.message}`, 'error');
        }
    }

    renderSafetyReports(reports) {
        const tbody = document.getElementById('safetyReportsTableBody');
        if (!tbody) return;

        if (!reports || reports.length === 0) {
            tbody.innerHTML = '<tr><td colspan="8" class="text-center">No safety reports found</td></tr>';
            return;
        }

        tbody.innerHTML = reports.map(report => `
            <tr>
                <td>${report.id}</td>
                <td>${this.escapeHtml(report.event_title || 'N/A')}</td>
                <td>${this.escapeHtml(report.report_type || 'N/A')}</td>
                <td><span class="status-badge status-${report.severity}">${report.severity}</span></td>
                <td>${this.escapeHtml(report.location_description || 'N/A')}</td>
                <td><span class="status-badge status-${report.status}">${report.status}</span></td>
                <td>${report.created_at ? new Date(report.created_at).toLocaleDateString() : 'N/A'}</td>
                <td>
                    <button class="btn btn-sm btn-primary me-1" onclick="adminManager.viewSafetyReport(${report.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-warning me-1" onclick="adminManager.editSafetyReport(${report.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="adminManager.deleteSafetyReport(${report.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    async viewSafetyReport(id) {
        console.log('👁️ Viewing safety report:', id);
        
        try {
            // First try to find in cached data
            const reports = this.currentData?.reports || [];
            let report = reports.find(r => r.id == id);
            
            // If not found in cache, fetch from API
            if (!report) {
                const response = await fetch(`${this.apiBase}/safety-reports/${id}`, {
                    headers: { 'Authorization': `Bearer ${this.token}` }
                });
                
                const data = await response.json();
                if (data.success) {
                    report = data.data;
                } else {
                    throw new Error(data.error || 'Safety report not found');
                }
            }

            // Create and show view modal
            this.showSafetyReportModal(report, 'view');
            
        } catch (error) {
            console.error('❌ Error viewing safety report:', error);
            this.showNotification(`Error loading safety report: ${error.message}`, 'error');
        }
    }

    async editSafetyReport(id) {
        console.log('✏️ Editing safety report:', id);
        
        try {
            // First try to find in cached data
            const reports = this.currentData?.reports || [];
            let report = reports.find(r => r.id == id);
            
            // If not found in cache, fetch from API
            if (!report) {
                const response = await fetch(`${this.apiBase}/safety-reports/${id}`, {
                    headers: { 'Authorization': `Bearer ${this.token}` }
                });
                
                const data = await response.json();
                if (data.success) {
                    report = data.data;
                } else {
                    throw new Error(data.error || 'Safety report not found');
                }
            }

            // Create and show edit modal
            this.showSafetyReportModal(report, 'edit');
            
        } catch (error) {
            console.error('❌ Error editing safety report:', error);
            this.showNotification(`Error loading safety report: ${error.message}`, 'error');
        }
    }

    async deleteSafetyReport(id) {
        console.log('🗑️ Deleting safety report:', id);
        
        if (!confirm('Are you sure you want to delete this safety report? This action cannot be undone.')) {
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/safety-reports/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            const data = await response.json();
            
            if (data.success) {
                this.showNotification('Safety report deleted successfully', 'success');
                await this.loadSafetyReports(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to delete safety report');
            }
        } catch (error) {
            console.error('Error deleting safety report:', error);
            this.showNotification(`Error deleting safety report: ${error.message}`, 'error');
        }
    }

    showSafetyReportModal(report = null, mode = 'add') {
        console.log('🚨 Opening Safety Report Modal:', mode, report);
        const isEdit = mode === 'edit';
        const isView = mode === 'view';
        const title = isView ? 'View Safety Report' : (isEdit ? 'Edit Safety Report' : 'Add New Safety Report');
        
        // Create modal overlay
        const modalOverlay = document.createElement('div');
        modalOverlay.className = 'modal-overlay';
        modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        `;
        
        modalOverlay.innerHTML = `
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 700px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
                    color: white;
                    padding: 20px;
                    border-radius: 8px 8px 0 0;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; font-size: 18px;">${title}</h5>
                    <button type="button" class="close-btn" style="
                        background: rgba(255,255,255,0.2);
                        border: none;
                        color: white;
                        font-size: 20px;
                        cursor: pointer;
                        padding: 5px 10px;
                        border-radius: 4px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='rgba(255,255,255,0.3)'" onmouseout="this.style.background='rgba(255,255,255,0.2)'">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addSafetyReportForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="reportEventId" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Event *</label>
                                <select id="reportEventId" name="event_id" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select event</option>
                                    <option value="1" ${report?.event_id == 1 ? 'selected' : ''}>Ghana Music Festival 2024</option>
                                    <option value="2" ${report?.event_id == 2 ? 'selected' : ''}>Homowo Festival Celebration</option>
                                    <option value="3" ${report?.event_id == 3 ? 'selected' : ''}>Tech Innovation Summit</option>
                                    <option value="4" ${report?.event_id == 4 ? 'selected' : ''}>Accra Food Festival</option>
                                    <option value="5" ${report?.event_id == 5 ? 'selected' : ''}>Comedy Night Live</option>
                                </select>
                            </div>
                            <div>
                                <label for="reportType" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Report Type *</label>
                                <select id="reportType" name="report_type" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select type</option>
                                    <option value="emergency" ${report?.report_type === 'emergency' ? 'selected' : ''}>Emergency</option>
                                    <option value="security_concern" ${report?.report_type === 'security_concern' ? 'selected' : ''}>Security Concern</option>
                                    <option value="medical" ${report?.report_type === 'medical' ? 'selected' : ''}>Medical</option>
                                    <option value="harassment" ${report?.report_type === 'harassment' ? 'selected' : ''}>Harassment</option>
                                    <option value="disturbance" ${report?.report_type === 'disturbance' ? 'selected' : ''}>Disturbance</option>
                                    <option value="facility_issue" ${report?.report_type === 'facility_issue' ? 'selected' : ''}>Facility Issue</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="reportSeverity" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Severity *</label>
                                <select id="reportSeverity" name="severity" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select severity</option>
                                    <option value="low" ${report?.severity === 'low' ? 'selected' : ''}>Low</option>
                                    <option value="medium" ${report?.severity === 'medium' ? 'selected' : ''}>Medium</option>
                                    <option value="high" ${report?.severity === 'high' ? 'selected' : ''}>High</option>
                                    <option value="critical" ${report?.severity === 'critical' ? 'selected' : ''}>Critical</option>
                                </select>
                            </div>
                            <div>
                                <label for="reportStatus" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Status *</label>
                                <select id="reportStatus" name="status" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select status</option>
                                    <option value="reported" ${report?.status === 'reported' ? 'selected' : ''}>Reported</option>
                                    <option value="acknowledged" ${report?.status === 'acknowledged' ? 'selected' : ''}>Acknowledged</option>
                                    <option value="in_progress" ${report?.status === 'in_progress' ? 'selected' : ''}>In Progress</option>
                                    <option value="resolved" ${report?.status === 'resolved' ? 'selected' : ''}>Resolved</option>
                                    <option value="false_alarm" ${report?.status === 'false_alarm' ? 'selected' : ''}>False Alarm</option>
                                </select>
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="reportLocation" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Location Description *</label>
                            <input type="text" id="reportLocation" name="location_description" required placeholder="Enter location description" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                box-sizing: border-box;
                            " value="${report ? this.escapeHtml(report.location_description) : ''}" ${isView ? 'readonly' : ''}>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="reportDescription" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Description *</label>
                            <textarea id="reportDescription" name="description" required rows="4" placeholder="Enter detailed description of the incident" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                                box-sizing: border-box;
                            " ${isView ? 'readonly' : ''}>${report ? this.escapeHtml(report.description) : ''}</textarea>
                        </div>
                        ${!isView ? `
                        <div style="margin-bottom: 15px;">
                            <label for="reportActions" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Actions Taken</label>
                            <textarea id="reportActions" name="resolution_notes" rows="3" placeholder="Enter actions taken or resolution notes" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                                box-sizing: border-box;
                            ">${report ? this.escapeHtml(report.resolution_notes || '') : ''}</textarea>
                        </div>
                        ` : ''}
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                    background: #f8f9fa;
                    border-radius: 0 0 8px 8px;
                ">
                    ${!isView ? `
                    <button type="button" style="
                        background: #dc3545;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#c82333'" onmouseout="this.style.background='#dc3545'" onclick="adminManager.${isEdit ? 'updateSafetyReport' : 'createSafetyReport'}(${report ? report.id : 'null'})">
                        ${isEdit ? '✓ Update Report' : '+ Create Report'}
                    </button>
                    ` : ''}
                    <button type="button" style="
                        background: #6c757d;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#5a6268'" onmouseout="this.style.background='#6c757d'" onclick="this.closest('.modal-overlay').remove()">
                        ${isView ? 'Close' : 'Cancel'}
                    </button>
                </div>
            </div>
        `;
        
        // Add click outside to close
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) {
                modalOverlay.remove();
            }
        });
        
        // Add close button functionality
        modalOverlay.querySelector('.close-btn').addEventListener('click', () => {
            modalOverlay.remove();
        });
        
        document.body.appendChild(modalOverlay);
    }

    async createSafetyReport() {
        const form = document.getElementById('addSafetyReportForm');
        const formData = new FormData(form);
        
        const reportData = {
            event_id: formData.get('event_id'),
            report_type: formData.get('report_type'),
            severity: formData.get('severity'),
            status: formData.get('status'),
            location_description: formData.get('location_description'),
            description: formData.get('description'),
            resolution_notes: formData.get('resolution_notes')
        };
        
        // Validation
        if (!reportData.event_id || !reportData.report_type || !reportData.severity || !reportData.status || !reportData.location_description || !reportData.description) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Creating safety report:', reportData);
            
            const response = await fetch(`${this.apiBase}/safety-reports`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(reportData)
            });
            
            const data = await response.json();
            console.log('📥 Create report response:', data);
            
            if (data.success) {
                this.showNotification('Safety report created successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadSafetyReports(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to create safety report');
            }
        } catch (error) {
            console.error('❌ Error creating safety report:', error);
            this.showNotification(`Error creating safety report: ${error.message}`, 'error');
        }
    }

    async updateSafetyReport(id) {
        const form = document.getElementById('addSafetyReportForm');
        const formData = new FormData(form);
        
        const reportData = {
            event_id: formData.get('event_id'),
            report_type: formData.get('report_type'),
            severity: formData.get('severity'),
            status: formData.get('status'),
            location_description: formData.get('location_description'),
            description: formData.get('description'),
            resolution_notes: formData.get('resolution_notes')
        };
        
        // Validation
        if (!reportData.event_id || !reportData.report_type || !reportData.severity || !reportData.status || !reportData.location_description || !reportData.description) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Updating safety report:', id, reportData);
            
            const response = await fetch(`${this.apiBase}/safety-reports/${id}`, {
                method: 'PUT',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(reportData)
            });
            
            const data = await response.json();
            console.log('📥 Update report response:', data);
            
            if (data.success) {
                this.showNotification('Safety report updated successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadSafetyReports(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to update safety report');
            }
        } catch (error) {
            console.error('❌ Error updating safety report:', error);
            this.showNotification(`Error updating safety report: ${error.message}`, 'error');
        }
    }

    async loadPaymentTransactions(page = 1, search = '') {
        console.log('💳 Loading payment transactions...');
        
        try {
            let url = `${this.apiBase}/payment-transactions?page=${page}`;
            if (search) {
                url += `&search=${encodeURIComponent(search)}`;
            }
            
            console.log('📡 Making API call to:', url);

            const response = await fetch(url, {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            console.log('📥 Payment transactions response status:', response.status);

            const data = await response.json();
            console.log('📄 Payment transactions API Response:', data);
            
            if (data.success) {
                console.log('✅ Payment transactions loaded successfully:', data.data);
                this.renderPaymentTransactions(data.data || []);
            } else {
                console.error('❌ Payment transactions API returned error:', data.error);
                throw new Error(data.error || 'Failed to load payment transactions');
            }
        } catch (error) {
            console.error('💥 Error loading payment transactions:', error);
            const tbody = document.getElementById('paymentTransactionsTableBody');
            if (tbody) {
                tbody.innerHTML = `<tr><td colspan="8" class="text-center text-danger">Error: ${error.message}</td></tr>`;
            }
            this.showNotification(`Error loading payment transactions: ${error.message}`, 'error');
        }
    }

    renderPaymentTransactions(transactions) {
        const tbody = document.getElementById('paymentTransactionsTableBody');
        if (!tbody) return;

        if (!transactions || transactions.length === 0) {
            tbody.innerHTML = '<tr><td colspan="8" class="text-center">No payment transactions found</td></tr>';
            return;
        }

        tbody.innerHTML = transactions.map(transaction => `
            <tr>
                <td>${transaction.id}</td>
                <td>${this.escapeHtml(transaction.user_name || 'N/A')}</td>
                <td>${this.escapeHtml(transaction.transaction_type || 'N/A')}</td>
                <td>GHS ${transaction.amount || '0.00'}</td>
                <td>${this.escapeHtml(transaction.payment_method || 'N/A')}</td>
                <td><span class="status-badge status-${transaction.status}">${transaction.status}</span></td>
                <td>${transaction.created_at ? new Date(transaction.created_at).toLocaleDateString() : 'N/A'}</td>
                <td>
                    <button class="btn btn-sm btn-primary me-1" onclick="adminManager.viewPaymentTransaction(${transaction.id})">
                        <i class="fas fa-eye"></i>
                    </button>
                    <button class="btn btn-sm btn-warning me-1" onclick="adminManager.editPaymentTransaction(${transaction.id})">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="adminManager.deletePaymentTransaction(${transaction.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            </tr>
        `).join('');
    }

    viewPaymentTransaction(id) {
        console.log('👁️ Viewing payment transaction:', id);
        // Find the transaction data
        const transactions = this.currentData?.transactions || [];
        const transaction = transactions.find(t => t.id == id);
        
        if (!transaction) {
            this.showNotification('Payment transaction not found', 'error');
            return;
        }

        // Create and show view modal
        this.showPaymentTransactionModal(transaction, 'view');
    }

    editPaymentTransaction(id) {
        console.log('✏️ Editing payment transaction:', id);
        // Find the transaction data
        const transactions = this.currentData?.transactions || [];
        const transaction = transactions.find(t => t.id == id);
        
        if (!transaction) {
            this.showNotification('Payment transaction not found', 'error');
            return;
        }

        // Create and show edit modal
        this.showPaymentTransactionModal(transaction, 'edit');
    }

    async deletePaymentTransaction(id) {
        console.log('🗑️ Deleting payment transaction:', id);
        
        if (!confirm('Are you sure you want to delete this payment transaction? This action cannot be undone.')) {
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/payment-transactions/${id}`, {
                method: 'DELETE',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            const data = await response.json();
            
            if (data.success) {
                this.showNotification('Payment transaction deleted successfully', 'success');
                await this.loadPaymentTransactions(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to delete payment transaction');
            }
        } catch (error) {
            console.error('Error deleting payment transaction:', error);
            this.showNotification(`Error deleting payment transaction: ${error.message}`, 'error');
        }
    }

    showPaymentTransactionModal(transaction = null, mode = 'add') {
        console.log('💳 Opening Payment Transaction Modal:', mode, transaction);
        const isEdit = mode === 'edit';
        const isView = mode === 'view';
        const title = isView ? 'View Payment Transaction' : (isEdit ? 'Edit Payment Transaction' : 'Add New Payment Transaction');
        
        // Create modal overlay
        const modalOverlay = document.createElement('div');
        modalOverlay.className = 'modal-overlay';
        modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        `;
        
        modalOverlay.innerHTML = `
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 700px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
                    color: white;
                    padding: 20px;
                    border-radius: 8px 8px 0 0;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; font-size: 18px;">${title}</h5>
                    <button type="button" class="close-btn" style="
                        background: rgba(255,255,255,0.2);
                        border: none;
                        color: white;
                        font-size: 20px;
                        cursor: pointer;
                        padding: 5px 10px;
                        border-radius: 4px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='rgba(255,255,255,0.3)'" onmouseout="this.style.background='rgba(255,255,255,0.2)'">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addPaymentTransactionForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="transactionUserId" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">User *</label>
                                <select id="transactionUserId" name="user_id" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select user</option>
                                    <option value="1" ${transaction?.user_id == 1 ? 'selected' : ''}>System Administrator</option>
                                    <option value="2" ${transaction?.user_id == 2 ? 'selected' : ''}>Event Organizer</option>
                                    <option value="3" ${transaction?.user_id == 3 ? 'selected' : ''}>MARTHA COMMEY</option>
                                    <option value="4" ${transaction?.user_id == 4 ? 'selected' : ''}>Akwasi Bobie</option>
                                    <option value="5" ${transaction?.user_id == 5 ? 'selected' : ''}>Kwame Aseidu</option>
                                </select>
                            </div>
                            <div>
                                <label for="transactionType" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Transaction Type *</label>
                                <select id="transactionType" name="transaction_type" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select type</option>
                                    <option value="ticket_purchase" ${transaction?.transaction_type === 'ticket_purchase' ? 'selected' : ''}>Ticket Purchase</option>
                                    <option value="vendor_payment" ${transaction?.transaction_type === 'vendor_payment' ? 'selected' : ''}>Vendor Payment</option>
                                    <option value="accommodation" ${transaction?.transaction_type === 'accommodation' ? 'selected' : ''}>Accommodation</option>
                                    <option value="permit_fee" ${transaction?.transaction_type === 'permit_fee' ? 'selected' : ''}>Permit Fee</option>
                                    <option value="levy_payment" ${transaction?.transaction_type === 'levy_payment' ? 'selected' : ''}>Levy Payment</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="transactionAmount" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Amount (GHS) *</label>
                                <input type="number" id="transactionAmount" name="amount" required step="0.01" min="0" placeholder="Enter amount" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${transaction ? transaction.amount : ''}" ${isView ? 'readonly' : ''}>
                            </div>
                            <div>
                                <label for="paymentMethod" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Payment Method *</label>
                                <select id="paymentMethod" name="payment_method" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select method</option>
                                    <option value="mobile_money" ${transaction?.payment_method === 'mobile_money' ? 'selected' : ''}>Mobile Money</option>
                                    <option value="visa" ${transaction?.payment_method === 'visa' ? 'selected' : ''}>Visa</option>
                                    <option value="mastercard" ${transaction?.payment_method === 'mastercard' ? 'selected' : ''}>Mastercard</option>
                                    <option value="bank_transfer" ${transaction?.payment_method === 'bank_transfer' ? 'selected' : ''}>Bank Transfer</option>
                                    <option value="cash" ${transaction?.payment_method === 'cash' ? 'selected' : ''}>Cash</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="transactionStatus" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Status *</label>
                                <select id="transactionStatus" name="status" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " ${isView ? 'disabled' : ''}>
                                    <option value="">Select status</option>
                                    <option value="pending" ${transaction?.status === 'pending' ? 'selected' : ''}>Pending</option>
                                    <option value="processing" ${transaction?.status === 'processing' ? 'selected' : ''}>Processing</option>
                                    <option value="completed" ${transaction?.status === 'completed' ? 'selected' : ''}>Completed</option>
                                    <option value="failed" ${transaction?.status === 'failed' ? 'selected' : ''}>Failed</option>
                                    <option value="cancelled" ${transaction?.status === 'cancelled' ? 'selected' : ''}>Cancelled</option>
                                    <option value="refunded" ${transaction?.status === 'refunded' ? 'selected' : ''}>Refunded</option>
                                </select>
                            </div>
                            <div>
                                <label for="paymentProvider" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Payment Provider</label>
                                <input type="text" id="paymentProvider" name="payment_provider" placeholder="Enter payment provider" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${transaction ? this.escapeHtml(transaction.payment_provider || '') : ''}" ${isView ? 'readonly' : ''}>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="transactionFees" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Fees (GHS)</label>
                                <input type="number" id="transactionFees" name="fees" step="0.01" min="0" placeholder="Enter fees" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${transaction ? transaction.fees || '0.00' : '0.00'}" ${isView ? 'readonly' : ''}>
                            </div>
                            <div>
                                <label for="transactionTax" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Tax Amount (GHS)</label>
                                <input type="number" id="transactionTax" name="tax_amount" step="0.01" min="0" placeholder="Enter tax amount" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                    box-sizing: border-box;
                                " value="${transaction ? transaction.tax_amount || '0.00' : '0.00'}" ${isView ? 'readonly' : ''}>
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="providerReference" style="display: block; margin-bottom: 5px; font-weight: bold; color: #333;">Provider Reference</label>
                            <input type="text" id="providerReference" name="provider_reference" placeholder="Enter provider reference" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                box-sizing: border-box;
                            " value="${transaction ? this.escapeHtml(transaction.provider_reference || '') : ''}" ${isView ? 'readonly' : ''}>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                    background: #f8f9fa;
                    border-radius: 0 0 8px 8px;
                ">
                    ${!isView ? `
                    <button type="button" style="
                        background: #28a745;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#218838'" onmouseout="this.style.background='#28a745'" onclick="adminManager.${isEdit ? 'updatePaymentTransaction' : 'createPaymentTransaction'}(${transaction ? transaction.id : 'null'})">
                        ${isEdit ? '✓ Update Transaction' : '+ Create Transaction'}
                    </button>
                    ` : ''}
                    <button type="button" style="
                        background: #6c757d;
                        color: white;
                        border: none;
                        padding: 10px 20px;
                        border-radius: 4px;
                        cursor: pointer;
                        font-size: 14px;
                        transition: background 0.3s;
                    " onmouseover="this.style.background='#5a6268'" onmouseout="this.style.background='#6c757d'" onclick="this.closest('.modal-overlay').remove()">
                        ${isView ? 'Close' : 'Cancel'}
                    </button>
                </div>
            </div>
        `;
        
        // Add click outside to close
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) {
                modalOverlay.remove();
            }
        });
        
        // Add close button functionality
        modalOverlay.querySelector('.close-btn').addEventListener('click', () => {
            modalOverlay.remove();
        });
        
        document.body.appendChild(modalOverlay);
    }

    async createPaymentTransaction() {
        const form = document.getElementById('addPaymentTransactionForm');
        const formData = new FormData(form);
        
        const transactionData = {
            user_id: formData.get('user_id'),
            transaction_type: formData.get('transaction_type'),
            amount: formData.get('amount'),
            payment_method: formData.get('payment_method'),
            status: formData.get('status'),
            payment_provider: formData.get('payment_provider'),
            fees: formData.get('fees') || '0.00',
            tax_amount: formData.get('tax_amount') || '0.00',
            provider_reference: formData.get('provider_reference')
        };
        
        // Calculate net amount
        const amount = parseFloat(transactionData.amount) || 0;
        const fees = parseFloat(transactionData.fees) || 0;
        const tax = parseFloat(transactionData.tax_amount) || 0;
        transactionData.net_amount = (amount - fees - tax).toFixed(2);
        
        // Validation
        if (!transactionData.user_id || !transactionData.transaction_type || !transactionData.amount || !transactionData.payment_method || !transactionData.status) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Creating payment transaction:', transactionData);
            
            const response = await fetch(`${this.apiBase}/payment-transactions`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(transactionData)
            });
            
            const data = await response.json();
            console.log('📥 Create transaction response:', data);
            
            if (data.success) {
                this.showNotification('Payment transaction created successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadPaymentTransactions(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to create payment transaction');
            }
        } catch (error) {
            console.error('❌ Error creating payment transaction:', error);
            this.showNotification(`Error creating payment transaction: ${error.message}`, 'error');
        }
    }

    async updatePaymentTransaction(id) {
        const form = document.getElementById('addPaymentTransactionForm');
        const formData = new FormData(form);
        
        const transactionData = {
            user_id: formData.get('user_id'),
            transaction_type: formData.get('transaction_type'),
            amount: formData.get('amount'),
            payment_method: formData.get('payment_method'),
            status: formData.get('status'),
            payment_provider: formData.get('payment_provider'),
            fees: formData.get('fees') || '0.00',
            tax_amount: formData.get('tax_amount') || '0.00',
            provider_reference: formData.get('provider_reference')
        };
        
        // Calculate net amount
        const amount = parseFloat(transactionData.amount) || 0;
        const fees = parseFloat(transactionData.fees) || 0;
        const tax = parseFloat(transactionData.tax_amount) || 0;
        transactionData.net_amount = (amount - fees - tax).toFixed(2);
        
        // Validation
        if (!transactionData.user_id || !transactionData.transaction_type || !transactionData.amount || !transactionData.payment_method || !transactionData.status) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        try {
            console.log('📤 Updating payment transaction:', id, transactionData);
            
            const response = await fetch(`${this.apiBase}/payment-transactions/${id}`, {
                method: 'PUT',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(transactionData)
            });
            
            const data = await response.json();
            console.log('📥 Update transaction response:', data);
            
            if (data.success) {
                this.showNotification('Payment transaction updated successfully!', 'success');
                document.querySelector('.modal-overlay').remove();
                await this.loadPaymentTransactions(); // Reload the list
            } else {
                throw new Error(data.error || 'Failed to update payment transaction');
            }
        } catch (error) {
            console.error('❌ Error updating payment transaction:', error);
            this.showNotification(`Error updating payment transaction: ${error.message}`, 'error');
        }
    }

    // Event Permits Management
    async loadEventPermits(page = 1, search = '') {
        console.log('📋 Loading event permits...');
        
        try {
            const response = await fetch(`${this.apiBase}/event-permits?page=${page}`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success) {
                this.renderEventPermits(data.data || []);
            } else {
                throw new Error(data.error || 'Failed to load event permits');
            }
        } catch (error) {
            console.error('❌ Error loading event permits:', error);
            this.showNotification(`Error loading event permits: ${error.message}`, 'error');
        }
    }

    renderEventPermits(permits) {
        const tbody = document.getElementById('eventPermitsTableBody');
        if (!tbody) return;

        if (!permits || permits.length === 0) {
            tbody.innerHTML = '<tr><td colspan="8" class="text-center">No event permits found</td></tr>';
            return;
        }

        tbody.innerHTML = permits.map(permit => `
            <tr>
                <td>${permit.id}</td>
                <td>${permit.event_title || 'N/A'}</td>
                <td>${permit.agency_name || 'N/A'}</td>
                <td>${permit.permit_type || 'N/A'}</td>
                <td><span class="status-badge status-${permit.status}">${permit.status}</span></td>
                <td>${permit.issue_date ? new Date(permit.issue_date).toLocaleDateString() : 'N/A'}</td>
                <td>${permit.expiry_date ? new Date(permit.expiry_date).toLocaleDateString() : 'N/A'}</td>
                <td>
                    <button class="btn btn-primary btn-sm" onclick="adminManager.viewEventPermit(${permit.id})">View</button>
                    <button class="btn btn-warning btn-sm" onclick="adminManager.editEventPermit(${permit.id})">Edit</button>
                    <button class="btn btn-danger btn-sm" onclick="adminManager.deleteEventPermit(${permit.id})">Delete</button>
                </td>
            </tr>
        `).join('');
    }

    async viewEventPermit(id) {
        try {
            const response = await fetch(`${this.apiBase}/event-permits/${id}`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success) {
                this.showEventPermitModal(data.data, 'view');
            } else {
                throw new Error(data.error || 'Failed to load event permit');
            }
        } catch (error) {
            console.error('❌ Error loading event permit:', error);
            this.showNotification(`Error loading event permit: ${error.message}`, 'error');
        }
    }

    async editEventPermit(id) {
        try {
            const response = await fetch(`${this.apiBase}/event-permits/${id}`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success) {
                this.showEventPermitModal(data.data, 'edit');
            } else {
                throw new Error(data.error || 'Failed to load event permit');
            }
        } catch (error) {
            console.error('❌ Error loading event permit:', error);
            this.showNotification(`Error loading event permit: ${error.message}`, 'error');
        }
    }

    async deleteEventPermit(id) {
        if (!confirm('Are you sure you want to delete this event permit?')) {
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/event-permits/${id}`, {
                method: 'DELETE',
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success) {
                this.showNotification('Event permit deleted successfully', 'success');
                this.loadEventPermits();
            } else {
                throw new Error(data.error || 'Failed to delete event permit');
            }
        } catch (error) {
            console.error('❌ Error deleting event permit:', error);
            this.showNotification(`Error deleting event permit: ${error.message}`, 'error');
        }
    }

    showEventPermitModal(permit = null, mode = 'add') {
        console.log('📋 Opening Event Permit Modal:', mode, permit);
        const modalOverlay = document.createElement('div');
        modalOverlay.className = 'modal-overlay';
        modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        `;

        const isReadonly = mode === 'view';
        const title = mode === 'add' ? 'Add New Event Permit' : 
                     mode === 'edit' ? 'Edit Event Permit' : 'View Event Permit';

        modalOverlay.innerHTML = `
            <div class="modal-content" style="background: white; padding: 0; border-radius: 8px; width: 90%; max-width: 600px; max-height: 90vh; overflow-y: auto;">
                <div class="modal-header" style="background: #007bff; color: white; padding: 15px 20px; border-radius: 8px 8px 0 0; display: flex; justify-content: space-between; align-items: center;">
                    <h3 style="margin: 0; font-size: 18px;">${title}</h3>
                    <button type="button" class="close-btn" style="background: none; border: none; color: white; font-size: 24px; cursor: pointer; padding: 0; width: 30px; height: 30px; display: flex; align-items: center; justify-content: center;">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="eventPermitForm">
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Event:</label>
                            <select name="event_id" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="">Select Event</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Government Agency:</label>
                            <select name="agency_id" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="">Select Agency</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Permit Type:</label>
                            <select name="permit_type" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="">Select Type</option>
                                <option value="event_license">Event License</option>
                                <option value="noise_permit">Noise Permit</option>
                                <option value="food_permit">Food Permit</option>
                                <option value="alcohol_permit">Alcohol Permit</option>
                                <option value="fire_safety">Fire Safety Permit</option>
                                <option value="health_permit">Health Permit</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Status:</label>
                            <select name="status" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="pending">Pending</option>
                                <option value="approved">Approved</option>
                                <option value="rejected">Rejected</option>
                                <option value="expired">Expired</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Issue Date:</label>
                            <input type="date" name="issue_date" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'readonly' : ''}>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Expiry Date:</label>
                            <input type="date" name="expiry_date" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'readonly' : ''}>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Fee Amount:</label>
                            <input type="number" name="fee_amount" step="0.01" min="0" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'readonly' : ''}>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Notes:</label>
                            <textarea name="notes" rows="3" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; resize: vertical;" ${isReadonly ? 'readonly' : ''}></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="padding: 15px 20px; border-top: 1px solid #eee; display: flex; justify-content: flex-end; gap: 10px;">
                    <button type="button" class="btn-cancel" style="padding: 8px 16px; border: 1px solid #ddd; background: white; border-radius: 4px; cursor: pointer;">Cancel</button>
                    ${!isReadonly ? `<button type="button" class="btn-save" style="padding: 8px 16px; border: none; background: #007bff; color: white; border-radius: 4px; cursor: pointer;">${mode === 'add' ? 'Create' : 'Update'}</button>` : ''}
                </div>
            </div>
        `;

        // Close modal functionality
        const closeModal = () => modalOverlay.remove();
        modalOverlay.querySelector('.close-btn').addEventListener('click', closeModal);
        modalOverlay.querySelector('.btn-cancel').addEventListener('click', closeModal);
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) closeModal();
        });

        // Load events and agencies for dropdowns
        this.loadEventsForSelect(modalOverlay.querySelector('select[name="event_id"]'));
        this.loadAgenciesForSelect(modalOverlay.querySelector('select[name="agency_id"]'));

        // Populate form if editing or viewing
        if (permit && (mode === 'edit' || mode === 'view')) {
            const form = modalOverlay.querySelector('#eventPermitForm');
            Object.keys(permit).forEach(key => {
                const field = form.querySelector(`[name="${key}"]`);
                if (field) {
                    if (field.type === 'date' && permit[key]) {
                        field.value = permit[key].split(' ')[0]; // Extract date part
                    } else {
                        field.value = permit[key] || '';
                    }
                }
            });
        }

        // Save functionality
        if (!isReadonly) {
            modalOverlay.querySelector('.btn-save').addEventListener('click', () => {
                this.saveEventPermit(modalOverlay, mode, permit?.id);
            });
        }

        document.body.appendChild(modalOverlay);
    }

    async saveEventPermit(modalOverlay, mode, permitId = null) {
        const form = modalOverlay.querySelector('#eventPermitForm');
        const formData = new FormData(form);
        
        // Validation
        const requiredFields = ['event_id', 'agency_id', 'permit_type', 'status'];
        for (const field of requiredFields) {
            if (!formData.get(field)) {
                this.showNotification(`Please fill in the ${field.replace('_', ' ')} field`, 'error');
                return;
            }
        }

        const permitData = {};
        for (const [key, value] of formData.entries()) {
            permitData[key] = value;
        }

        try {
            const url = mode === 'add' ? `${this.apiBase}/event-permits` : `${this.apiBase}/event-permits/${permitId}`;
            const method = mode === 'add' ? 'POST' : 'PUT';
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(permitData)
            });
            
            const data = await response.json();
            if (data.success) {
                this.showNotification(`Event permit ${mode === 'add' ? 'created' : 'updated'} successfully`, 'success');
                modalOverlay.remove();
                this.loadEventPermits();
            } else {
                throw new Error(data.error || `Failed to ${mode} event permit`);
            }
        } catch (error) {
            console.error(`❌ Error ${mode === 'add' ? 'creating' : 'updating'} event permit:`, error);
            this.showNotification(`Error ${mode === 'add' ? 'creating' : 'updating'} event permit: ${error.message}`, 'error');
        }
    }

    async loadEventsForSelect(selectElement) {
        try {
            const response = await fetch(`${this.apiBase}/events`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success && data.data) {
                data.data.forEach(event => {
                    const option = document.createElement('option');
                    option.value = event.id;
                    option.textContent = event.title;
                    selectElement.appendChild(option);
                });
            }
        } catch (error) {
            console.error('Error loading events for select:', error);
        }
    }

    async loadAgenciesForSelect(selectElement) {
        try {
            const response = await fetch(`${this.apiBase}/government-agencies`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success && data.data) {
                data.data.forEach(agency => {
                    const option = document.createElement('option');
                    option.value = agency.id;
                    option.textContent = agency.name;
                    selectElement.appendChild(option);
                });
            }
        } catch (error) {
            console.error('Error loading agencies for select:', error);
        }
    }

    async loadRoles() {
        console.log('👤 Loading user roles...');
        
        try {
            const response = await fetch(`${this.apiBase}/roles`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayRoles(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load roles');
                }
            } else {
                throw new Error('Failed to fetch roles');
            }
        } catch (error) {
            console.error('❌ Error loading roles:', error);
            this.showNotification('Error loading roles: ' + error.message, 'error');
        }
    }

    displayRoles(roles) {
        const tbody = document.getElementById('rolesTableBody');
        if (!tbody) return;

        if (!roles || roles.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center">No roles found</td></tr>';
            return;
        }

        tbody.innerHTML = roles.map(role => `
            <tr>
                <td>${role.user_name || 'N/A'}</td>
                <td>${role.role_type || 'N/A'}</td>
                <td>${role.role_level || 'N/A'}</td>
                <td>${role.department || 'N/A'}</td>
                <td>${role.permissions ? JSON.stringify(role.permissions) : 'N/A'}</td>
                <td>${role.assigned_by_name || 'N/A'}</td>
                <td>
                    <button class="btn btn-primary btn-sm" onclick="adminManager.viewUserRole(${role.id})">View</button>
                    <button class="btn btn-warning btn-sm" onclick="adminManager.editUserRole(${role.id})">Edit</button>
                    <button class="btn btn-danger btn-sm" onclick="adminManager.deleteUserRole(${role.id})">Delete</button>
                </td>
            </tr>
        `).join('');
    }

    async loadSystemLogs() {
        console.log('📊 Loading system logs...');
        
        try {
            const response = await fetch(`${this.apiBase}/logs`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displaySystemLogs(result.data || []);
                } else {
                    throw new Error(result.error || 'Failed to load system logs');
                }
            } else {
                throw new Error('Failed to fetch system logs');
            }
        } catch (error) {
            console.error('❌ Error loading system logs:', error);
            this.showNotification('Error loading system logs: ' + error.message, 'error');
        }
    }

    displaySystemLogs(logs) {
        const container = document.getElementById('systemLogsContainer');
        if (!container) return;

        if (!logs || logs.length === 0) {
            container.innerHTML = '<div style="color: #888; text-align: center; padding: 2rem;">No logs available</div>';
            return;
        }

        container.innerHTML = logs.map(log => `
            <div style="margin-bottom: 10px; padding: 10px; background: #2d3748; border-left: 3px solid #4299e1; border-radius: 4px;">
                <div style="color: #e2e8f0; font-size: 0.9rem;">
                    <span style="color: #4299e1;">[${log.timestamp}]</span>
                    <span style="color: #f6e05e;">[${log.level}]</span>
                    ${log.message}
                </div>
            </div>
        `).join('');
    }

    async loadAnalytics() {
        console.log('📈 Loading analytics...');
        
        try {
            const response = await fetch(`${this.apiBase}/analytics`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.displayAnalytics(result.data || {});
                } else {
                    throw new Error(result.error || 'Failed to load analytics');
                }
            } else {
                throw new Error('Failed to fetch analytics');
            }
        } catch (error) {
            console.error('❌ Error loading analytics:', error);
            this.showNotification('Error loading analytics: ' + error.message, 'error');
        }
    }

    displayAnalytics(data) {
        const container = document.getElementById('analyticsContainer');
        if (!container) return;

        container.innerHTML = `
            <div class="analytics-grid">
                <div class="analytics-card">
                    <h4>Total Page Views</h4>
                    <div class="analytics-number">${data.page_views || 0}</div>
                </div>
                <div class="analytics-card">
                    <h4>Active Users</h4>
                    <div class="analytics-number">${data.active_users || 0}</div>
                </div>
                <div class="analytics-card">
                    <h4>Event Registrations</h4>
                    <div class="analytics-number">${data.registrations || 0}</div>
                </div>
                <div class="analytics-card">
                    <h4>Revenue</h4>
                    <div class="analytics-number">$${data.revenue || 0}</div>
                </div>
            </div>
        `;
    }

    showNotification(message, type = 'info') {
        // Remove any existing notifications
        const existingNotifications = document.querySelectorAll('.notification');
        existingNotifications.forEach(notification => notification.remove());

        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;

        document.body.appendChild(notification);

        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);

        console.log(`📢 Notification (${type}): ${message}`);
    }

    // User Role Modal Methods
    showUserRoleModal(role = null, mode = 'add') {
        console.log(`🔧 Opening User Role modal in ${mode} mode:`, role);
        
        const isReadonly = mode === 'view';
        const modalOverlay = document.createElement('div');
        modalOverlay.className = 'modal-overlay';
        modalOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 1000;
        `;

        modalOverlay.innerHTML = `
            <div class="modal-content" style="background: white; padding: 0; border-radius: 10px; width: 90%; max-width: 600px; max-height: 90vh; overflow-y: auto; box-shadow: 0 4px 20px rgba(0,0,0,0.15);">
                <div class="modal-header" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 10px 10px 0 0; position: relative;">
                    <h2 style="margin: 0; font-size: 1.5rem;">${mode === 'add' ? 'Add New User Role' : mode === 'edit' ? 'Edit User Role' : 'View User Role'}</h2>
                    <button class="close-modal" style="position: absolute; top: 15px; right: 20px; background: none; border: none; color: white; font-size: 24px; cursor: pointer; padding: 0; width: 30px; height: 30px; display: flex; align-items: center; justify-content: center;">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="userRoleForm">
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">User:</label>
                            <select name="user_id" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="">Select User</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Role Type:</label>
                            <select name="role_type" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="">Select Role Type</option>
                                <option value="admin">Admin</option>
                                <option value="government_official">Government Official</option>
                                <option value="event_organizer">Event Organizer</option>
                                <option value="venue_owner">Venue Owner</option>
                                <option value="tourist">Tourist</option>
                                <option value="vendor">Vendor</option>
                                <option value="emergency_responder">Emergency Responder</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Role Level:</label>
                            <select name="role_level" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'disabled' : ''}>
                                <option value="local">Local</option>
                                <option value="district">District</option>
                                <option value="regional">Regional</option>
                                <option value="national">National</option>
                            </select>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Department:</label>
                            <input type="text" name="department" placeholder="e.g., Ministry of Tourism, Police Service" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;" ${isReadonly ? 'readonly' : ''}>
                        </div>
                        <div class="form-group" style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: bold;">Permissions (JSON):</label>
                            <textarea name="permissions" rows="4" placeholder='{"read": true, "write": false, "delete": false}' style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; resize: vertical; font-family: monospace;" ${isReadonly ? 'readonly' : ''}></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="padding: 15px 20px; border-top: 1px solid #eee; display: flex; justify-content: flex-end; gap: 10px;">
                    <button type="button" class="btn-cancel" style="padding: 8px 16px; border: 1px solid #ddd; background: white; border-radius: 4px; cursor: pointer;">Cancel</button>
                    ${!isReadonly ? `<button type="button" class="btn-save" style="padding: 8px 16px; border: none; background: #007bff; color: white; border-radius: 4px; cursor: pointer;">${mode === 'add' ? 'Create' : 'Update'}</button>` : ''}
                </div>
            </div>
        `;

        // Close modal functionality
        modalOverlay.querySelector('.close-modal').addEventListener('click', () => modalOverlay.remove());
        modalOverlay.querySelector('.btn-cancel').addEventListener('click', () => modalOverlay.remove());
        modalOverlay.addEventListener('click', (e) => {
            if (e.target === modalOverlay) modalOverlay.remove();
        });

        // Load users for dropdown
        this.loadUsersForSelect(modalOverlay.querySelector('select[name="user_id"]'));

        // Populate form if editing or viewing
        if (role && (mode === 'edit' || mode === 'view')) {
            const form = modalOverlay.querySelector('#userRoleForm');
            Object.keys(role).forEach(key => {
                const field = form.querySelector(`[name="${key}"]`);
                if (field) {
                    if (key === 'permissions' && role[key]) {
                        field.value = typeof role[key] === 'string' ? role[key] : JSON.stringify(role[key], null, 2);
                    } else {
                        field.value = role[key] || '';
                    }
                }
            });
        }

        // Save functionality
        if (!isReadonly) {
            modalOverlay.querySelector('.btn-save').addEventListener('click', () => {
                this.saveUserRole(modalOverlay, mode, role?.id);
            });
        }

        document.body.appendChild(modalOverlay);
    }

    async saveUserRole(modalOverlay, mode, roleId = null) {
        const form = modalOverlay.querySelector('#userRoleForm');
        const formData = new FormData(form);
        
        // Validation
        const requiredFields = ['user_id', 'role_type'];
        for (const field of requiredFields) {
            if (!formData.get(field)) {
                this.showNotification(`Please fill in the ${field.replace('_', ' ')} field`, 'error');
                return;
            }
        }

        const roleData = {};
        for (const [key, value] of formData.entries()) {
            if (key === 'permissions' && value) {
                try {
                    roleData[key] = JSON.parse(value);
                } catch (e) {
                    this.showNotification('Invalid JSON format for permissions', 'error');
                    return;
                }
            } else {
                roleData[key] = value;
            }
        }

        try {
            const url = mode === 'add' ? `${this.apiBase}/roles` : `${this.apiBase}/roles/${roleId}`;
            const method = mode === 'add' ? 'POST' : 'PUT';
            
            const response = await fetch(url, {
                method: method,
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(roleData)
            });
            
            const data = await response.json();
            if (data.success) {
                this.showNotification(`User role ${mode === 'add' ? 'created' : 'updated'} successfully`, 'success');
                modalOverlay.remove();
                this.loadRoles();
            } else {
                throw new Error(data.error || `Failed to ${mode} user role`);
            }
        } catch (error) {
            console.error(`❌ Error ${mode === 'add' ? 'creating' : 'updating'} user role:`, error);
            this.showNotification(`Error ${mode === 'add' ? 'creating' : 'updating'} user role: ${error.message}`, 'error');
        }
    }

    async viewUserRole(id) {
        try {
            const response = await fetch(`${this.apiBase}/roles/${id}`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.showUserRoleModal(result.data, 'view');
                } else {
                    throw new Error(result.error || 'Failed to fetch user role');
                }
            } else {
                throw new Error('Failed to fetch user role');
            }
        } catch (error) {
            console.error('❌ Error fetching user role:', error);
            this.showNotification('Error fetching user role: ' + error.message, 'error');
        }
    }

    async editUserRole(id) {
        try {
            const response = await fetch(`${this.apiBase}/roles/${id}`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success) {
                    this.showUserRoleModal(result.data, 'edit');
                } else {
                    throw new Error(result.error || 'Failed to fetch user role');
                }
            } else {
                throw new Error('Failed to fetch user role');
            }
        } catch (error) {
            console.error('❌ Error fetching user role:', error);
            this.showNotification('Error fetching user role: ' + error.message, 'error');
        }
    }

    async deleteUserRole(id) {
        if (!confirm('Are you sure you want to delete this user role?')) {
            return;
        }

        try {
            const response = await fetch(`${this.apiBase}/roles/${id}`, {
                method: 'DELETE',
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            const data = await response.json();
            if (data.success) {
                this.showNotification('User role deleted successfully', 'success');
                this.loadRoles();
            } else {
                throw new Error(data.error || 'Failed to delete user role');
            }
        } catch (error) {
            console.error('❌ Error deleting user role:', error);
            this.showNotification('Error deleting user role: ' + error.message, 'error');
        }
    }

    async loadUsersForSelect(selectElement) {
        try {
            const response = await fetch(`${this.apiBase}/users`, {
                headers: { 'Authorization': `Bearer ${this.token}` }
            });
            
            if (response.ok) {
                const result = await response.json();
                if (result.success && result.data) {
                    selectElement.innerHTML = '<option value="">Select User</option>' + 
                        result.data.map(user => `<option value="${user.id}">${user.full_name} (${user.email})</option>`).join('');
                }
            }
        } catch (error) {
            console.error('❌ Error loading users:', error);
        }
    }
}

// Global functions for user actions
async function approveUser(userId) {
    console.log(`✅ Approving user ${userId}`);
    
    if (!confirm('Are you sure you want to approve this user?')) return;

    try {
        const response = await fetch(`${adminManager.apiBase}/users/approve`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                user_id: userId,
                status: 'active'
            })
        });

        if (response.ok) {
            const data = await response.json();
            if (data.success) {
                adminManager.showNotification('User approved successfully!', 'success');
                adminManager.loadPendingUsers();
            } else {
                throw new Error(data.error || 'Failed to approve user');
            }
        } else {
            throw new Error('Failed to approve user');
        }
    } catch (error) {
        console.error('❌ Error approving user:', error);
        adminManager.showNotification('Error approving user: ' + error.message, 'error');
    }
}

async function rejectUser(userId) {
    const reason = prompt('Please provide a reason for rejection:');
    if (!reason) return;

    try {
        const response = await fetch(`${adminManager.apiBase}/users/reject`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                user_id: userId,
                status: 'rejected',
                reason: reason
            })
        });

        if (response.ok) {
            const data = await response.json();
            if (data.success) {
                adminManager.showNotification('User rejected successfully!', 'success');
                adminManager.loadPendingUsers();
            } else {
                throw new Error(data.error || 'Failed to reject user');
            }
        } else {
            throw new Error('Failed to reject user');
        }
    } catch (error) {
        console.error('❌ Error rejecting user:', error);
        adminManager.showNotification('Error rejecting user: ' + error.message, 'error');
    }
}

function showAddCategoryModal() {
    const modalHtml = `
        <div id="addCategoryModal" class="custom-modal" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        ">
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 500px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    padding: 20px;
                    border-bottom: 1px solid #dee2e6;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; color: #333;">Add New Category</h5>
                    <button type="button" onclick="closeAddCategoryModal()" style="
                        background: none;
                        border: none;
                        font-size: 24px;
                        cursor: pointer;
                        color: #666;
                        padding: 0;
                        width: 30px;
                        height: 30px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addCategoryForm">
                        <div style="margin-bottom: 15px;">
                            <label for="categoryName" style="display: block; margin-bottom: 5px; font-weight: bold;">Category Name *</label>
                            <input type="text" id="categoryName" name="name" required 
                                   value="" 
                                   style="
                                        width: 100%;
                                        padding: 10px;
                                        border: 1px solid #ddd;
                                        border-radius: 4px;
                                        font-size: 14px;
                                   ">
                        </div>
                        
                        <div style="margin-bottom: 15px;">
                            <label for="categoryDescription" style="display: block; margin-bottom: 5px; font-weight: bold;">Description</label>
                            <textarea id="categoryDescription" rows="3" placeholder="Enter category description (optional)" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="categoryColor" style="display: block; margin-bottom: 5px; font-weight: bold;">Category Color</label>
                            <input type="color" id="categoryColor" value="#6c757d" title="Choose category color" style="
                                width: 60px;
                                height: 40px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                cursor: pointer;
                            ">
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                ">
                    <button type="button" onclick="closeAddCategoryModal()" style="
                        padding: 8px 16px;
                        border: 1px solid #6c757d;
                        background: #6c757d;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">Cancel</button>
                    <button type="button" onclick="createCategory()" style="
                        padding: 8px 16px;
                        border: 1px solid #007bff;
                        background: #007bff;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">
                        <i class="fas fa-plus"></i> Create Category
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if any
    const existingModal = document.getElementById('addCategoryModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Focus on the name input
    setTimeout(() => {
        document.getElementById('categoryName').focus();
    }, 100);
    
    // Close modal when clicking outside
    document.getElementById('addCategoryModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeAddCategoryModal();
        }
    });
    
    // Close modal with Escape key
    document.addEventListener('keydown', function escapeHandler(e) {
        if (e.key === 'Escape') {
            closeAddCategoryModal();
            document.removeEventListener('keydown', escapeHandler);
        }
    });
}

function closeAddCategoryModal() {
    const modal = document.getElementById('addCategoryModal');
    if (modal) {
        modal.remove();
    }
}

async function createCategory() {
    const name = document.getElementById('categoryName').value.trim();
    const description = document.getElementById('categoryDescription').value.trim();
    const color = document.getElementById('categoryColor').value;
    
    // Validation
    if (!name) {
        adminManager?.showNotification('Category name is required', 'error');
        document.getElementById('categoryName').focus();
        return;
    }
    
    if (name.length < 2) {
        adminManager?.showNotification('Category name must be at least 2 characters long', 'error');
        document.getElementById('categoryName').focus();
        return;
    }
    
    try {
        console.log('🔄 Creating new category:', { name, description, color });
        
        const response = await fetch(`${adminManager.apiBase}/categories`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                name: name,
                description: description,
                color: color
            })
        });
        
        console.log('📥 Create category response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Create category response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Category created successfully!', 'success');
            
            // Close modal
            const modal = document.getElementById('addCategoryModal');
            if (modal) {
                modal.remove();
            }
            
            // Reload categories to show the new one
            if (adminManager) {
                await adminManager.loadCategories();
            }
        } else {
            throw new Error(data.error || 'Failed to create category');
        }
    } catch (error) {
        console.error('❌ Error creating category:', error);
        adminManager?.showNotification(`Error creating category: ${error.message}`, 'error');
    }
}

function editCategory(id) {
    adminManager?.showNotification(`Edit Category ${id} - Feature coming soon`, 'info');
}

function deleteCategory(id) {
    if (confirm('Are you sure you want to delete this category? This action cannot be undone.')) {
        adminManager?.showNotification(`Delete Category ${id} - Feature coming soon`, 'info');
    }
}

function showAddVenueModal() {
    const modalHtml = `
        <div id="addVenueModal" class="custom-modal" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        ">
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 600px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    padding: 20px;
                    border-bottom: 1px solid #dee2e6;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; color: #333;">Add New Venue</h5>
                    <button type="button" onclick="closeAddVenueModal()" style="
                        background: none;
                        border: none;
                        font-size: 24px;
                        cursor: pointer;
                        color: #666;
                        padding: 0;
                        width: 30px;
                        height: 30px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addVenueForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="venueName" style="display: block; margin-bottom: 5px; font-weight: bold;">Venue Name *</label>
                                <input type="text" id="venueName" required placeholder="Enter venue name" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="venueType" style="display: block; margin-bottom: 5px; font-weight: bold;">Venue Type *</label>
                                <select id="venueType" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                                    <option value="">Select venue type</option>
                                    <option value="hotel">Hotel</option>
                                    <option value="event_center">Event Center</option>
                                    <option value="cultural_site">Cultural Site</option>
                                    <option value="restaurant">Restaurant</option>
                                    <option value="conference_hall">Conference Hall</option>
                                    <option value="outdoor_space">Outdoor Space</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="venueAddress" style="display: block; margin-bottom: 5px; font-weight: bold;">Address *</label>
                            <textarea id="venueAddress" required rows="2" placeholder="Enter venue address" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="venueCity" style="display: block; margin-bottom: 5px; font-weight: bold;">City *</label>
                                <input type="text" id="venueCity" required placeholder="Enter city" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="venueRegion" style="display: block; margin-bottom: 5px; font-weight: bold;">Region *</label>
                                <select id="venueRegion" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                                    <option value="">Select region</option>
                                    <option value="Greater Accra">Greater Accra</option>
                                    <option value="Ashanti">Ashanti</option>
                                    <option value="Western">Western</option>
                                    <option value="Central">Central</option>
                                    <option value="Eastern">Eastern</option>
                                    <option value="Volta">Volta</option>
                                    <option value="Northern">Northern</option>
                                    <option value="Upper East">Upper East</option>
                                    <option value="Upper West">Upper West</option>
                                    <option value="Brong Ahafo">Brong Ahafo</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="venueCapacity" style="display: block; margin-bottom: 5px; font-weight: bold;">Capacity</label>
                                <input type="number" id="venueCapacity" min="1" placeholder="Enter capacity" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="venuePhone" style="display: block; margin-bottom: 5px; font-weight: bold;">Phone</label>
                                <input type="tel" id="venuePhone" placeholder="Enter phone number" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="venueDescription" style="display: block; margin-bottom: 5px; font-weight: bold;">Description</label>
                            <textarea id="venueDescription" rows="3" placeholder="Enter venue description (optional)" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                ">
                    <button type="button" onclick="closeAddVenueModal()" style="
                        padding: 8px 16px;
                        border: 1px solid #6c757d;
                        background: #6c757d;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">Cancel</button>
                    <button type="button" onclick="createVenue()" style="
                        padding: 8px 16px;
                        border: 1px solid #28a745;
                        background: #28a745;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">
                        <i class="fas fa-plus"></i> Create Venue
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if any
    const existingModal = document.getElementById('addVenueModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Focus on the name input
    setTimeout(() => {
        document.getElementById('venueName').focus();
    }, 100);
    
    // Close modal when clicking outside
    document.getElementById('addVenueModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeAddVenueModal();
        }
    });
    
    // Close modal with Escape key
    document.addEventListener('keydown', function escapeHandler(e) {
        if (e.key === 'Escape') {
            closeAddVenueModal();
            document.removeEventListener('keydown', escapeHandler);
        }
    });
}

function closeAddVenueModal() {
    const modal = document.getElementById('addVenueModal');
    if (modal) {
        modal.remove();
    }
}

async function createVenue() {
    const name = document.getElementById('venueName').value.trim();
    const type = document.getElementById('venueType').value;
    const address = document.getElementById('venueAddress').value.trim();
    const city = document.getElementById('venueCity').value.trim();
    const region = document.getElementById('venueRegion').value;
    const capacity = document.getElementById('venueCapacity').value;
    const phone = document.getElementById('venuePhone').value.trim();
    const description = document.getElementById('venueDescription').value.trim();
    
    // Validation
    if (!name) {
        adminManager?.showNotification('Venue name is required', 'error');
        document.getElementById('venueName').focus();
        return;
    }
    
    if (!type) {
        adminManager?.showNotification('Venue type is required', 'error');
        document.getElementById('venueType').focus();
        return;
    }
    
    if (!address) {
        adminManager?.showNotification('Address is required', 'error');
        document.getElementById('venueAddress').focus();
        return;
    }
    
    if (!city) {
        adminManager?.showNotification('City is required', 'error');
        document.getElementById('venueCity').focus();
        return;
    }
    
    if (!region) {
        adminManager?.showNotification('Region is required', 'error');
        document.getElementById('venueRegion').focus();
        return;
    }
    
    try {
        console.log('🔄 Creating new venue:', { name, type, address, city, region, capacity, phone, description });
        
        const response = await fetch(`${adminManager.apiBase}/venues`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                name: name,
                type: type,
                address: address,
                city: city,
                region: region,
                capacity: capacity ? parseInt(capacity) : null,
                contact_phone: phone,
                description: description
            })
        });
        
        console.log('📥 Create venue response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Create venue response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Venue created successfully!', 'success');
            
            // Close modal
            const modal = document.getElementById('addVenueModal');
            if (modal) {
                modal.remove();
            }
            
            // Reload venues to show the new one
            if (adminManager) {
                await adminManager.loadVenues();
            }
        } else {
            throw new Error(data.error || 'Failed to create venue');
        }
    } catch (error) {
        console.error('❌ Error creating venue:', error);
        adminManager?.showNotification(`Error creating venue: ${error.message}`, 'error');
    }
}

function showAddUserModal() { adminManager?.showNotification('Add User Modal - Feature coming soon', 'info'); }
function showAddEventModal() { adminManager?.showNotification('Add Event Modal - Feature coming soon', 'info'); }

function showAddVendorModal() {
    const modalHtml = `
        <div id="addVendorModal" class="custom-modal" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        ">
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 600px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    padding: 20px;
                    border-bottom: 1px solid #dee2e6;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; color: #333;">Add New Vendor</h5>
                    <button type="button" onclick="closeAddVendorModal()" style="
                        background: none;
                        border: none;
                        font-size: 24px;
                        cursor: pointer;
                        color: #666;
                        padding: 0;
                        width: 30px;
                        height: 30px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addVendorForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="vendorName" style="display: block; margin-bottom: 5px; font-weight: bold;">Vendor Name *</label>
                                <input type="text" id="vendorName" required placeholder="Enter vendor name" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="vendorType" style="display: block; margin-bottom: 5px; font-weight: bold;">Vendor Type *</label>
                                <select id="vendorType" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                                    <option value="">Select vendor type</option>
                                    <option value="catering">Catering</option>
                                    <option value="decoration">Decoration</option>
                                    <option value="photography">Photography</option>
                                    <option value="sound_equipment">Sound Equipment</option>
                                    <option value="security">Security</option>
                                    <option value="transportation">Transportation</option>
                                    <option value="entertainment">Entertainment</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="vendorEmail" style="display: block; margin-bottom: 5px; font-weight: bold;">Email *</label>
                                <input type="email" id="vendorEmail" required placeholder="Enter email address" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="vendorPhone" style="display: block; margin-bottom: 5px; font-weight: bold;">Phone *</label>
                                <input type="tel" id="vendorPhone" required placeholder="Enter phone number" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="vendorAddress" style="display: block; margin-bottom: 5px; font-weight: bold;">Address *</label>
                            <textarea id="vendorAddress" required rows="2" placeholder="Enter vendor address" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="vendorServices" style="display: block; margin-bottom: 5px; font-weight: bold;">Services Offered</label>
                            <textarea id="vendorServices" rows="3" placeholder="Describe services offered (optional)" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                ">
                    <button type="button" onclick="closeAddVendorModal()" style="
                        padding: 8px 16px;
                        border: 1px solid #6c757d;
                        background: #6c757d;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">Cancel</button>
                    <button type="button" onclick="createVendor()" style="
                        padding: 8px 16px;
                        border: 1px solid #28a745;
                        background: #28a745;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">
                        <i class="fas fa-plus"></i> Create Vendor
                    </button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if any
    const existingModal = document.getElementById('addVendorModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Focus on the name input
    setTimeout(() => {
        document.getElementById('vendorName').focus();
    }, 100);
    
    // Close modal when clicking outside
    document.getElementById('addVendorModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeAddVendorModal();
        }
    });
    
    // Close modal with Escape key
    document.addEventListener('keydown', function escapeHandler(e) {
        if (e.key === 'Escape') {
            closeAddVendorModal();
            document.removeEventListener('keydown', escapeHandler);
        }
    });
}

function closeAddVendorModal() {
    const modal = document.getElementById('addVendorModal');
    if (modal) {
        modal.remove();
    }
}

async function createVendor() {
    const name = document.getElementById('vendorName').value.trim();
    const type = document.getElementById('vendorType').value;
    const email = document.getElementById('vendorEmail').value.trim();
    const phone = document.getElementById('vendorPhone').value.trim();
    const address = document.getElementById('vendorAddress').value.trim();
    const services = document.getElementById('vendorServices').value.trim();
    
    // Validation
    if (!name) {
        adminManager?.showNotification('Vendor name is required', 'error');
        document.getElementById('vendorName').focus();
        return;
    }
    
    if (!type) {
        adminManager?.showNotification('Vendor type is required', 'error');
        document.getElementById('vendorType').focus();
        return;
    }
    
    if (!email) {
        adminManager?.showNotification('Email is required', 'error');
        document.getElementById('vendorEmail').focus();
        return;
    }
    
    if (!phone) {
        adminManager?.showNotification('Phone is required', 'error');
        document.getElementById('vendorPhone').focus();
        return;
    }
    
    if (!address) {
        adminManager?.showNotification('Address is required', 'error');
        document.getElementById('vendorAddress').focus();
        return;
    }
    
    try {
        console.log('🔄 Creating new vendor:', { name, type, email, phone, address, services });
        
        const response = await fetch(`${adminManager.apiBase}/vendors`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                name: name,
                type: type,
                email: email,
                phone: phone,
                address: address,
                services: services
            })
        });
        
        console.log('📥 Create vendor response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Create vendor response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Vendor created successfully!', 'success');
            
            // Close modal
            const modal = document.getElementById('addVendorModal');
            if (modal) {
                modal.remove();
            }
            
            // Reload vendors to show the new one
            if (adminManager) {
                await adminManager.loadVendors();
            }
        } else {
            throw new Error(data.error || 'Failed to create vendor');
        }
    } catch (error) {
        console.error('❌ Error creating vendor:', error);
        adminManager?.showNotification(`Error creating vendor: ${error.message}`, 'error');
    }
}

function showAddGovernmentAgencyModal() {
    const modalHtml = `
        <div id="addGovernmentAgencyModal" class="custom-modal" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        ">
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 600px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    padding: 20px;
                    border-bottom: 1px solid #dee2e6;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; color: #333;">Add New Government Agency</h5>
                    <button type="button" onclick="closeAddGovernmentAgencyModal()" style="
                        background: none;
                        border: none;
                        font-size: 24px;
                        cursor: pointer;
                        color: #666;
                        padding: 0;
                        width: 30px;
                        height: 30px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <form id="addGovernmentAgencyForm">
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="agencyName" style="display: block; margin-bottom: 5px; font-weight: bold;">Agency Name *</label>
                                <input type="text" id="agencyName" required placeholder="Enter agency name" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="agencyType" style="display: block; margin-bottom: 5px; font-weight: bold;">Agency Type *</label>
                                <select id="agencyType" required style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                                    <option value="">Select agency type</option>
                                    <option value="ministry">Ministry</option>
                                    <option value="police">Police</option>
                                    <option value="security">Security</option>
                                    <option value="health">Health Department</option>
                                    <option value="fire_service">Fire Service</option>
                                    <option value="customs">Customs</option>
                                </select>
                            </div>
                        </div>
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; margin-bottom: 15px;">
                            <div>
                                <label for="agencyEmail" style="display: block; margin-bottom: 5px; font-weight: bold;">Contact Email *</label>
                                <input type="email" id="agencyEmail" required placeholder="Enter email address" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                            <div>
                                <label for="agencyPhone" style="display: block; margin-bottom: 5px; font-weight: bold;">Contact Phone *</label>
                                <input type="tel" id="agencyPhone" required placeholder="Enter phone number" style="
                                    width: 100%;
                                    padding: 10px;
                                    border: 1px solid #ddd;
                                    border-radius: 4px;
                                    font-size: 14px;
                                ">
                            </div>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label for="agencyAddress" style="display: block; margin-bottom: 5px; font-weight: bold;">Address *</label>
                            <textarea id="agencyAddress" required rows="2" placeholder="Enter agency address" style="
                                width: 100%;
                                padding: 10px;
                                border: 1px solid #ddd;
                                border-radius: 4px;
                                font-size: 14px;
                                resize: vertical;
                            "></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                    gap: 10px;
                ">
                    <button type="button" onclick="closeAddGovernmentAgencyModal()" style="
                        padding: 8px 16px;
                        border: 1px solid #6c757d;
                        background: #6c757d;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">Cancel</button>
                    <button type="button" onclick="createGovernmentAgency()" style="
                        padding: 8px 16px;
                        border: 1px solid #28a745;
                        background: #28a745;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">
                        <i class="fas fa-plus"></i> Create Agency
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Add event listeners for modal close
    const modal = document.getElementById('addGovernmentAgencyModal');
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeAddGovernmentAgencyModal();
        }
    });
    
    // Add escape key listener
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeAddGovernmentAgencyModal();
        }
    });
}

function closeAddGovernmentAgencyModal() {
    const modal = document.getElementById('addGovernmentAgencyModal');
    if (modal) {
        modal.remove();
    }
}

async function createGovernmentAgency() {
    const name = document.getElementById('agencyName').value.trim();
    const type = document.getElementById('agencyType').value;
    const email = document.getElementById('agencyEmail').value.trim();
    const phone = document.getElementById('agencyPhone').value.trim();
    const address = document.getElementById('agencyAddress').value.trim();
    
    // Validation
    if (!name) {
        adminManager?.showNotification('Agency name is required', 'error');
        document.getElementById('agencyName').focus();
        return;
    }
    
    if (!type) {
        adminManager?.showNotification('Agency type is required', 'error');
        document.getElementById('agencyType').focus();
        return;
    }
    
    if (!email) {
        adminManager?.showNotification('Email is required', 'error');
        document.getElementById('agencyEmail').focus();
        return;
    }
    
    if (!phone) {
        adminManager?.showNotification('Phone is required', 'error');
        document.getElementById('agencyPhone').focus();
        return;
    }
    
    if (!address) {
        adminManager?.showNotification('Address is required', 'error');
        document.getElementById('agencyAddress').focus();
        return;
    }
    
    try {
        console.log('🔄 Creating new government agency:', { name, type, email, phone, address });
        
        const response = await fetch(`${adminManager.apiBase}/government-agencies`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                name: name,
                type: type,
                email: email,
                phone: phone,
                address: address
            })
        });
        
        console.log('📥 Create agency response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Create agency response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Government agency created successfully!', 'success');
            
            // Close modal
            const modal = document.getElementById('addGovernmentAgencyModal');
            if (modal) {
                modal.remove();
            }
            
            // Reload government agencies to show the new one
            if (adminManager) {
                await adminManager.loadGovernmentAgencies();
            }
        } else {
            throw new Error(data.error || 'Failed to create government agency');
        }
    } catch (error) {
        console.error('❌ Error creating government agency:', error);
        adminManager?.showNotification(`Error creating government agency: ${error.message}`, 'error');
    }
}

function showAddSafetyReportModal() { adminManager?.showNotification('Add Safety Report Modal - Feature coming soon', 'info'); }
function showAddPaymentTransactionModal() { adminManager?.showNotification('Add Payment Transaction Modal - Feature coming soon', 'info'); }
function showAddEventPermitModal() { adminManager?.showNotification('Add Event Permit Modal - Feature coming soon', 'info'); }
function showAddRoleModal() { adminManager?.showNotification('Add Role Modal - Feature coming soon', 'info'); }
function clearLogs() { if (confirm('Are you sure you want to clear all logs?')) adminManager?.showNotification('Clear Logs - Feature coming soon', 'info'); }
function exportAnalytics() { adminManager?.showNotification('Export Analytics - Feature coming soon', 'info'); }

// Venue action functions
function viewVenue(venueId) {
    const modalHtml = `
        <div id="viewVenueModal" class="custom-modal" style="
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 10000;
        ">
            <div class="modal-content" style="
                background: white;
                border-radius: 8px;
                padding: 0;
                max-width: 600px;
                width: 90%;
                max-height: 90vh;
                overflow-y: auto;
                box-shadow: 0 4px 20px rgba(0,0,0,0.3);
            ">
                <div class="modal-header" style="
                    padding: 20px;
                    border-bottom: 1px solid #dee2e6;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                ">
                    <h5 style="margin: 0; color: #333;">Venue Details</h5>
                    <button type="button" onclick="closeViewVenueModal()" style="
                        background: none;
                        border: none;
                        font-size: 24px;
                        cursor: pointer;
                        color: #666;
                        padding: 0;
                        width: 30px;
                        height: 30px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">&times;</button>
                </div>
                <div class="modal-body" style="padding: 20px;">
                    <div id="venueDetailsContent">
                        <div style="text-align: center; padding: 20px;">
                            <i class="fas fa-spinner fa-spin" style="font-size: 24px; color: #007bff;"></i>
                            <p style="margin-top: 10px;">Loading venue details...</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer" style="
                    padding: 20px;
                    border-top: 1px solid #dee2e6;
                    display: flex;
                    justify-content: flex-end;
                ">
                    <button type="button" onclick="closeViewVenueModal()" style="
                        padding: 8px 16px;
                        border: 1px solid #6c757d;
                        background: #6c757d;
                        color: white;
                        border-radius: 4px;
                        cursor: pointer;
                    ">Close</button>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal if any
    const existingModal = document.getElementById('viewVenueModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Add modal to body
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Load venue details
    loadVenueDetails(venueId);
    
    // Close modal when clicking outside
    document.getElementById('viewVenueModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeViewVenueModal();
        }
    });
}

function closeViewVenueModal() {
    const modal = document.getElementById('viewVenueModal');
    if (modal) {
        modal.remove();
    }
}

async function loadVenueDetails(venueId) {
    try {
        const response = await fetch(`${adminManager.apiBase}/venues/${venueId}`, {
            headers: {
                'Authorization': `Bearer ${adminManager.token}`,
                'Content-Type': 'application/json'
            }
        });
        
        const data = await response.json();
        
        if (data.success && data.data) {
            const venue = data.data;
            const content = document.getElementById('venueDetailsContent');
            if (content) {
                content.innerHTML = `
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div>
                            <h6 style="color: #666; margin-bottom: 5px;">Venue Name</h6>
                            <p style="margin-bottom: 15px; font-weight: bold;">${venue.name || 'N/A'}</p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">Type</h6>
                            <p style="margin-bottom: 15px;">${venue.type || 'N/A'}</p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">City</h6>
                            <p style="margin-bottom: 15px;">${venue.city || 'N/A'}</p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">Capacity</h6>
                            <p style="margin-bottom: 15px;">${venue.capacity || 'N/A'}</p>
                        </div>
                        <div>
                            <h6 style="color: #666; margin-bottom: 5px;">Region</h6>
                            <p style="margin-bottom: 15px;">${venue.region || 'N/A'}</p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">Status</h6>
                            <p style="margin-bottom: 15px;">
                                <span style="padding: 4px 8px; border-radius: 4px; background: ${venue.status === 'active' ? '#28a745' : '#dc3545'}; color: white;">
                                    ${venue.status || 'N/A'}
                                </span>
                            </p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">Phone</h6>
                            <p style="margin-bottom: 15px;">${venue.phone || 'N/A'}</p>
                            
                            <h6 style="color: #666; margin-bottom: 5px;">Events Count</h6>
                            <p style="margin-bottom: 15px;">${venue.event_count || 0}</p>
                        </div>
                    </div>
                    <div style="margin-top: 20px;">
                        <h6 style="color: #666; margin-bottom: 5px;">Address</h6>
                        <p style="margin-bottom: 15px;">${venue.address || 'N/A'}</p>
                        
                        <h6 style="color: #666; margin-bottom: 5px;">Description</h6>
                        <p style="margin-bottom: 0;">${venue.description || 'No description available'}</p>
                    </div>
                `;
            }
        } else {
            throw new Error(data.error || 'Failed to load venue details');
        }
    } catch (error) {
        console.error('❌ Error loading venue details:', error);
        const content = document.getElementById('venueDetailsContent');
        if (content) {
            content.innerHTML = `
                <div style="text-align: center; padding: 20px; color: #dc3545;">
                    <i class="fas fa-exclamation-triangle" style="font-size: 24px; margin-bottom: 10px;"></i>
                    <p>Error loading venue details: ${error.message}</p>
                </div>
            `;
        }
    }
}

async function verifyVenue(venueId) {
    if (!confirm('Are you sure you want to verify this venue? This will mark it as verified and active.')) {
        return;
    }
    
    try {
        console.log(`✅ Verifying venue ${venueId}`);
        
        const response = await fetch(`${adminManager.apiBase}/venues/${venueId}/verify`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                status: 'verified',
                verified_at: new Date().toISOString()
            })
        });
        
        console.log('📥 Verify venue response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Verify venue response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Venue verified successfully!', 'success');
            
            // Reload venues to show updated status
            if (adminManager) {
                await adminManager.loadVenues();
            }
        } else {
            throw new Error(data.error || 'Failed to verify venue');
        }
    } catch (error) {
        console.error('❌ Error verifying venue:', error);
        adminManager?.showNotification(`Error verifying venue: ${error.message}`, 'error');
    }
}

async function suspendVenue(venueId) {
    const reason = prompt('Please provide a reason for suspending this venue:');
    if (!reason) return;
    
    try {
        console.log(`⛔ Suspending venue ${venueId}`);
        
        const response = await fetch(`${adminManager.apiBase}/venues/${venueId}/suspend`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminManager.token}`
            },
            body: JSON.stringify({
                status: 'suspended',
                reason: reason,
                suspended_at: new Date().toISOString()
            })
        });
        
        console.log('📥 Suspend venue response status:', response.status);
        
        const data = await response.json();
        console.log('📄 Suspend venue response:', data);
        
        if (data.success) {
            adminManager?.showNotification('Venue suspended successfully!', 'success');
            
            // Reload venues to show updated status
            if (adminManager) {
                await adminManager.loadVenues();
            }
        } else {
            throw new Error(data.error || 'Failed to suspend venue');
        }
    } catch (error) {
        console.error('❌ Error suspending venue:', error);
        adminManager?.showNotification(`Error suspending venue: ${error.message}`, 'error');
    }
}

// Global admin manager instance
adminManager;

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Initializing Admin Dashboard...');
    adminManager = new AdminManager();
    
    // Show dashboard by default
    setTimeout(() => {
        adminManager.showSection('dashboard-section');
    }, 100);
});

// Global helper functions
function showSection(sectionId, element) {
    if (adminManager) {
        adminManager.showSection(sectionId, element);
    }
}

// Search functions
function searchPendingUsers() {
    const searchTerm = document.getElementById('searchPendingUsers')?.value.toLowerCase() || '';
    const rows = document.querySelectorAll('#pendingUsersTableBody tr');
    
    rows.forEach(row => {
        const text = row.textContent.toLowerCase();
        row.style.display = text.includes(searchTerm) ? '' : 'none';
    });
}
