// Dashboard JavaScript
class Dashboard {
    constructor() {
        this.user = null;
        this.userRole = null;
        this.stats = {};
        this.currentSection = 'overview';
        
        this.init();
    }

    async init() {
        await this.checkAuth();
        await this.loadUserData();
        this.setupUI();
        this.loadDashboardData();
    }

    async checkAuth() {
        const token = localStorage.getItem('token');
        if (!token) {
            window.location.href = 'index.html';
            return;
        }

        try {
            const response = await fetch('/api/auth.php?action=verify', {
                headers: {
                    'Authorization': `Bearer ${token}`
                }
            });

            if (!response.ok) {
                throw new Error('Invalid token');
            }

            const data = await response.json();
            this.user = data.user;
            this.userRole = data.user.role;
        } catch (error) {
            console.error('Auth check failed:', error);
            localStorage.removeItem('token');
            window.location.href = 'index.html';
        }
    }

    async loadUserData() {
        if (!this.user) return;

        // Update user info in header
        document.getElementById('userName').textContent = this.user.full_name || this.user.email;
        document.getElementById('userAvatar').textContent = (this.user.full_name || this.user.email).charAt(0).toUpperCase();
        
        // Update dashboard title based on role
        const titles = {
            'organizer': 'Event Organizer Dashboard',
            'attendee': 'My Events Dashboard',
            'venue': 'Venue Management Dashboard',
            'government': 'Government Oversight Dashboard'
        };
        
        document.getElementById('dashboardTitle').textContent = titles[this.userRole] || 'Dashboard';
        document.getElementById('dashboardSubtitle').textContent = `Welcome back, ${this.user.full_name || this.user.email}!`;
    }

    setupUI() {
        this.setupSidebar();
        this.setupQuickActions();
        document.body.className = `role-${this.userRole}`;
    }

    setupSidebar() {
        const sidebarMenu = document.getElementById('sidebarMenu');
        const menuItems = this.getMenuItems();
        
        sidebarMenu.innerHTML = menuItems.map(item => `
            <li>
                <a href="#" onclick="dashboard.switchSection('${item.section}')" 
                   class="${item.section === 'overview' ? 'active' : ''}" 
                   data-section="${item.section}">
                    <i class="${item.icon}"></i>
                    <span>${item.title}</span>
                </a>
            </li>
        `).join('');
    }

    getMenuItems() {
        const commonItems = [
            { section: 'overview', title: 'Overview', icon: 'fas fa-tachometer-alt' },
            { section: 'notifications', title: 'Notifications', icon: 'fas fa-bell' },
            { section: 'profile', title: 'Profile', icon: 'fas fa-user' }
        ];

        const roleSpecificItems = {
            'organizer': [
                { section: 'events', title: 'My Events', icon: 'fas fa-calendar' },
                { section: 'create-event', title: 'Create Event', icon: 'fas fa-plus' },
                { section: 'analytics', title: 'Analytics', icon: 'fas fa-chart-bar' },
                { section: 'payments', title: 'Payments', icon: 'fas fa-credit-card' }
            ],
            'attendee': [
                { section: 'my-events', title: 'My Events', icon: 'fas fa-ticket-alt' },
                { section: 'recommendations', title: 'Recommendations', icon: 'fas fa-star' },
                { section: 'chat', title: 'Event Chats', icon: 'fas fa-comments' }
            ],
            'venue': [
                { section: 'bookings', title: 'Bookings', icon: 'fas fa-calendar-check' },
                { section: 'venue-profile', title: 'Venue Profile', icon: 'fas fa-building' },
                { section: 'revenue', title: 'Revenue', icon: 'fas fa-money-bill' }
            ],
            'government': [
                { section: 'oversight', title: 'Event Oversight', icon: 'fas fa-eye' },
                { section: 'approvals', title: 'Approvals', icon: 'fas fa-check-circle' },
                { section: 'reports', title: 'Reports', icon: 'fas fa-file-alt' },
                { section: 'analytics', title: 'Analytics', icon: 'fas fa-chart-line' }
            ]
        };

        return [
            ...commonItems.slice(0, 1),
            ...(roleSpecificItems[this.userRole] || []),
            ...commonItems.slice(1)
        ];
    }

    setupQuickActions() {
        const quickActions = document.getElementById('quickActions');
        const actions = this.getQuickActions();
        
        quickActions.innerHTML = actions.map(action => `
            <div class="action-card" onclick="${action.onclick}">
                <div class="action-icon">
                    <i class="${action.icon}"></i>
                </div>
                <div class="action-title">${action.title}</div>
                <div class="action-description">${action.description}</div>
            </div>
        `).join('');
    }

    getQuickActions() {
        const roleActions = {
            'organizer': [
                { title: 'Create Event', description: 'Start planning a new event', icon: 'fas fa-plus', onclick: 'dashboard.createEvent()' },
                { title: 'View Analytics', description: 'Check event performance', icon: 'fas fa-chart-bar', onclick: 'dashboard.switchSection("analytics")' },
                { title: 'Manage Payments', description: 'Handle transactions', icon: 'fas fa-credit-card', onclick: 'dashboard.switchSection("payments")' }
            ],
            'attendee': [
                { title: 'Browse Events', description: 'Find new events to attend', icon: 'fas fa-search', onclick: 'window.location.href="events.html"' },
                { title: 'My Tickets', description: 'View purchased tickets', icon: 'fas fa-ticket-alt', onclick: 'dashboard.switchSection("my-events")' },
                { title: 'Get Recommendations', description: 'AI-powered suggestions', icon: 'fas fa-star', onclick: 'dashboard.switchSection("recommendations")' }
            ],
            'venue': [
                { title: 'Update Profile', description: 'Manage venue information', icon: 'fas fa-edit', onclick: 'dashboard.switchSection("venue-profile")' },
                { title: 'View Bookings', description: 'Check upcoming bookings', icon: 'fas fa-calendar-check', onclick: 'dashboard.switchSection("bookings")' },
                { title: 'Revenue Report', description: 'Track earnings', icon: 'fas fa-money-bill', onclick: 'dashboard.switchSection("revenue")' }
            ],
            'government': [
                { title: 'Review Events', description: 'Approve pending events', icon: 'fas fa-gavel', onclick: 'dashboard.switchSection("approvals")' },
                { title: 'Safety Reports', description: 'Monitor safety incidents', icon: 'fas fa-shield-alt', onclick: 'dashboard.switchSection("reports")' },
                { title: 'Platform Analytics', description: 'View system metrics', icon: 'fas fa-chart-line', onclick: 'dashboard.switchSection("analytics")' }
            ]
        };

        return roleActions[this.userRole] || [];
    }

    async loadDashboardData() {
        try {
            await this.loadStats();
            await this.loadContent();
            
            document.getElementById('loadingState').style.display = 'none';
            document.getElementById('dashboardContainer').style.display = 'block';
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showError('Failed to load dashboard data');
        }
    }

    async loadStats() {
        try {
            const response = await fetch(`/api/government.php?action=dashboard_stats&role=${this.userRole}`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            });

            if (response.ok) {
                const data = await response.json();
                this.stats = data.stats || {};
                this.displayStats();
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    displayStats() {
        const statsGrid = document.getElementById('statsGrid');
        const statCards = this.getStatCards();
        
        statsGrid.innerHTML = statCards.map(stat => `
            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">${stat.title}</div>
                    <div class="stat-icon">
                        <i class="${stat.icon}"></i>
                    </div>
                </div>
                <div class="stat-value">${stat.value}</div>
                ${stat.change ? `
                    <div class="stat-change ${stat.change > 0 ? 'positive' : 'negative'}">
                        <i class="fas fa-${stat.change > 0 ? 'arrow-up' : 'arrow-down'}"></i>
                        ${Math.abs(stat.change)}% from last month
                    </div>
                ` : ''}
            </div>
        `).join('');
    }

    getStatCards() {
        const roleStats = {
            'organizer': [
                { title: 'Total Events', value: this.stats.total_events || 0, icon: 'fas fa-calendar', change: 12 },
                { title: 'Active Events', value: this.stats.active_events || 0, icon: 'fas fa-play', change: 5 },
                { title: 'Total Attendees', value: this.stats.total_attendees || 0, icon: 'fas fa-users', change: 18 },
                { title: 'Revenue', value: `GHS ${this.stats.total_revenue || 0}`, icon: 'fas fa-money-bill', change: 25 }
            ],
            'attendee': [
                { title: 'Events Attended', value: this.stats.events_attended || 0, icon: 'fas fa-ticket-alt' },
                { title: 'Upcoming Events', value: this.stats.upcoming_events || 0, icon: 'fas fa-calendar-plus' },
                { title: 'Favorite Categories', value: this.stats.favorite_category || 'Music', icon: 'fas fa-heart' },
                { title: 'Total Spent', value: `GHS ${this.stats.total_spent || 0}`, icon: 'fas fa-credit-card' }
            ],
            'venue': [
                { title: 'Total Bookings', value: this.stats.total_bookings || 0, icon: 'fas fa-calendar-check', change: 8 },
                { title: 'Active Bookings', value: this.stats.active_bookings || 0, icon: 'fas fa-clock' },
                { title: 'Revenue', value: `GHS ${this.stats.venue_revenue || 0}`, icon: 'fas fa-money-bill', change: 15 },
                { title: 'Rating', value: `${this.stats.average_rating || 0}/5`, icon: 'fas fa-star' }
            ],
            'government': [
                { title: 'Total Events', value: this.stats.platform_events || 0, icon: 'fas fa-calendar', change: 20 },
                { title: 'Pending Approvals', value: this.stats.pending_approvals || 0, icon: 'fas fa-clock' },
                { title: 'Safety Reports', value: this.stats.safety_reports || 0, icon: 'fas fa-shield-alt' },
                { title: 'Platform Revenue', value: `GHS ${this.stats.platform_revenue || 0}`, icon: 'fas fa-chart-line', change: 30 }
            ]
        };

        return roleStats[this.userRole] || [];
    }

    async loadContent() {
        const contentSections = document.getElementById('contentSections');
        const content = await this.getContentSections();
        contentSections.innerHTML = content;
    }

    async getContentSections() {
        switch (this.userRole) {
            case 'organizer':
                return this.getOrganizerContent();
            case 'attendee':
                return this.getAttendeeContent();
            case 'venue':
                return this.getVenueContent();
            case 'government':
                return this.getGovernmentContent();
            default:
                return '<div class="content-section"><p>Welcome to your dashboard!</p></div>';
        }
    }

    getOrganizerContent() {
        return `
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Recent Events</h3>
                    <button class="btn btn-primary" onclick="dashboard.createEvent()">
                        <i class="fas fa-plus"></i> Create Event
                    </button>
                </div>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Event Name</th>
                                <th>Date</th>
                                <th>Attendees</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="eventsTable">
                            <tr><td colspan="5" class="loading">Loading events...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    }

    getAttendeeContent() {
        return `
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Upcoming Events</h3>
                    <a href="events.html" class="btn btn-primary">
                        <i class="fas fa-search"></i> Browse Events
                    </a>
                </div>
                <div id="upcomingEvents">Loading...</div>
            </div>
            
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Recommended Events</h3>
                </div>
                <div id="recommendedEvents">Loading recommendations...</div>
            </div>
        `;
    }

    getVenueContent() {
        return `
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Recent Bookings</h3>
                    <button class="btn btn-primary" onclick="dashboard.switchSection('venue-profile')">
                        <i class="fas fa-edit"></i> Update Profile
                    </button>
                </div>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Event</th>
                                <th>Date</th>
                                <th>Organizer</th>
                                <th>Revenue</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody id="bookingsTable">
                            <tr><td colspan="5" class="loading">Loading bookings...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        `;
    }

    getGovernmentContent() {
        return `
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Pending Event Approvals</h3>
                    <button class="btn btn-primary" onclick="dashboard.switchSection('approvals')">
                        <i class="fas fa-gavel"></i> Review All
                    </button>
                </div>
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Event Name</th>
                                <th>Organizer</th>
                                <th>Date</th>
                                <th>Location</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="approvalsTable">
                            <tr><td colspan="5" class="loading">Loading approvals...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <div class="content-section">
                <div class="section-header">
                    <h3 class="section-title">Platform Analytics</h3>
                </div>
                <div class="chart-container">
                    <i class="fas fa-chart-line" style="font-size: 3rem; margin-bottom: 1rem;"></i>
                    <p>Analytics charts will be displayed here</p>
                </div>
            </div>
        `;
    }

    switchSection(section) {
        // Update active menu item
        document.querySelectorAll('.sidebar-menu a').forEach(link => {
            link.classList.remove('active');
        });
        document.querySelector(`[data-section="${section}"]`).classList.add('active');
        
        this.currentSection = section;
        this.loadSectionContent(section);
    }

    async loadSectionContent(section) {
        // Implementation for loading specific section content
        console.log(`Loading section: ${section}`);
    }

    createEvent() {
        // Show create event modal or redirect
        window.location.href = 'events.html#create';
    }

    logout() {
        localStorage.removeItem('token');
        window.location.href = 'index.html';
    }

    showError(message) {
        console.error(message);
        // Show error notification
    }
}

// Initialize dashboard
let dashboard;
document.addEventListener('DOMContentLoaded', () => {
    dashboard = new Dashboard();
});

// Modal styles
const modalStyles = `
    .modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 10000;
        padding: 2rem;
    }

    .modal-content {
        background: white;
        border-radius: 16px;
        max-width: 600px;
        width: 100%;
        max-height: 90vh;
        overflow-y: auto;
        box-shadow: 0 20px 25px rgba(0,0,0,0.25);
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 2rem 2rem 1rem;
        border-bottom: 1px solid #e2e8f0;
    }

    .modal-close {
        background: none;
        border: none;
        font-size: 1.5rem;
        cursor: pointer;
        color: #718096;
        padding: 0.5rem;
        border-radius: 50%;
        transition: all 0.3s ease;
    }

    .modal-close:hover {
        background: #f7fafc;
        color: #1a202c;
    }

    .modal-body {
        padding: 2rem;
    }

    .modal-footer {
        display: flex;
        justify-content: flex-end;
        gap: 1rem;
        padding: 1rem 2rem 2rem;
        border-top: 1px solid #e2e8f0;
    }
`;

// Inject modal styles
const styleSheet = document.createElement('style');
styleSheet.textContent = modalStyles;
document.head.appendChild(styleSheet);
