// Events Page JavaScript
class EventsManager {
    constructor() {
        this.currentPage = 1;
        this.itemsPerPage = 12;
        this.currentView = 'grid';
        this.filters = {};
        this.events = [];
        this.categories = [];
        this.totalEvents = 0;
        
        this.init();
    }

    async init() {
        await this.loadCategories();
        await this.loadEvents();
        this.setupEventListeners();
        this.updateAuthUI();
    }

    setupEventListeners() {
        // Search form
        const searchForm = document.getElementById('searchForm');
        if (searchForm) {
            searchForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleSearch();
            });
        }

        // Real-time search
        const searchQuery = document.getElementById('searchQuery');
        if (searchQuery) {
            let searchTimeout;
            searchQuery.addEventListener('input', (e) => {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    this.handleSearch();
                }, 500);
            });
        }

        // Filter changes
        const filters = ['categoryFilter', 'locationFilter', 'dateFromFilter', 'dateToFilter', 'priceFilter', 'verifiedFilter'];
        filters.forEach(filterId => {
            const element = document.getElementById(filterId);
            if (element) {
                element.addEventListener('change', () => {
                    this.handleSearch();
                });
            }
        });
    }

    async loadCategories() {
        try {
            console.log('🔄 Loading event categories...');
            
            const response = await fetch('api/events.php?action=categories', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                }
            });
            
            console.log('📡 Categories response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            console.log('📦 Categories data:', {
                success: data.success,
                categoriesCount: Array.isArray(data.data) ? data.data.length : 'not array',
                hasError: !!data.error
            });
            
            if (data.success && data.data) {
                console.log('✅ Categories loaded:', data.data.length, 'categories');
                this.categories = data.data;
                this.populateCategories();
            } else {
                console.error('❌ Categories API error:', data.error);
                throw new Error(data.error || 'Failed to load categories');
            }
            
        } catch (error) {
            console.error('💥 Error loading categories:', error);
        }
    }

    populateCategories() {
        const categorySelect = document.getElementById('categoryFilter');
        if (categorySelect && this.categories.length > 0) {
            // Clear existing options except "All Categories"
            categorySelect.innerHTML = '<option value="">All Categories</option>';
            
            this.categories.forEach(category => {
                const option = document.createElement('option');
                option.value = category.name;
                option.textContent = category.name;
                categorySelect.appendChild(option);
            });
        }
    }

    async loadEvents(page = 1) {
        try {
            this.showLoading();
            
            const params = new URLSearchParams({
                action: 'list',
                page: page,
                limit: this.itemsPerPage,
                ...this.filters
            });

            const apiUrl = `api/events.php?${params}`;
            console.log('🔄 Loading events from:', apiUrl);
            console.log('📋 Request parameters:', Object.fromEntries(params));
            
            const response = await fetch(apiUrl, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                }
            });
            
            console.log('📡 Response status:', response.status, response.statusText);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            console.log('📦 Response data structure:', {
                success: data.success,
                dataType: typeof data.data,
                dataLength: Array.isArray(data.data) ? data.data.length : 'not array',
                pagination: data.pagination,
                hasError: !!data.error
            });
            
            if (data.success && data.data) {
                console.log('✅ Events loaded successfully:', data.data.length, 'events');
                this.events = data.data;
                this.totalEvents = data.pagination?.total || 0;
                this.currentPage = page;
                
                this.displayEvents();
                this.updateResultsCount();
                this.setupPagination();
                this.hideLoading();
            } else {
                console.error('❌ API returned error:', data.error || 'Unknown error');
                throw new Error(data.error || 'Failed to load events');
            }
            
        } catch (error) {
            console.error('💥 Error loading events:', error);
            this.showError('Failed to load events: ' + error.message);
        } finally {
            this.hideLoading();
        }
    }

    handleSearch() {
        // Collect filter values
        this.filters = {};
        
        const searchQuery = document.getElementById('searchQuery').value.trim();
        if (searchQuery) {
            this.filters.search = searchQuery;
        }

        const category = document.getElementById('categoryFilter').value;
        if (category) {
            this.filters.category = category;
        }

        const location = document.getElementById('locationFilter').value;
        if (location) {
            this.filters.location = location;
        }

        const dateFrom = document.getElementById('dateFromFilter').value;
        if (dateFrom) {
            this.filters.date_from = dateFrom;
        }

        const dateTo = document.getElementById('dateToFilter').value;
        if (dateTo) {
            this.filters.date_to = dateTo;
        }

        const priceRange = document.getElementById('priceFilter').value;
        if (priceRange) {
            this.filters.price_range = priceRange;
        }

        const verified = document.getElementById('verifiedFilter').value;
        if (verified) {
            this.filters.verified = verified;
        }

        // Reset to first page and load events
        this.loadEvents(1);
    }

    displayEvents() {
        const eventsGrid = document.getElementById('eventsGrid');
        const eventsList = document.getElementById('eventsList');
        const emptyState = document.getElementById('emptyState');

        if (this.events.length === 0) {
            eventsGrid.style.display = 'none';
            eventsList.style.display = 'none';
            emptyState.style.display = 'block';
            return;
        }

        emptyState.style.display = 'none';

        if (this.currentView === 'grid') {
            eventsGrid.innerHTML = this.events.map(event => this.createEventCard(event)).join('');
            eventsGrid.style.display = 'grid';
            eventsList.style.display = 'none';
        } else {
            eventsList.innerHTML = this.events.map(event => this.createEventListItem(event)).join('');
            eventsList.style.display = 'flex';
            eventsGrid.style.display = 'none';
        }
    }

    createEventCard(event) {
        const eventDate = new Date(event.start_datetime);
        const formattedDate = eventDate.toLocaleDateString('en-GB', {
            weekday: 'short',
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
        
        const formattedTime = event.start_datetime ? 
            new Date(event.start_datetime).toLocaleTimeString('en-GB', {
                hour: '2-digit',
                minute: '2-digit'
            }) : '';

        const price = event.ticket_price > 0 ? 
            `GHS ${parseFloat(event.ticket_price).toFixed(2)}` : 'FREE';
        
        const priceClass = event.ticket_price > 0 ? '' : 'free';

        return `
            <div class="event-card" data-event-id="${event.id}">
                <div class="event-image">
                    ${event.image_url ? 
                        `<img src="${event.image_url}" alt="${event.title}" onerror="this.style.display='none'">` : 
                        ''
                    }
                    ${event.government_verified ? 
                        '<div class="event-badge verified">Government Verified</div>' : 
                        '<div class="event-badge">Pending Verification</div>'
                    }
                </div>
                <div class="event-content">
                    <div class="event-category">${event.category_name || 'General'}</div>
                    <h3 class="event-title">${event.title}</h3>
                    <div class="event-meta">
                        <div class="event-meta-item">
                            <i class="fas fa-calendar"></i>
                            <span>${formattedDate}</span>
                        </div>
                        ${formattedTime ? `
                            <div class="event-meta-item">
                                <i class="fas fa-clock"></i>
                                <span>${formattedTime}</span>
                            </div>
                        ` : ''}
                        <div class="event-meta-item">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>${event.venue_address}, ${event.city}</span>
                        </div>
                        ${event.max_attendees ? `
                            <div class="event-meta-item">
                                <i class="fas fa-users"></i>
                                <span>${event.current_attendees || 0}/${event.max_attendees} attendees</span>
                            </div>
                        ` : ''}
                    </div>
                    <div class="event-price ${priceClass}">${price}</div>
                    <div class="event-actions">
                        <button class="btn-book" onclick="eventsManager.bookEvent(${event.id})">
                            <i class="fas fa-ticket-alt"></i>
                            Book Now
                        </button>
                        <button class="btn-details" onclick="eventsManager.showEventDetails(${event.id})">
                            Details
                        </button>
                    </div>
                </div>
            </div>
        `;
    }

    createEventListItem(event) {
        const eventDate = new Date(event.start_datetime);
        const formattedDate = eventDate.toLocaleDateString('en-GB', {
            weekday: 'short',
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
        
        const price = event.ticket_price > 0 ? 
            `GHS ${parseFloat(event.ticket_price).toFixed(2)}` : 'FREE';

        return `
            <div class="event-card-list" data-event-id="${event.id}">
                <div class="event-image-list">
                    ${event.image_url ? 
                        `<img src="${event.image_url}" alt="${event.title}" style="width: 100%; height: 100%; object-fit: cover; border-radius: 8px;" onerror="this.style.display='none'">` : 
                        ''
                    }
                </div>
                <div class="event-content-list">
                    <div class="event-category">${event.category_name || 'General'}</div>
                    <h3 class="event-title">${event.title}</h3>
                    <div class="event-meta">
                        <div class="event-meta-item">
                            <i class="fas fa-calendar"></i>
                            <span>${formattedDate}</span>
                        </div>
                        <div class="event-meta-item">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>${event.venue_address}, ${event.city}</span>
                        </div>
                    </div>
                </div>
                <div class="event-actions-list">
                    <div class="event-price">${price}</div>
                    <button class="btn-book" onclick="eventsManager.bookEvent(${event.id})">Book</button>
                    <button class="btn-details" onclick="eventsManager.showEventDetails(${event.id})">Details</button>
                </div>
            </div>
        `;
    }

    setupPagination() {
        const pagination = document.getElementById('pagination');
        const totalPages = Math.ceil(this.totalEvents / this.itemsPerPage);

        if (totalPages <= 1) {
            pagination.style.display = 'none';
            return;
        }

        pagination.style.display = 'flex';
        
        let paginationHTML = '';

        // Previous button
        paginationHTML += `
            <button ${this.currentPage === 1 ? 'disabled' : ''} 
                    onclick="eventsManager.loadEvents(${this.currentPage - 1})">
                <i class="fas fa-chevron-left"></i>
            </button>
        `;

        // Page numbers
        const startPage = Math.max(1, this.currentPage - 2);
        const endPage = Math.min(totalPages, this.currentPage + 2);

        if (startPage > 1) {
            paginationHTML += `<button onclick="eventsManager.loadEvents(1)">1</button>`;
            if (startPage > 2) {
                paginationHTML += `<span>...</span>`;
            }
        }

        for (let i = startPage; i <= endPage; i++) {
            paginationHTML += `
                <button class="${i === this.currentPage ? 'active' : ''}" 
                        onclick="eventsManager.loadEvents(${i})">
                    ${i}
                </button>
            `;
        }

        if (endPage < totalPages) {
            if (endPage < totalPages - 1) {
                paginationHTML += `<span>...</span>`;
            }
            paginationHTML += `<button onclick="eventsManager.loadEvents(${totalPages})">${totalPages}</button>`;
        }

        // Next button
        paginationHTML += `
            <button ${this.currentPage === totalPages ? 'disabled' : ''} 
                    onclick="eventsManager.loadEvents(${this.currentPage + 1})">
                <i class="fas fa-chevron-right"></i>
            </button>
        `;

        pagination.innerHTML = paginationHTML;
    }

    updateResultsCount() {
        const resultsCount = document.getElementById('resultsCount');
        if (resultsCount) {
            const start = (this.currentPage - 1) * this.itemsPerPage + 1;
            const end = Math.min(this.currentPage * this.itemsPerPage, this.totalEvents);
            resultsCount.textContent = `Showing ${start}-${end} of ${this.totalEvents} events`;
        }
    }

    async showEventDetails(eventId) {
        try {
            console.log('🔄 Loading event details for ID:', eventId);
            
            const response = await fetch(`api/events.php?action=details&id=${eventId}`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                }
            });
            
            console.log('📡 Event details response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            console.log('📦 Event details data:', {
                success: data.success,
                hasEventData: !!data.data,
                eventTitle: data.data?.title,
                hasError: !!data.error
            });
            
            if (data.success && data.data) {
                console.log('✅ Event details loaded for:', data.data.title);
                this.displayEventModal(data.data);
            } else {
                console.error('❌ Event details API error:', data.error);
                throw new Error(data.error || 'Failed to load event details');
            }
            
        } catch (error) {
            console.error('💥 Error loading event details:', error);
            this.showError('Failed to load event details: ' + error.message);
        }
    }

    displayEventModal(event) {
        const modal = document.getElementById('eventDetailModal');
        const eventDate = new Date(event.start_datetime);
        const formattedDate = eventDate.toLocaleDateString('en-GB', {
            weekday: 'long',
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        
        const formattedTime = event.start_datetime ? 
            new Date(event.start_datetime).toLocaleTimeString('en-GB', {
                hour: '2-digit',
                minute: '2-digit'
            }) : '';

        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h2>${event.title}</h2>
                    <button class="modal-close" onclick="this.closest('.modal').style.display='none'">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div class="modal-body">
                    ${event.image_url ? `
                        <img src="${event.image_url}" alt="${event.title}" style="width: 100%; height: 300px; object-fit: cover; border-radius: 12px; margin-bottom: 1.5rem;">
                    ` : ''}
                    
                    <div class="event-details-grid">
                        <div class="event-info">
                            <h3>Event Information</h3>
                            <div class="info-item">
                                <i class="fas fa-calendar"></i>
                                <span><strong>Date:</strong> ${formattedDate}</span>
                            </div>
                            ${formattedTime ? `
                                <div class="info-item">
                                    <i class="fas fa-clock"></i>
                                    <span><strong>Time:</strong> ${formattedTime}</span>
                                </div>
                            ` : ''}
                            <div class="info-item">
                                <i class="fas fa-map-marker-alt"></i>
                                <span><strong>Location:</strong> ${event.venue_address}, ${event.city}</span>
                            </div>
                            <div class="info-item">
                                <i class="fas fa-tag"></i>
                                <span><strong>Category:</strong> ${event.category_name || 'General'}</span>
                            </div>
                            ${event.max_attendees ? `
                                <div class="info-item">
                                    <i class="fas fa-users"></i>
                                    <span><strong>Capacity:</strong> ${event.current_attendees || 0}/${event.max_attendees}</span>
                                </div>
                            ` : ''}
                            <div class="info-item">
                                <i class="fas fa-money-bill"></i>
                                <span><strong>Price:</strong> ${event.ticket_price > 0 ? `GHS ${parseFloat(event.ticket_price).toFixed(2)}` : 'FREE'}</span>
                            </div>
                            ${event.government_verified ? `
                                <div class="info-item verified">
                                    <i class="fas fa-check-circle"></i>
                                    <span><strong>Government Verified</strong></span>
                                </div>
                            ` : ''}
                        </div>
                        
                        <div class="event-description">
                            <h3>Description</h3>
                            <p>${event.description || 'No description available.'}</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-outline" onclick="this.closest('.modal').style.display='none'">Close</button>
                    <button class="btn btn-primary" onclick="eventsManager.bookEvent(${event.id})">
                        <i class="fas fa-ticket-alt"></i>
                        Book Tickets
                    </button>
                </div>
            </div>
        `;

        modal.style.display = 'flex';
    }

    async bookEvent(eventId) {
        const token = localStorage.getItem('token');
        if (!token) {
            this.showNotification('Please login to book tickets', 'warning');
            app.showLoginModal();
            return;
        }

        try {
            console.log('🎫 Booking event:', eventId);
            
            const response = await fetch('api/events.php?action=register', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    event_id: eventId,
                    tickets: 1
                })
            });
            
            console.log('📡 Booking response status:', response.status);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            console.log('📦 Booking response:', {
                success: data.success,
                ticketId: data.ticket_id,
                message: data.message,
                hasError: !!data.error
            });
            
            if (data.success) {
                console.log('✅ Event booked successfully, ticket ID:', data.ticket_id);
                this.showNotification(data.message || 'Event booked successfully!', 'success');
                // Refresh events to update attendee count
                this.loadEvents(this.currentPage);
            } else {
                console.error('❌ Booking API error:', data.error);
                throw new Error(data.error || 'Failed to book event');
            }
            
        } catch (error) {
            console.error('💥 Error booking event:', error);
            this.showError('Failed to book event: ' + error.message);
        }
    }

    showLoading() {
        document.getElementById('loadingState').style.display = 'block';
        document.getElementById('eventsGrid').style.display = 'none';
        document.getElementById('eventsList').style.display = 'none';
        document.getElementById('emptyState').style.display = 'none';
    }

    hideLoading() {
        document.getElementById('loadingState').style.display = 'none';
    }

    showError(message) {
        console.error('🚨 Showing error to user:', message);
        
        // Remove any existing error messages
        const existingError = document.querySelector('.error-message');
        if (existingError) {
            existingError.remove();
        }
        
        // Create and show error message
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #ff4757;
            color: white;
            padding: 15px 20px;
            border-radius: 5px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            z-index: 10000;
            max-width: 400px;
        `;
        errorDiv.textContent = message;
        
        document.body.appendChild(errorDiv);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (errorDiv.parentNode) {
                errorDiv.remove();
            }
        }, 5000);
    }
    
    showNotification(message, type = 'info') {
        console.log('✅ Showing notification to user:', message);
        
        // Remove any existing success messages
        const existingSuccess = document.querySelector('.success-message');
        if (existingSuccess) {
            existingSuccess.remove();
        }
        
        // Create and show success message
        const successDiv = document.createElement('div');
        successDiv.className = 'success-message';
        successDiv.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #2ed573;
            color: white;
            padding: 15px 20px;
            border-radius: 5px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            z-index: 10000;
            max-width: 400px;
        `;
        successDiv.textContent = message;
        
        document.body.appendChild(successDiv);
        
        // Auto remove after 3 seconds
        setTimeout(() => {
            if (successDiv.parentNode) {
                successDiv.remove();
            }
        }, 3000);
    }

    updateAuthUI() {
        const token = localStorage.getItem('token');
        const authButtons = document.getElementById('authButtons');
        
        if (token && authButtons) {
            // User is logged in, show user menu
            authButtons.innerHTML = `
                <a href="dashboard.html" class="btn btn-outline">
                    <i class="fas fa-tachometer-alt"></i>
                    Dashboard
                </a>
                <button class="btn btn-primary" onclick="app.logout()">
                    <i class="fas fa-sign-out-alt"></i>
                    Logout
                </button>
            `;
        }
    }
}

// Global functions for HTML onclick events
function toggleView(view) {
    eventsManager.currentView = view;
    
    // Update button states
    document.querySelectorAll('.view-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    
    if (view === 'grid') {
        document.querySelector('.view-btn[onclick*="grid"]').classList.add('active');
    } else {
        document.querySelector('.view-btn[onclick*="list"]').classList.add('active');
    }
    
    eventsManager.displayEvents();
}

function toggleFilters() {
    const advancedFilters = document.getElementById('advancedFilters');
    advancedFilters.classList.toggle('active');
}

// Initialize events manager when page loads
let eventsManager;
document.addEventListener('DOMContentLoaded', () => {
    eventsManager = new EventsManager();
});

// Add modal styles
const modalStyles = `
    .modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 10000;
        padding: 2rem;
    }

    .modal-content {
        background: white;
        border-radius: 16px;
        max-width: 800px;
        width: 100%;
        max-height: 90vh;
        overflow-y: auto;
        box-shadow: 0 20px 25px rgba(0,0,0,0.25);
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 2rem 2rem 1rem;
        border-bottom: 1px solid #e2e8f0;
    }

    .modal-header h2 {
        margin: 0;
        color: #1a202c;
    }

    .modal-close {
        background: none;
        border: none;
        font-size: 1.5rem;
        cursor: pointer;
        color: #718096;
        padding: 0.5rem;
        border-radius: 50%;
        transition: all 0.3s ease;
    }

    .modal-close:hover {
        background: #f7fafc;
        color: #1a202c;
    }

    .modal-body {
        padding: 2rem;
    }

    .event-details-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
    }

    .event-info .info-item {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        margin-bottom: 1rem;
        padding: 0.75rem;
        background: #f7fafc;
        border-radius: 8px;
    }

    .event-info .info-item i {
        color: #667eea;
        width: 20px;
    }

    .event-info .info-item.verified {
        background: #f0fff4;
        color: #38a169;
    }

    .modal-footer {
        display: flex;
        justify-content: flex-end;
        gap: 1rem;
        padding: 1rem 2rem 2rem;
        border-top: 1px solid #e2e8f0;
    }

    .notification {
        position: fixed;
        top: 100px;
        right: 2rem;
        background: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        box-shadow: 0 10px 15px rgba(0,0,0,0.1);
        display: flex;
        align-items: center;
        gap: 0.75rem;
        z-index: 10001;
        border-left: 4px solid #667eea;
        animation: slideIn 0.3s ease;
    }

    .notification-success {
        border-left-color: #48bb78;
        color: #38a169;
    }

    .notification-error {
        border-left-color: #f56565;
        color: #e53e3e;
    }

    .notification-warning {
        border-left-color: #ed8936;
        color: #dd6b20;
    }

    .notification button {
        background: none;
        border: none;
        cursor: pointer;
        color: #718096;
        padding: 0.25rem;
    }

    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }

    @media (max-width: 768px) {
        .event-details-grid {
            grid-template-columns: 1fr;
        }
        
        .modal {
            padding: 1rem;
        }
        
        .modal-content {
            max-height: 95vh;
        }
    }
`;

// Inject modal styles
const styleSheet = document.createElement('style');
styleSheet.textContent = modalStyles;
document.head.appendChild(styleSheet);
