/**
 * Ghana National Event Management Platform
 * Main JavaScript Application
 */

class GhanaEventsApp {
    constructor() {
        this.apiBase = '/GHTouristApp/api';
        this.token = localStorage.getItem('auth_token');
        this.user = null;
        this.init();
    }

    async init() {
        // Check if user is logged in
        if (this.token) {
            await this.verifyToken();
        }

        // Initialize event listeners
        this.initEventListeners();
        
        // Load initial data
        this.loadEventCategories();
        this.animateStats();
    }

    initEventListeners() {
        // Mobile menu toggle
        const mobileToggle = document.querySelector('.mobile-menu-toggle');
        if (mobileToggle) {
            mobileToggle.addEventListener('click', this.toggleMobileMenu);
        }

        // Smooth scrolling for navigation links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Intersection Observer for animations
        this.initScrollAnimations();
    }

    initScrollAnimations() {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('fade-in-up');
                }
            });
        }, { threshold: 0.1 });

        document.querySelectorAll('.feature-card').forEach(card => {
            observer.observe(card);
        });
    }

    toggleMobileMenu() {
        const navMenu = document.querySelector('.nav-menu');
        navMenu.classList.toggle('active');
    }

    // Authentication Methods
    async showLoginModal() {
        const modalHtml = `
            <div class="modal-overlay" onclick="this.parentElement.style.display='none'">
                <div class="modal-content" onclick="event.stopPropagation()">
                    <div class="modal-header">
                        <h2><i class="fas fa-sign-in-alt"></i> Login to Ghana Events</h2>
                        <button class="modal-close" onclick="this.closest('.modal').style.display='none'">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <form id="loginForm" class="auth-form">
                        <div class="form-group">
                            <label for="loginEmail">Username or Email</label>
                            <input type="text" id="loginEmail" name="email" required>
                        </div>
                        <div class="form-group">
                            <label for="loginPassword">Password</label>
                            <input type="password" id="loginPassword" name="password" required>
                        </div>
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="remember_me">
                                <span class="checkmark"></span>
                                Remember me
                            </label>
                        </div>
                        <button type="submit" class="btn btn-primary btn-full">
                            <span class="btn-text">Login</span>
                            <span class="btn-loading" style="display: none;">
                                <div class="loading"></div>
                            </span>
                        </button>
                    </form>
                    <div class="auth-links">
                        <a href="#" onclick="showForgotPasswordModal()">Forgot Password?</a>
                        <span>Don't have an account? <a href="#" onclick="showRegisterModal()">Register here</a></span>
                    </div>
                </div>
            </div>
        `;

        this.showModal('loginModal', modalHtml);
        
        // Add form submit handler
        document.getElementById('loginForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleLogin(e.target);
        });
    }

    async showRegisterModal() {
        const modalHtml = `
            <div class="modal-overlay" onclick="this.parentElement.style.display='none'">
                <div class="modal-content" onclick="event.stopPropagation()">
                    <div class="modal-header">
                        <h2><i class="fas fa-user-plus"></i> Join Ghana Events</h2>
                        <button class="modal-close" onclick="this.closest('.modal').style.display='none'">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <form id="registerForm" class="auth-form">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="registerFullName">Full Name</label>
                                <input type="text" id="registerFullName" name="full_name" required>
                            </div>
                            <div class="form-group">
                                <label for="registerUsername">Username</label>
                                <input type="text" id="registerUsername" name="username" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label for="registerEmail">Email Address</label>
                            <input type="email" id="registerEmail" name="email" required>
                        </div>
                        <div class="form-group">
                            <label for="registerPhone">Phone Number</label>
                            <input type="tel" id="registerPhone" name="phone" placeholder="+233...">
                        </div>
                        <div class="form-group">
                            <label for="registerRole">Account Type</label>
                            <select id="registerRole" name="role_type" required>
                                <option value="">Select Account Type</option>
                                <option value="tourist">Tourist/Attendee</option>
                                <option value="event_organizer">Event Organizer</option>
                                <option value="venue_owner">Venue Owner</option>
                                <option value="vendor">Service Provider/Vendor</option>
                            </select>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label for="registerPassword">Password</label>
                                <input type="password" id="registerPassword" name="password" required>
                            </div>
                            <div class="form-group">
                                <label for="registerConfirmPassword">Confirm Password</label>
                                <input type="password" id="registerConfirmPassword" name="confirm_password" required>
                            </div>
                        </div>
                        <div class="form-group">
                            <label class="checkbox-label">
                                <input type="checkbox" name="terms" required>
                                <span class="checkmark"></span>
                                I agree to the <a href="#" target="_blank">Terms of Service</a> and <a href="#" target="_blank">Privacy Policy</a>
                            </label>
                        </div>
                        <button type="submit" class="btn btn-primary btn-full">
                            <span class="btn-text">Create Account</span>
                            <span class="btn-loading" style="display: none;">
                                <div class="loading"></div>
                            </span>
                        </button>
                    </form>
                    <div class="auth-links">
                        <span>Already have an account? <a href="#" onclick="showLoginModal()">Login here</a></span>
                    </div>
                </div>
            </div>
        `;

        this.showModal('registerModal', modalHtml);
        
        // Add form submit handler
        document.getElementById('registerForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleRegister(e.target);
        });
    }

    showModal(modalId, content) {
        let modal = document.getElementById(modalId);
        if (!modal) {
            modal = document.createElement('div');
            modal.id = modalId;
            modal.className = 'modal';
            document.body.appendChild(modal);
        }
        
        modal.innerHTML = content;
        modal.style.display = 'flex';
        
        // Add modal styles if not already added
        if (!document.getElementById('modal-styles')) {
            const styles = document.createElement('style');
            styles.id = 'modal-styles';
            styles.textContent = `
                .modal {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.8);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 10000;
                    backdrop-filter: blur(5px);
                }
                
                .modal-overlay {
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                }
                
                .modal-content {
                    background: white;
                    border-radius: 16px;
                    max-width: 500px;
                    width: 90%;
                    max-height: 90vh;
                    overflow-y: auto;
                    position: relative;
                    box-shadow: 0 25px 50px rgba(0,0,0,0.25);
                }
                
                .modal-header {
                    padding: 2rem 2rem 1rem;
                    border-bottom: 1px solid #e2e8f0;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                
                .modal-header h2 {
                    margin: 0;
                    color: #1a202c;
                    font-size: 1.5rem;
                }
                
                .modal-close {
                    background: none;
                    border: none;
                    font-size: 1.5rem;
                    cursor: pointer;
                    color: #718096;
                    padding: 0.5rem;
                    border-radius: 8px;
                    transition: all 0.3s ease;
                }
                
                .modal-close:hover {
                    background: #f7fafc;
                    color: #1a202c;
                }
                
                .auth-form {
                    padding: 2rem;
                }
                
                .form-group {
                    margin-bottom: 1.5rem;
                }
                
                .form-row {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 1rem;
                }
                
                .form-group label {
                    display: block;
                    margin-bottom: 0.5rem;
                    font-weight: 600;
                    color: #1a202c;
                }
                
                .form-group input,
                .form-group select {
                    width: 100%;
                    padding: 0.75rem;
                    border: 2px solid #e2e8f0;
                    border-radius: 8px;
                    font-size: 1rem;
                    transition: border-color 0.3s ease;
                }
                
                .form-group input:focus,
                .form-group select:focus {
                    outline: none;
                    border-color: #667eea;
                    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
                }
                
                .form-group input.error,
                .form-group select.error {
                    border-color: #e53e3e;
                    box-shadow: 0 0 0 3px rgba(229, 62, 62, 0.1);
                }
                
                .field-error {
                    color: #e53e3e;
                    font-size: 0.875rem;
                    margin-top: 0.5rem;
                    display: flex;
                    align-items: center;
                    gap: 0.25rem;
                }
                
                .field-error::before {
                    content: "⚠";
                    font-size: 1rem;
                }
                
                .checkbox-label {
                    display: flex;
                    align-items: center;
                    cursor: pointer;
                    font-weight: 400 !important;
                }
                
                .checkbox-label input[type="checkbox"] {
                    margin-right: 0.5rem;
                    width: auto;
                }
                
                .btn-full {
                    width: 100%;
                    justify-content: center;
                }
                
                .auth-links {
                    padding: 0 2rem 2rem;
                    text-align: center;
                    color: #718096;
                }
                
                .auth-links a {
                    color: #667eea;
                    text-decoration: none;
                    font-weight: 600;
                }
                
                .auth-links a:hover {
                    text-decoration: underline;
                }
                
                @media (max-width: 768px) {
                    .form-row {
                        grid-template-columns: 1fr;
                    }
                }
            `;
            document.head.appendChild(styles);
        }
    }

    async handleLogin(form) {
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);
        
        this.setButtonLoading(form.querySelector('button[type="submit"]'), true);
        
        try {
            const response = await fetch(`${this.apiBase}/auth.php/login`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.token = result.token;
                this.user = result.user;
                localStorage.setItem('auth_token', this.token);
                localStorage.setItem('user_data', JSON.stringify(this.user));
                
                console.log('Login successful, user data:', this.user);
                console.log('Checking role status:', this.user.role_type, this.user.username);
                console.log('Full result object:', result);
                
                this.showNotification('Login successful! Welcome back.', 'success');
                document.getElementById('loginModal').style.display = 'none';
                
                // Check user role and redirect to appropriate dashboard
                if (this.user.role_type === 'admin' || this.user.username === 'admin') {
                    console.log('Redirecting to admin dashboard...');
                    window.location.href = 'admin-dashboard.html';
                } else if (this.user.role_type === 'event_organizer') {
                    console.log('Redirecting to organizer dashboard...');
                    window.location.href = 'organizer-dashboard.html';
                } else if (this.user.role_type === 'tourist' || this.user.role_type === 'attendee') {
                    console.log('Redirecting to tourist dashboard...');
                    window.location.href = 'tourist-dashboard.html';
                } else {
                    console.log('Regular user, updating UI...');
                    this.updateAuthUI();
                }
            } else {
                this.showNotification(result.error || result.message || 'Login failed', 'error');
            }
        } catch (error) {
            this.showNotification('Network error. Please try again.', 'error');
        } finally {
            this.setButtonLoading(form.querySelector('button[type="submit"]'), false);
        }
    }

    async handleRegister(form) {
        const formData = new FormData(form);
        const data = Object.fromEntries(formData.entries());
        
        // Client-side validation
        if (data.password !== data.confirm_password) {
            this.showNotification('Passwords do not match', 'error');
            return;
        }
        
        if (data.password.length < 6) {
            this.showNotification('Password must be at least 6 characters long', 'error');
            return;
        }
        
        // Clear any previous field errors
        this.clearFieldErrors(form);
        
        // Map form fields to API expected format
        const registrationData = {
            full_name: data.full_name,
            username: data.username,
            email: data.email,
            phone: data.phone,
            password: data.password,
            role_type: data.role_type
        };
        
        this.setButtonLoading(form.querySelector('button[type="submit"]'), true);
        
        try {
            const response = await fetch(`${this.apiBase}/auth.php/register`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(registrationData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification(result.message || 'Registration successful! Please check your email for verification.', 'success');
                document.getElementById('registerModal').style.display = 'none';
                this.showLoginModal();
            } else {
                // Handle specific field errors
                if (result.field) {
                    this.showFieldError(form, result.field, result.error);
                } else {
                    this.showNotification(result.error || 'Registration failed', 'error');
                }
            }
        } catch (error) {
            console.error('Registration error:', error);
            
            // Check if this might be a successful registration that failed to return properly
            if (error.message.includes('JSON') || error.message.includes('Unexpected')) {
                this.showNotification(
                    'Registration may have been successful but we couldn\'t confirm it. Please try logging in, or contact support if you continue having issues.',
                    'warning'
                );
            } else {
                this.showNotification('Network error occurred. If you see a "username exists" error when trying again, your account may have been created successfully. Please try logging in.', 'error');
            }
        } finally {
            this.setButtonLoading(form.querySelector('button[type="submit"]'), false);
        }
    }

    clearFieldErrors(form) {
        const errorElements = form.querySelectorAll('.field-error');
        errorElements.forEach(el => el.remove());
        
        const inputElements = form.querySelectorAll('.error');
        inputElements.forEach(el => el.classList.remove('error'));
    }

    showFieldError(form, fieldName, message) {
        const field = form.querySelector(`[name="${fieldName}"]`);
        if (field) {
            field.classList.add('error');
            
            // Remove existing error message
            const existingError = field.parentNode.querySelector('.field-error');
            if (existingError) {
                existingError.remove();
            }
            
            // Add new error message
            const errorDiv = document.createElement('div');
            errorDiv.className = 'field-error';
            errorDiv.textContent = message;
            field.parentNode.appendChild(errorDiv);
        }
        
        // Also show as notification
        this.showNotification(message, 'error');
    }

    async verifyToken() {
        try {
            const response = await fetch(`${this.apiBase}/auth.php/verify-token`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.user = result.user;
                this.updateAuthUI();
            } else {
                this.logout();
            }
        } catch (error) {
            this.logout();
        }
    }

    logout() {
        this.token = null;
        this.user = null;
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        this.updateAuthUI();
        this.showNotification('Logged out successfully', 'info');
    }

    updateAuthUI() {
        const authButtons = document.querySelector('.auth-buttons');
        
        if (this.user) {
            authButtons.innerHTML = `
                <div class="user-menu">
                    <button class="btn btn-outline user-menu-toggle">
                        <i class="fas fa-user"></i>
                        ${this.user.full_name}
                        <i class="fas fa-chevron-down"></i>
                    </button>
                    <div class="user-dropdown">
                        <a href="#" onclick="app.showProfile()"><i class="fas fa-user"></i> Profile</a>
                        <a href="#" onclick="app.showDashboard()"><i class="fas fa-tachometer-alt"></i> Dashboard</a>
                        <a href="#" onclick="app.logout()"><i class="fas fa-sign-out-alt"></i> Logout</a>
                    </div>
                </div>
            `;
        } else {
            authButtons.innerHTML = `
                <a href="#" class="btn btn-outline" onclick="app.showLoginModal()">
                    <i class="fas fa-sign-in-alt"></i>
                    Login
                </a>
                <a href="#" class="btn btn-primary" onclick="app.showRegisterModal()">
                    <i class="fas fa-user-plus"></i>
                    Register
                </a>
            `;
        }
    }

    // Event Management Methods
    async loadEventCategories() {
        try {
            const response = await fetch(`${this.apiBase}/events.php/categories`);
            const result = await response.json();
            
            if (result.success) {
                this.eventCategories = result.data;
            }
        } catch (error) {
            console.error('Failed to load event categories:', error);
        }
    }

    async exploreEvents() {
        // Redirect to events page or show events modal
        window.location.href = 'events.html';
    }

    // Utility Methods
    setButtonLoading(button, loading) {
        const text = button.querySelector('.btn-text');
        const loader = button.querySelector('.btn-loading');
        
        if (loading) {
            text.style.display = 'none';
            loader.style.display = 'inline-flex';
            button.disabled = true;
        } else {
            text.style.display = 'inline';
            loader.style.display = 'none';
            button.disabled = false;
        }
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${this.getNotificationIcon(type)}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;
        
        // Add notification styles if not already added
        if (!document.getElementById('notification-styles')) {
            const styles = document.createElement('style');
            styles.id = 'notification-styles';
            styles.textContent = `
                .notification {
                    position: fixed;
                    top: 100px;
                    right: 2rem;
                    background: white;
                    padding: 1rem 1.5rem;
                    border-radius: 8px;
                    box-shadow: 0 10px 15px rgba(0,0,0,0.1);
                    display: flex;
                    align-items: center;
                    gap: 0.75rem;
                    z-index: 10001;
                    border-left: 4px solid #667eea;
                    animation: slideIn 0.3s ease;
                    max-width: 400px;
                }
                
                .notification-success {
                    border-left-color: #48bb78;
                    color: #38a169;
                }
                
                .notification-error {
                    border-left-color: #f56565;
                    color: #e53e3e;
                }
                
                .notification-warning {
                    border-left-color: #ed8936;
                    color: #dd6b20;
                }
                
                @keyframes slideIn {
                    from {
                        transform: translateX(100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
                
                @media (max-width: 768px) {
                    .notification {
                        right: 1rem;
                        left: 1rem;
                        max-width: none;
                    }
                }
            `;
            document.head.appendChild(styles);
        }
        
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 5000);
    }

    getNotificationIcon(type) {
        const icons = {
            success: 'check-circle',
            error: 'exclamation-circle',
            warning: 'exclamation-triangle',
            info: 'info-circle'
        };
        return icons[type] || 'info-circle';
    }

    animateStats() {
        const stats = document.querySelectorAll('.stat-item h3');
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const target = entry.target;
                    const text = target.textContent;
                    const number = parseInt(text.replace(/[^\d]/g, ''));
                    
                    if (number) {
                        this.animateNumber(target, 0, number, 2000);
                    }
                    
                    observer.unobserve(target);
                }
            });
        });
        
        stats.forEach(stat => observer.observe(stat));
    }

    animateNumber(element, start, end, duration) {
        const startTime = performance.now();
        const suffix = element.textContent.replace(/[\d,]/g, '');
        
        const animate = (currentTime) => {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            const current = Math.floor(start + (end - start) * progress);
            element.textContent = current.toLocaleString() + suffix;
            
            if (progress < 1) {
                requestAnimationFrame(animate);
            }
        };
        
        requestAnimationFrame(animate);
    }
}

// Global functions for onclick handlers
function showLoginModal() {
    app.showLoginModal();
}

function showRegisterModal() {
    app.showRegisterModal();
}

function exploreEvents() {
    app.exploreEvents();
}

// Initialize the app
const app = new GhanaEventsApp();

// Export for use in other files
window.GhanaEventsApp = app;
