// Organizer Dashboard JavaScript
class OrganizerDashboard {
    constructor() {
        this.token = localStorage.getItem('auth_token');
        this.user = JSON.parse(localStorage.getItem('user_data') || '{}');
        this.init();
    }

    async init() {
        if (!this.token || !this.user.id) {
            window.location.href = 'index.html';
            return;
        }

        this.updateUserInfo();
        await this.loadDashboardData();
        await this.loadCategories();
        await this.loadVenues();
        this.setupEventListeners();
    }

    updateUserInfo() {
        const userName = document.getElementById('userName');
        const welcomeName = document.getElementById('welcomeName');
        const userAvatar = document.getElementById('userAvatar');

        if (this.user.full_name) {
            userName.textContent = this.user.full_name;
            welcomeName.textContent = this.user.full_name;
            
            // Create initials for avatar
            const initials = this.user.full_name
                .split(' ')
                .map(name => name.charAt(0))
                .join('')
                .toUpperCase()
                .substring(0, 2);
            userAvatar.textContent = initials;
        }
    }

    async loadDashboardData() {
        try {
            // Load statistics
            await this.loadStats();
            
            // Load recent events
            await this.loadRecentEvents();
            
            // Load notifications
            await this.loadNotifications();
        } catch (error) {
            console.error('Error loading dashboard data:', error);
            this.showNotification('Error loading dashboard data', 'error');
        }
    }

    async loadStats() {
        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=organizer-stats', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    document.getElementById('totalEvents').textContent = data.data.total_events || 0;
                    document.getElementById('totalTickets').textContent = data.data.total_tickets || 0;
                    document.getElementById('totalRevenue').textContent = `₵${data.data.total_revenue || 0}`;
                    document.getElementById('pendingApprovals').textContent = data.data.pending_approvals || 0;
                }
            }
        } catch (error) {
            console.error('Error loading stats:', error);
        }
    }

    async loadRecentEvents() {
        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=organizer-events', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            const eventList = document.getElementById('eventList');

            if (response.ok) {
                const data = await response.json();
                if (data.success && data.data.length > 0) {
                    eventList.innerHTML = data.data.map(event => `
                        <div class="event-item">
                            <div class="event-info">
                                <h4>${this.escapeHtml(event.title)}</h4>
                                <div class="event-meta">
                                    <i class="fas fa-calendar"></i> ${this.formatDate(event.start_datetime)}
                                    <br>
                                    <i class="fas fa-map-marker-alt"></i> ${this.escapeHtml(event.venue_name || 'TBD')}
                                </div>
                            </div>
                            <div class="status-badge status-${event.approval_status}">
                                ${this.formatStatus(event.approval_status)}
                            </div>
                        </div>
                    `).join('');
                } else {
                    eventList.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-calendar-plus"></i>
                            <p>No events yet. Create your first event!</p>
                        </div>
                    `;
                }
            } else {
                throw new Error('Failed to load events');
            }
        } catch (error) {
            console.error('Error loading events:', error);
            document.getElementById('eventList').innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-exclamation-triangle"></i>
                    <p>Error loading events</p>
                </div>
            `;
        }
    }

    async loadNotifications() {
        try {
            const response = await fetch('/GHTouristApp/api/notifications.php?action=list', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            const notificationList = document.getElementById('notificationList');

            if (response.ok) {
                const data = await response.json();
                if (data.success && data.data.length > 0) {
                    notificationList.innerHTML = data.data.map(notification => `
                        <div class="notification-item ${notification.read_at ? '' : 'unread'}">
                            <div>
                                <strong>${this.escapeHtml(notification.title)}</strong>
                                <p>${this.escapeHtml(notification.message)}</p>
                                <small>${this.formatDate(notification.created_at)}</small>
                            </div>
                        </div>
                    `).join('');
                } else {
                    notificationList.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-bell-slash"></i>
                            <p>No notifications</p>
                        </div>
                    `;
                }
            } else {
                throw new Error('Failed to load notifications');
            }
        } catch (error) {
            console.error('Error loading notifications:', error);
            document.getElementById('notificationList').innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-exclamation-triangle"></i>
                    <p>Error loading notifications</p>
                </div>
            `;
        }
    }

    async loadCategories() {
        try {
            console.log('Loading categories...');
            const response = await fetch('/GHTouristApp/api/events.php?action=categories');
            
            if (response.ok) {
                const data = await response.json();
                console.log('Categories API Response:', data);
                if (data.success && data.data) {
                    const categorySelect = document.getElementById('eventCategory');
                    if (categorySelect) {
                        categorySelect.innerHTML = '<option value="">Select Category</option>' +
                            data.data.map(category => 
                                `<option value="${category.id}">${this.escapeHtml(category.name)}</option>`
                            ).join('');
                        console.log('Categories loaded successfully');
                    } else {
                        console.error('Category select element not found');
                    }
                } else {
                    console.error('Failed to load categories:', data);
                }
            } else {
                console.error('Categories API response not OK:', response.status);
            }
        } catch (error) {
            console.error('Error loading categories:', error);
        }
    }

    async loadVenues() {
        try {
            console.log('Loading venues...');
            const response = await fetch('/GHTouristApp/api/venues.php?action=list');
            
            if (response.ok) {
                const data = await response.json();
                console.log('Venues API Response:', data);
                if (data.success && data.venues) {
                    const venueSelect = document.getElementById('eventVenue');
                    if (venueSelect) {
                        venueSelect.innerHTML = '<option value="">Select Venue</option>' +
                            data.venues.map(venue => 
                                `<option value="${venue.id}">${this.escapeHtml(venue.name)} - ${this.escapeHtml(venue.city || venue.address)}</option>`
                            ).join('') +
                            '<option value="create_new">+ Create New Venue</option>';
                        console.log('Venues loaded successfully');
                    } else {
                        console.error('Venue select element not found');
                    }
                } else {
                    console.error('Failed to load venues:', data);
                    // Try to load venues anyway with empty dropdown + create option
                    const venueSelect = document.getElementById('eventVenue');
                    if (venueSelect) {
                        venueSelect.innerHTML = '<option value="">Select Venue</option><option value="create_new">+ Create New Venue</option>';
                    }
                }
            } else {
                console.error('Venues API response not OK:', response.status);
                // Fallback to create new venue option
                const venueSelect = document.getElementById('eventVenue');
                if (venueSelect) {
                    venueSelect.innerHTML = '<option value="">Select Venue</option><option value="create_new">+ Create New Venue</option>';
                }
            }
        } catch (error) {
            console.error('Error loading venues:', error);
            // Fallback to create new venue option
            const venueSelect = document.getElementById('eventVenue');
            if (venueSelect) {
                venueSelect.innerHTML = '<option value="">Select Venue</option><option value="create_new">+ Create New Venue</option>';
            }
        }
    }

    setupEventListeners() {
        // Venue selection handler
        const venueSelect = document.getElementById('eventVenue');
        if (venueSelect) {
            venueSelect.addEventListener('change', (e) => {
                if (e.target.value === 'create_new') {
                    this.showVenueCreationModal();
                }
            });
        }

        // Create Event Form
        const createEventForm = document.getElementById('createEventForm');
        createEventForm.addEventListener('submit', (e) => this.handleCreateEvent(e));

        // Modal close events
        window.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal')) {
                e.target.style.display = 'none';
            }
        });
    }

    showVenueCreationModal() {
        // Create venue creation modal HTML
        const modalHtml = `
            <div id="venueCreationModal" class="modal" style="display: block;">
                <div class="modal-content">
                    <div class="modal-header">
                        <h3>Create New Venue</h3>
                        <span class="close" onclick="this.parentElement.parentElement.parentElement.style.display='none'">&times;</span>
                    </div>
                    <form id="createVenueForm">
                        <div class="form-group">
                            <label for="venueName">Venue Name *</label>
                            <input type="text" id="venueName" name="name" required>
                        </div>
                        <div class="form-group">
                            <label for="venueType">Venue Type *</label>
                            <select id="venueType" name="type" required>
                                <option value="">Select Type</option>
                                <option value="event_center">Event Center</option>
                                <option value="hotel">Hotel</option>
                                <option value="restaurant">Restaurant</option>
                                <option value="outdoor_space">Outdoor Space</option>
                                <option value="cultural_site">Cultural Site</option>
                                <option value="sports_facility">Sports Facility</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="venueAddress">Address *</label>
                            <textarea id="venueAddress" name="address" required></textarea>
                        </div>
                        <div class="form-group">
                            <label for="venueCity">City *</label>
                            <input type="text" id="venueCity" name="city" required>
                        </div>
                        <div class="form-group">
                            <label for="venueRegion">Region</label>
                            <input type="text" id="venueRegion" name="region">
                        </div>
                        <div class="form-group">
                            <label for="venueCapacity">Capacity *</label>
                            <input type="number" id="venueCapacity" name="capacity" required>
                        </div>
                        <div class="form-group">
                            <label for="venuePhone">Contact Phone *</label>
                            <input type="tel" id="venuePhone" name="contact_phone" required>
                        </div>
                        <div class="form-group">
                            <label for="venueEmail">Contact Email *</label>
                            <input type="email" id="venueEmail" name="contact_email" required>
                        </div>
                        <div class="form-group">
                            <label for="venueDescription">Description</label>
                            <textarea id="venueDescription" name="description"></textarea>
                        </div>
                        <div class="form-actions">
                            <button type="button" onclick="this.closest('.modal').style.display='none'">Cancel</button>
                            <button type="submit">Create Venue</button>
                        </div>
                    </form>
                </div>
            </div>
        `;
        
        // Remove existing modal if any
        const existingModal = document.getElementById('venueCreationModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Add form submit handler
        const form = document.getElementById('createVenueForm');
        form.addEventListener('submit', (e) => this.handleCreateVenue(e));
    }

    async handleCreateVenue(e) {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const venueData = Object.fromEntries(formData.entries());
        
        try {
            const response = await fetch('/GHTouristApp/api/venues.php?endpoint=register', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(venueData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Close modal
                document.getElementById('venueCreationModal').style.display = 'none';
                
                // Reload venues
                await this.loadVenues();
                
                // Select the new venue if venue_id is returned
                if (result.venue_id) {
                    const venueSelect = document.getElementById('eventVenue');
                    venueSelect.value = result.venue_id;
                }
                
                this.showNotification('Venue created successfully!', 'success');
            } else {
                this.showNotification(result.error || 'Failed to create venue', 'error');
            }
        } catch (error) {
            console.error('Error creating venue:', error);
            this.showNotification('Error creating venue', 'error');
        }
    }

    async handleCreateEvent(e) {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const eventData = {
            title: formData.get('title'),
            description: formData.get('description'),
            category_id: formData.get('category_id'),
            venue_id: formData.get('venue_id'),
            start_datetime: formData.get('start_datetime'),
            end_datetime: formData.get('end_datetime'),
            max_attendees: formData.get('max_attendees') || null,
            ticket_price: formData.get('ticket_price') || 0,
            organizer_id: this.user.id
        };

        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=create', {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(eventData)
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.showNotification('Event created successfully!', 'success');
                    this.closeModal('createEventModal');
                    document.getElementById('createEventForm').reset();
                    await this.loadDashboardData();
                } else {
                    throw new Error(data.message || 'Failed to create event');
                }
            } else {
                throw new Error('Failed to create event');
            }
        } catch (error) {
            console.error('Error creating event:', error);
            this.showNotification('Error creating event: ' + error.message, 'error');
        }
    }

    async loadAnalytics() {
        try {
            const response = await fetch('/GHTouristApp/api/tickets.php?action=sales-report', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.displayAnalytics(data.data);
                }
            }
        } catch (error) {
            console.error('Error loading analytics:', error);
        }
    }

    async loadTicketSales() {
        try {
            const response = await fetch('/GHTouristApp/api/tickets.php?action=organizer-tickets&organizer_id=6', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success && data.data) {
                    this.displayTicketSales(data.data);
                } else {
                    document.getElementById('ticketSalesContent').innerHTML = '<p>No ticket sales data available.</p>';
                }
            } else {
                console.error('Failed to load ticket sales');
                document.getElementById('ticketSalesContent').innerHTML = '<p>Error loading ticket sales.</p>';
            }
        } catch (error) {
            console.error('Error loading ticket sales:', error);
            document.getElementById('ticketSalesContent').innerHTML = '<p>Error loading ticket sales.</p>';
        }
    }

    displayAnalytics(analyticsData) {
        const analyticsContent = document.getElementById('analyticsContent');
        const summary = analyticsData.summary;
        const dailySales = analyticsData.daily_sales;

        analyticsContent.innerHTML = `
            <div class="analytics-summary">
                <div class="analytics-grid">
                    <div class="analytics-card">
                        <h4>Total Tickets</h4>
                        <div class="analytics-number">${summary.total_tickets || 0}</div>
                    </div>
                    <div class="analytics-card">
                        <h4>Sold Tickets</h4>
                        <div class="analytics-number">${summary.sold_tickets || 0}</div>
                    </div>
                    <div class="analytics-card">
                        <h4>Total Revenue</h4>
                        <div class="analytics-number">₵${summary.total_revenue || 0}</div>
                    </div>
                    <div class="analytics-card">
                        <h4>Avg Ticket Price</h4>
                        <div class="analytics-number">₵${summary.avg_ticket_price || 0}</div>
                    </div>
                </div>
            </div>
            <div class="analytics-chart">
                <h4>Daily Sales (Last 30 Days)</h4>
                <div class="chart-container">
                    ${dailySales.map(day => `
                        <div class="chart-bar">
                            <div class="bar-value">${day.tickets_sold}</div>
                            <div class="bar" style="height: ${Math.max(20, (day.tickets_sold / Math.max(...dailySales.map(d => d.tickets_sold))) * 100)}px"></div>
                            <div class="bar-label">${this.formatDate(day.sale_date)}</div>
                        </div>
                    `).join('')}
                </div>
            </div>
        `;
    }

    async loadVendorMarketplace() {
        try {
            const response = await fetch('/GHTouristApp/api/venues.php?endpoint=vendors', {
                headers: {
                    'Authorization': `Bearer ${this.token}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    this.displayVendorMarketplace(data.vendors || []);
                } else {
                    document.getElementById('vendorMarketplaceContent').innerHTML = '<p>No vendors available.</p>';
                }
            } else {
                console.error('Failed to load vendor marketplace');
                document.getElementById('vendorMarketplaceContent').innerHTML = '<p>Error loading vendor marketplace.</p>';
            }
        } catch (error) {
            console.error('Error loading vendor marketplace:', error);
            document.getElementById('vendorMarketplaceContent').innerHTML = '<p>Error loading vendor marketplace.</p>';
        }
    }

    displayVendorMarketplace(vendors) {
        const vendorContent = document.getElementById('vendorMarketplaceContent');
        
        if (vendors.length === 0) {
            vendorContent.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-store"></i>
                    <p>No vendors available at the moment</p>
                    <button class="btn-primary" onclick="this.registerAsVendor()">Register as Vendor</button>
                </div>
            `;
            return;
        }

        vendorContent.innerHTML = `
            <div class="vendor-grid">
                ${vendors.map(vendor => `
                    <div class="vendor-card">
                        <div class="vendor-header">
                            <h4>${this.escapeHtml(vendor.business_name)}</h4>
                            <span class="vendor-category">${this.escapeHtml(vendor.category)}</span>
                        </div>
                        <div class="vendor-info">
                            <p>${this.escapeHtml(vendor.description || 'No description available')}</p>
                            <div class="vendor-contact">
                                <i class="fas fa-phone"></i> ${this.escapeHtml(vendor.contact_phone)}
                                <br>
                                <i class="fas fa-envelope"></i> ${this.escapeHtml(vendor.contact_email)}
                            </div>
                        </div>
                        <div class="vendor-actions">
                            <button class="btn-primary" onclick="this.contactVendor('${vendor.id}')">
                                Contact Vendor
                            </button>
                        </div>
                    </div>
                `).join('')}
            </div>
        `;
    }

    openCreateEventModal() {
        document.getElementById('createEventModal').style.display = 'block';
        // Reload categories and venues when modal opens
        this.loadCategories();
        this.loadVenues();
    }

    openAnalyticsModal() {
        document.getElementById('analyticsModal').style.display = 'block';
        this.loadAnalytics();
    }

    openVendorModal() {
        document.getElementById('vendorModal').style.display = 'block';
        this.loadVendorMarketplace();
    }

    closeModal(modalId) {
        document.getElementById(modalId).style.display = 'none';
    }

    contactVendor(vendorId) {
        // Implement vendor contact functionality
        this.showNotification('Contact feature coming soon!', 'info');
    }

    registerAsVendor() {
        // Implement vendor registration
        this.showNotification('Vendor registration feature coming soon!', 'info');
    }

    showNotification(message, type = 'success') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            ${message}
            <button onclick="this.parentElement.remove()">&times;</button>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 5000);
    }

    escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.toString().replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('en-GB', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }

    formatStatus(status) {
        const statusMap = {
            'pending': 'Pending',
            'approved': 'Approved',
            'rejected': 'Rejected'
        };
        return statusMap[status] || status;
    }
}

// Initialize dashboard when page loads
document.addEventListener('DOMContentLoaded', function() {
    window.dashboard = new OrganizerDashboard();
});

// Sidebar Navigation Functions
function toggleSidebar() {
            const sidebar = document.getElementById('sidebar');
            const mainContent = document.getElementById('mainContentWrapper');
            
            if (window.innerWidth <= 768) {
                sidebar.classList.toggle('open');
            } else {
                sidebar.classList.toggle('collapsed');
                mainContent.classList.toggle('expanded');
            }
        }

        function showPage(pageId) {
            // Hide all page contents
            const pages = document.querySelectorAll('.page-content');
            pages.forEach(page => page.classList.remove('active'));
            
            // Show selected page
            const targetPage = document.getElementById(`${pageId}-page`);
            if (targetPage) {
                targetPage.classList.add('active');
            }
            
            // Update menu item active state
            const menuItems = document.querySelectorAll('.menu-item');
            menuItems.forEach(item => item.classList.remove('active'));
            
            // Find and activate the clicked menu item
            const activeItem = document.querySelector(`[onclick="showPage('${pageId}')"]`);
            if (activeItem) {
                activeItem.classList.add('active');
            }
            
            // Load page-specific content
            loadPageContent(pageId);
            
            // Close sidebar on mobile after selection
            if (window.innerWidth <= 768) {
                const sidebar = document.getElementById('sidebar');
                sidebar.classList.remove('open');
            }
        }

        function loadPageContent(pageId) {
            switch(pageId) {
                case 'dashboard':
                    loadDashboardData();
                    break;
                case 'events':
                    loadEventsPage();
                    break;
                case 'event-approval':
                    loadPendingEvents();
                    break;
                case 'tickets':
                    loadTicketSales();
                    break;
                case 'attendees':
                    loadAttendees();
                    break;
                case 'analytics':
                    loadAnalyticsPage();
                    break;
                case 'reports':
                    loadReports();
                    break;
                case 'chat-rooms':
                    loadChatRooms();
                    break;
                case 'notifications':
                    loadNotificationsPage();
                    break;
                case 'vendors':
                    loadVendorsPage();
                    break;
                case 'venues':
                    loadVenuesPage();
                    break;
            }
        }

        // Page Content Loading Functions
        function loadEventsPage() {
            const eventsGrid = document.getElementById('eventsGrid');
            eventsGrid.innerHTML = '<p>Loading events...</p>';
            
            fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                return response.json();
            })
            .then(data => {
                console.log('Events API Response:', data);
                if (data.success && data.data) {
                    displayEventsGrid(data.data, 'eventsGrid');
                } else if (data.error) {
                    eventsGrid.innerHTML = `<p>Error: ${data.error}</p>`;
                } else {
                    eventsGrid.innerHTML = '<p>No events found.</p>';
                }
            })
            .catch(error => {
                console.error('Error loading events:', error);
                eventsGrid.innerHTML = `<p>Error loading events: ${error.message}</p>`;
            });
        }

        function loadPendingEvents() {
            const pendingGrid = document.getElementById('pendingEventsGrid');
            pendingGrid.innerHTML = '<p>Loading pending events...</p>';
            
            fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6&status=pending', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.data) {
                    const pendingEvents = data.data.filter(event => event.approval_status === 'pending');
                    displayEventsGrid(pendingEvents, 'pendingEventsGrid');
                } else {
                    pendingGrid.innerHTML = '<p>No pending events found.</p>';
                }
            })
            .catch(error => {
                console.error('Error loading pending events:', error);
                pendingGrid.innerHTML = '<p>Error loading pending events.</p>';
            });
        }

        function loadTicketSales() {
            const ticketContent = document.getElementById('ticketSalesContent');
            ticketContent.innerHTML = '<p>Loading ticket sales...</p>';
            
            fetch('/api/tickets.php?action=organizer-tickets&organizer_id=6', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.tickets) {
                    displayTicketSales(data.tickets);
                } else {
                    ticketContent.innerHTML = '<p>No ticket sales data available.</p>';
                }
            })
            .catch(error => {
                console.error('Error loading ticket sales:', error);
                ticketContent.innerHTML = '<p>Error loading ticket sales.</p>';
            });
        }

        function loadAttendees() {
            const attendeesContent = document.getElementById('attendeesContent');
            attendeesContent.innerHTML = '<p>Loading attendees...</p>';
            
            // This would integrate with a future attendees API
            setTimeout(() => {
                attendeesContent.innerHTML = '<p>Attendees management coming soon...</p>';
            }, 1000);
        }

        function loadAnalyticsPage() {
            const analyticsContent = document.getElementById('analyticsPageContent');
            analyticsContent.innerHTML = '<p>Loading analytics...</p>';
            
            loadAnalytics().then(analyticsData => {
                analyticsContent.innerHTML = analyticsData;
            });
        }

        function loadReports() {
            const reportsContent = document.getElementById('reportsContent');
            reportsContent.innerHTML = '<p>Loading reports...</p>';
            
            setTimeout(() => {
                reportsContent.innerHTML = '<p>Reports functionality coming soon...</p>';
            }, 1000);
        }

        function loadChatRooms() {
            const chatContent = document.getElementById('chatRoomsContent');
            chatContent.innerHTML = '<p>Loading chat rooms...</p>';
            
            setTimeout(() => {
                chatContent.innerHTML = '<p>Chat room management coming soon...</p>';
            }, 1000);
        }

        function loadNotificationsPage() {
            const notificationsContent = document.getElementById('notificationsPageContent');
            notificationsContent.innerHTML = '<p>Loading notifications...</p>';
            
            loadNotifications().then(notifications => {
                displayNotificationsPage(notifications);
            });
        }

        function loadVendorsPage() {
            const vendorsContent = document.getElementById('vendorsPageContent');
            vendorsContent.innerHTML = '<p>Loading vendors...</p>';
            
            loadVendorMarketplace().then(vendorData => {
                vendorsContent.innerHTML = vendorData;
            });
        }

        function loadVenuesPage() {
            const venuesContent = document.getElementById('venuesPageContent');
            venuesContent.innerHTML = '<p>Loading venues...</p>';
            
            fetch('/GHTouristApp/api/venues.php?action=list', {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token')}`,
                    'Content-Type': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && data.venues) {
                    displayVenuesPage(data.venues);
                } else {
                    venuesContent.innerHTML = '<p>No venues available.</p>';
                }
            })
            .catch(error => {
                console.error('Error loading venues:', error);
                venuesContent.innerHTML = '<p>Error loading venues.</p>';
            });
        }

        // Display Functions
        function displayEventsGrid(events, containerId) {
            const container = document.getElementById(containerId);
            if (!events || events.length === 0) {
                container.innerHTML = '<p>No events found.</p>';
                return;
            }
            
            const eventsHtml = events.map(event => `
                <div class="event-card">
                    <h3>${event.title}</h3>
                    <p><i class="fas fa-calendar"></i> ${new Date(event.start_datetime).toLocaleDateString()}</p>
                    <p><i class="fas fa-map-marker-alt"></i> ${event.venue_name || 'TBD'}</p>
                    <p><i class="fas fa-users"></i> ${event.max_attendees || 'Unlimited'} attendees</p>
                    <div class="event-status ${event.approval_status}">${event.approval_status}</div>
                </div>
            `).join('');
            
            container.innerHTML = eventsHtml;
        }

        function displayTicketSales(tickets) {
            const container = document.getElementById('ticketSalesContent');
            if (!tickets || tickets.length === 0) {
                container.innerHTML = '<p>No ticket sales data available.</p>';
                return;
            }
            
            const ticketsHtml = `
                <div class="tickets-overview">
                    <h3>Ticket Sales Overview</h3>
                    <div class="tickets-grid">
                        ${tickets.map(ticket => `
                            <div class="ticket-card">
                                <h4>${ticket.event_title}</h4>
                                <p>Price: ₵${ticket.price}</p>
                                <p>Sold: ${ticket.quantity_sold || 0}</p>
                                <p>Status: ${ticket.status}</p>
                            </div>
                        `).join('')}
                    </div>
                </div>
            `;
            
            container.innerHTML = ticketsHtml;
        }

        function displayNotificationsPage(notifications) {
            const container = document.getElementById('notificationsPageContent');
            if (!notifications || notifications.length === 0) {
                container.innerHTML = '<p>No notifications available.</p>';
                return;
            }
            
            const notificationsHtml = `
                <div class="notifications-list">
                    ${notifications.map(notification => `
                        <div class="notification-item ${notification.is_read ? 'read' : 'unread'}">
                            <div class="notification-icon">
                                <i class="fas fa-${notification.type === 'event' ? 'calendar' : 'bell'}"></i>
                            </div>
                            <div class="notification-content">
                                <h4>${notification.title}</h4>
                                <p>${notification.message}</p>
                                <small>${new Date(notification.created_at).toLocaleString()}</small>
                            </div>
                        </div>
                    `).join('')}
                </div>
            `;
            
            container.innerHTML = notificationsHtml;
        }

        function displayVenuesPage(venues) {
            const container = document.getElementById('venuesPageContent');
            if (!venues || venues.length === 0) {
                container.innerHTML = '<p>No venues available.</p>';
                return;
            }
            
            const venuesHtml = `
                <div class="venues-grid">
                    ${venues.map(venue => `
                        <div class="venue-card">
                            <h3>${venue.name}</h3>
                            <p><i class="fas fa-map-marker-alt"></i> ${venue.location}</p>
                            <p><i class="fas fa-users"></i> Capacity: ${venue.capacity}</p>
                            <p><i class="fas fa-dollar-sign"></i> ₵${venue.price_per_hour}/hour</p>
                            <div class="venue-status ${venue.verification_status}">${venue.verification_status}</div>
                            <button class="btn-primary" onclick="selectVenue(${venue.id})">Select Venue</button>
                        </div>
                    `).join('')}
                </div>
            `;
            
            container.innerHTML = venuesHtml;
        }

        function selectVenue(venueId) {
            showNotification('Venue selection functionality coming soon!', 'info');
        }

        // Utility functions
        function showNotification(message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 3000);
        }

function showPage(pageId) {
    // Hide all page contents
    const pages = document.querySelectorAll('.page-content');
    pages.forEach(page => page.classList.remove('active'));
    
    // Show selected page
    const targetPage = document.getElementById(`${pageId}-page`);
    if (targetPage) {
        targetPage.classList.add('active');
    }
    
    // Update menu item active state
    const menuItems = document.querySelectorAll('.menu-item');
    menuItems.forEach(item => item.classList.remove('active'));
    
    // Find and activate the clicked menu item
    const activeItem = document.querySelector(`[onclick="showPage('${pageId}')"]`);
    if (activeItem) {
        activeItem.classList.add('active');
    }
    
    // Load page-specific content
    loadPageContent(pageId);
    
    // Close sidebar on mobile after selection
    if (window.innerWidth <= 768) {
        const sidebar = document.getElementById('sidebar');
        sidebar.classList.remove('open');
    }
}

function loadPageContent(pageId) {
    switch(pageId) {
        case 'dashboard':
            if (window.dashboard) {
                window.dashboard.loadDashboardData();
            }
            break;
        case 'events':
            loadEventsPage();
            break;
        case 'event-approval':
            loadPendingEvents();
            break;
        case 'tickets':
            loadTicketSales();
            break;
        case 'attendees':
            loadAttendees();
            break;
        case 'chat-rooms':
            loadChatRooms();
            break;
        case 'notifications':
            loadNotifications();
            break;
        case 'analytics':
            loadAnalyticsPage();
            break;
        case 'reports':
            loadReports();
            break;
        case 'vendors':
            loadVendorsPage();
            break;
        case 'venues':
            loadVenuesPage();
            break;
    }
}

// Page Content Loading Functions
function loadEventsPage() {
    const eventsGrid = document.getElementById('eventsGrid');
    eventsGrid.innerHTML = '<p>Loading events...</p>';
    
    fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6', {
        headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            displayEventsGrid(data.data, 'eventsGrid');
        } else {
            eventsGrid.innerHTML = '<p>No events found.</p>';
        }
    })
    .catch(error => {
        console.error('Error loading events:', error);
        eventsGrid.innerHTML = '<p>Error loading events.</p>';
    });
}

function loadPendingEvents() {
    const pendingGrid = document.getElementById('pendingEventsGrid');
    pendingGrid.innerHTML = '<p>Loading pending events...</p>';
    
    fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6&status=pending', {
        headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.data) {
            const pendingEvents = data.data.filter(event => event.approval_status === 'pending');
            displayEventsGrid(pendingEvents, 'pendingEventsGrid');
        } else {
            pendingGrid.innerHTML = '<p>No pending events found.</p>';
        }
    })
    .catch(error => {
        console.error('Error loading pending events:', error);
        pendingGrid.innerHTML = '<p>Error loading pending events.</p>';
    });
}

function loadTicketSales() {
    const ticketContent = document.getElementById('ticketSalesContent');
    ticketContent.innerHTML = '<p>Loading ticket sales...</p>';
    
    fetch('/api/tickets.php?action=organizer-tickets&organizer_id=6', {
        headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.tickets) {
            displayTicketSales(data.tickets);
        } else {
            ticketContent.innerHTML = '<p>No ticket sales data available.</p>';
        }
    })
    .catch(error => {
        console.error('Error loading ticket sales:', error);
        ticketContent.innerHTML = '<p>Error loading ticket sales.</p>';
    });
}

function loadAttendees() {
    const attendeesContent = document.getElementById('attendeesContent');
    attendeesContent.innerHTML = '<p>Loading attendees...</p>';
    
    setTimeout(() => {
        attendeesContent.innerHTML = '<p>Attendees management coming soon...</p>';
    }, 1000);
}

function loadAnalyticsPage() {
    const analyticsContent = document.getElementById('analyticsPageContent');
    analyticsContent.innerHTML = '<p>Loading analytics...</p>';
    
    if (window.dashboard) {
        window.dashboard.loadAnalytics().then(analyticsData => {
            analyticsContent.innerHTML = analyticsData;
        });
    }
}

function loadReports() {
    const reportsContent = document.getElementById('reportsContent');
    reportsContent.innerHTML = '<p>Loading reports...</p>';
    
    setTimeout(() => {
        reportsContent.innerHTML = '<p>Reports functionality coming soon...</p>';
    }, 1000);
}

function loadChatRooms() {
    const chatContent = document.getElementById('chatRoomsContent');
    chatContent.innerHTML = '<p>Loading chat rooms...</p>';
    
    setTimeout(() => {
        chatContent.innerHTML = '<p>Chat room management coming soon...</p>';
    }, 1000);
}

function loadNotificationsPage() {
    const notificationsContent = document.getElementById('notificationsPageContent');
    notificationsContent.innerHTML = '<p>Loading notifications...</p>';
    
    if (window.dashboard) {
        window.dashboard.loadNotifications().then(notifications => {
            displayNotificationsPage(notifications);
        });
    }
}

function loadVendorsPage() {
    const vendorsContent = document.getElementById('vendorsPageContent');
    vendorsContent.innerHTML = '<p>Loading vendors...</p>';
    
    if (window.dashboard) {
        window.dashboard.loadVendorMarketplace().then(vendorData => {
            vendorsContent.innerHTML = vendorData;
        });
    }
}

function loadVenuesPage() {
    const venuesContent = document.getElementById('venuesPageContent');
    venuesContent.innerHTML = '<p>Loading venues...</p>';
    
    fetch('/GHTouristApp/api/venues.php?action=list', {
        headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success && data.venues) {
            displayVenuesPage(data.venues);
        } else {
            venuesContent.innerHTML = '<p>No venues available.</p>';
        }
    })
    .catch(error => {
        console.error('Error loading venues:', error);
        venuesContent.innerHTML = '<p>Error loading venues.</p>';
    });
}

// Display Functions
function displayEventsGrid(events, containerId) {
    const container = document.getElementById(containerId);
    if (!events || events.length === 0) {
        container.innerHTML = '<p>No events found.</p>';
        return;
    }
    
    const eventsHtml = events.map(event => `
        <div class="event-card">
            <h3>${event.title}</h3>
            <p><i class="fas fa-calendar"></i> ${new Date(event.start_datetime).toLocaleDateString()}</p>
            <p><i class="fas fa-map-marker-alt"></i> ${event.venue_name || 'TBD'}</p>
            <p><i class="fas fa-users"></i> ${event.max_attendees || 'Unlimited'} attendees</p>
            <div class="event-status ${event.approval_status}">${event.approval_status}</div>
        </div>
    `).join('');
    
    container.innerHTML = eventsHtml;
}

function displayTicketSales(tickets) {
    const container = document.getElementById('ticketSalesContent');
    if (!tickets || tickets.length === 0) {
        container.innerHTML = '<p>No ticket sales data available.</p>';
        return;
    }
    
    const ticketsHtml = `
        <div class="tickets-overview">
            <h3>Ticket Sales Overview</h3>
            <div class="tickets-grid">
                ${tickets.map(ticket => `
                    <div class="ticket-card">
                        <h4>${ticket.event_title}</h4>
                        <p>Price: ₵${ticket.price}</p>
                        <p>Sold: ${ticket.quantity_sold || 0}</p>
                        <p>Status: ${ticket.status}</p>
                    </div>
                `).join('')}
            </div>
        </div>
    `;
    
    container.innerHTML = ticketsHtml;
}

function displayNotificationsPage(notifications) {
    const container = document.getElementById('notificationsPageContent');
    if (!notifications || notifications.length === 0) {
        container.innerHTML = '<p>No notifications available.</p>';
        return;
    }
    
    const notificationsHtml = `
        <div class="notifications-list">
            ${notifications.map(notification => `
                <div class="notification-item ${notification.is_read ? 'read' : 'unread'}">
                    <div class="notification-icon">
                        <i class="fas fa-${notification.type === 'event' ? 'calendar' : 'bell'}"></i>
                    </div>
                    <div class="notification-content">
                        <h4>${notification.title}</h4>
                        <p>${notification.message}</p>
                        <small>${new Date(notification.created_at).toLocaleString()}</small>
                    </div>
                </div>
            `).join('')}
        </div>
    `;
    
    container.innerHTML = notificationsHtml;
}

function displayVenuesPage(venues) {
    const container = document.getElementById('venuesPageContent');
    if (!venues || venues.length === 0) {
        container.innerHTML = '<p>No venues available.</p>';
        return;
    }
    
    const venuesHtml = `
        <div class="venues-grid">
            ${venues.map(venue => `
                <div class="venue-card">
                    <h3>${venue.name}</h3>
                    <p><i class="fas fa-map-marker-alt"></i> ${venue.address || venue.city || 'Location TBD'}</p>
                    <p><i class="fas fa-users"></i> Capacity: ${venue.capacity}</p>
                    <p><i class="fas fa-tag"></i> Type: ${venue.type}</p>
                    <p><i class="fas fa-phone"></i> ${venue.contact_phone}</p>
                    <div class="venue-actions">
                        <button class="btn-primary" onclick="viewVenueDetails(${venue.id})">View Details</button>
                    </div>
                </div>
            `).join('')}
        </div>
    `;
    
    container.innerHTML = venuesHtml;
}

function selectVenue(venueId) {
    showNotification('Venue selection functionality coming soon!', 'info');
}

function formatStatus(status) {
    const statusMap = {
        'approved': 'Approved',
        'pending': 'Pending',
        'rejected': 'Rejected'
    };
    return statusMap[status] || status;
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <span>${message}</span>
        <button onclick="this.parentElement.remove()">&times;</button>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentElement) {
            notification.remove();
        }
    }, 5000);
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Global functions for HTML onclick events
function openCreateEventModal() {
    if (window.organizerDashboard) {
        window.organizerDashboard.openCreateEventModal();
    }
}

function openAnalyticsModal() {
    if (window.organizerDashboard) {
        window.organizerDashboard.openAnalyticsModal();
    }
}

function openVendorModal() {
    if (window.organizerDashboard) {
        window.organizerDashboard.openVendorModal();
    }
}

function closeModal(modalId) {
    if (window.organizerDashboard) {
        window.organizerDashboard.closeModal(modalId);
    }
}

function logout() {
    localStorage.removeItem('auth_token');
    localStorage.removeItem('user_data');
    window.location.href = 'index.html';
}

// Initialize dashboard when page loads
let dashboard;
document.addEventListener('DOMContentLoaded', () => {
    dashboard = new OrganizerDashboard();
    window.organizerDashboard = dashboard;
    
    // Chat Room Creation Form Handler
    const createChatRoomForm = document.getElementById('createChatRoomForm');
    if (createChatRoomForm) {
        createChatRoomForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const chatRoomData = {
                name: formData.get('name'),
                event_id: formData.get('event_id'),
                room_type: formData.get('room_type'),
                description: formData.get('description'),
                max_participants: parseInt(formData.get('max_participants')),
                moderated: document.getElementById('chatRoomModerated').checked ? 1 : 0,
                allow_media: document.getElementById('chatRoomAllowMedia').checked ? 1 : 0,
                organizer_id: 6 // Default organizer ID
            };
            
            try {
                const response = await fetch('/GHTouristApp/api/chat.php?endpoint=rooms&organizer_id=6', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(chatRoomData)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    document.getElementById('createChatRoomModal').style.display = 'none';
                    this.reset();
                    await loadChatRooms();
                    alert('Chat room created successfully!');
                } else {
                    alert('Error creating chat room: ' + (result.message || result.error || 'Unknown error'));
                }
            } catch (error) {
                console.error('Error creating chat room:', error);
                alert('Error creating chat room: ' + error.message);
            }
        });
    }
});

// Global functions for HTML onclick events
function openCreateEventModal() {
    dashboard.openCreateEventModal();
}

function openAnalyticsModal() {
    dashboard.openAnalyticsModal();
}

function openVendorModal() {
    dashboard.openVendorModal();
}

function closeModal(modalId) {
    dashboard.closeModal(modalId);
}

// Ticket Sales Functions
async function loadTicketSales() {
    try {
        const response = await fetch('/GHTouristApp/api/tickets.php?action=sales-report&organizer_id=6');
        const result = await response.json();
        
        if (result.success) {
            updateTicketSalesStats(result.data.summary);
            loadTicketSalesTable();
            loadEventFilters();
        }
    } catch (error) {
        console.error('Error loading ticket sales:', error);
    }
}

function updateTicketSalesStats(summary) {
    document.getElementById('totalTicketsSold').textContent = summary.sold_tickets || 0;
    document.getElementById('totalRevenue').textContent = `GHS ${(summary.total_revenue || 0).toFixed(2)}`;
    document.getElementById('pendingTickets').textContent = summary.pending_tickets || 0;
    document.getElementById('avgTicketPrice').textContent = `GHS ${(summary.avg_ticket_price || 0).toFixed(2)}`;
}

async function loadTicketSalesTable() {
    try {
        const response = await fetch('/GHTouristApp/api/tickets.php?action=organizer-tickets&organizer_id=6');
        const result = await response.json();
        
        if (result.success) {
            const tbody = document.getElementById('ticketSalesTableBody');
            tbody.innerHTML = '';
            
            if (result.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="8" class="text-center">No ticket sales found</td></tr>';
                return;
            }
            
            result.data.forEach(ticket => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${ticket.ticket_code}</td>
                    <td>${ticket.event_title}</td>
                    <td>${ticket.attendee_name}</td>
                    <td>${ticket.attendee_email}</td>
                    <td>GHS ${ticket.price_paid}</td>
                    <td><span class="status-badge status-${ticket.payment_status}">${ticket.payment_status}</span></td>
                    <td>${new Date(ticket.purchased_at).toLocaleDateString()}</td>
                    <td>
                        <button class="btn-secondary" onclick="viewTicketDetails('${ticket.ticket_code}')">
                            <i class="fas fa-eye"></i> View
                        </button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading ticket sales table:', error);
    }
}

async function loadEventFilters() {
    try {
        const response = await fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6');
        const result = await response.json();
        
        if (result.success) {
            const eventFilter = document.getElementById('eventFilterSelect');
            const attendeeEventFilter = document.getElementById('attendeeEventFilterSelect');
            
            eventFilter.innerHTML = '<option value="">All Events</option>';
            attendeeEventFilter.innerHTML = '<option value="">All Events</option>';
            
            result.data.forEach(event => {
                const option = `<option value="${event.id}">${event.title}</option>`;
                eventFilter.innerHTML += option;
                attendeeEventFilter.innerHTML += option;
            });
        }
    } catch (error) {
        console.error('Error loading event filters:', error);
    }
}

function filterTicketsByEvent() {
    const eventId = document.getElementById('eventFilterSelect').value;
    loadTicketSalesTable(eventId);
}

function refreshTicketSales() {
    loadTicketSales();
}

async function viewTicketDetails(ticketReference) {
    try {
        const response = await fetch(`/GHTouristApp/api/tickets.php?action=details&reference=${ticketReference}`);
        const result = await response.json();
        
        if (result.success) {
            alert(`Ticket Details:\n\nReference: ${result.data.ticket_reference}\nEvent: ${result.data.event_title}\nAttendee: ${result.data.attendee_name}\nEmail: ${result.data.attendee_email}\nStatus: ${result.data.payment_status}`);
        }
    } catch (error) {
        console.error('Error loading ticket details:', error);
    }
}

// Attendees Functions
async function loadAttendees() {
    try {
        const response = await fetch('/GHTouristApp/api/tickets.php?action=organizer-tickets&organizer_id=6&status=completed');
        const result = await response.json();
        
        if (result.success) {
            updateAttendeeStats(result.data);
            loadAttendeesTable(result.data);
        }
    } catch (error) {
        console.error('Error loading attendees:', error);
    }
}

function updateAttendeeStats(attendees) {
    const totalAttendees = attendees.length;
    const checkedIn = attendees.filter(a => a.check_in_status).length;
    const pendingCheckin = totalAttendees - checkedIn;
    const attendanceRate = totalAttendees > 0 ? ((checkedIn / totalAttendees) * 100).toFixed(1) : 0;
    
    document.getElementById('totalAttendees').textContent = totalAttendees;
    document.getElementById('checkedInAttendees').textContent = checkedIn;
    document.getElementById('pendingAttendees').textContent = pendingCheckin;
    document.getElementById('attendanceRate').textContent = `${attendanceRate}%`;
}

function loadAttendeesTable(attendees) {
    const tbody = document.getElementById('attendeesTableBody');
    tbody.innerHTML = '';
    
    if (attendees.length === 0) {
        tbody.innerHTML = '<tr><td colspan="8" class="text-center">No attendees found</td></tr>';
        return;
    }
    
    attendees.forEach(attendee => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${attendee.ticket_code}</td>
            <td>${attendee.event_title}</td>
            <td>${attendee.attendee_name}</td>
            <td>${attendee.attendee_email}</td>
            <td>${attendee.attendee_phone || 'N/A'}</td>
            <td><span class="status-badge ${attendee.check_in_status ? 'status-completed' : 'status-pending'}">${attendee.check_in_status ? 'Checked In' : 'Pending'}</span></td>
            <td>${attendee.check_in_time ? new Date(attendee.check_in_time).toLocaleString() : 'N/A'}</td>
            <td>
                ${!attendee.check_in_status ? 
                    `<button class="btn-primary" onclick="checkInAttendeeById('${attendee.ticket_code}')">
                        <i class="fas fa-check"></i> Check In
                    </button>` : 
                    '<span class="text-muted">Checked In</span>'
                }
            </td>
        `;
        tbody.appendChild(row);
    });
}

function filterAttendeesByEvent() {
    const eventId = document.getElementById('attendeeEventFilterSelect').value;
    loadAttendees(eventId);
}

function refreshAttendees() {
    loadAttendees();
}

async function checkInAttendee() {
    const ticketReference = document.getElementById('ticketReferenceInput').value.trim();
    if (!ticketReference) {
        showCheckinResult('Please enter a ticket reference', 'error');
        return;
    }
    
    await checkInAttendeeById(ticketReference);
}

async function checkInAttendeeById(ticketReference) {
    try {
        const response = await fetch('/GHTouristApp/api/tickets.php?action=validate', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ ticket_code: ticketReference })
        });
        
        const result = await response.json();
        
        if (result.success) {
            showCheckinResult(`Successfully checked in: ${result.data.attendee_name}`, 'success');
            document.getElementById('ticketReferenceInput').value = '';
            refreshAttendees();
        } else {
            showCheckinResult(result.error || 'Check-in failed', 'error');
        }
    } catch (error) {
        console.error('Error checking in attendee:', error);
        showCheckinResult('Error during check-in', 'error');
    }
}

function showCheckinResult(message, type) {
    const resultDiv = document.getElementById('checkinResult');
    resultDiv.textContent = message;
    resultDiv.className = `checkin-result checkin-${type}`;
    resultDiv.style.display = 'block';
    
    setTimeout(() => {
        resultDiv.style.display = 'none';
    }, 5000);
}

function exportAttendeeList() {
    // Simple CSV export functionality
    const table = document.getElementById('attendeesTable');
    const rows = Array.from(table.querySelectorAll('tr'));
    
    const csvContent = rows.map(row => {
        const cells = Array.from(row.querySelectorAll('th, td'));
        return cells.slice(0, -1).map(cell => `"${cell.textContent.trim()}"`).join(',');
    }).join('\n');
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'attendees_list.csv';
    a.click();
    window.URL.revokeObjectURL(url);
}

// ==================== CHAT ROOMS FUNCTIONALITY ====================

let currentChatRoom = null;
let chatPollingInterval = null;

async function loadChatRooms() {
    try {
        await loadChatStats();
        await loadChatRoomsTable();
        await loadEventsForChatFilter();
    } catch (error) {
        console.error('Error loading chat rooms:', error);
    }
}

async function loadChatStats() {
    try {
        // Sample stats for now
        document.getElementById('totalChatRooms').textContent = '3';
        document.getElementById('totalParticipants').textContent = '45';
        document.getElementById('totalMessages').textContent = '127';
        document.getElementById('activeChatRooms').textContent = '2';
    } catch (error) {
        console.error('Error loading chat stats:', error);
    }
}

async function loadChatRoomsTable() {
    try {
        const tbody = document.getElementById('chatRoomsTableBody');
        
        // Load real chat rooms from API
        const response = await fetch('/GHTouristApp/api/chat.php?endpoint=rooms&organizer_id=6');
        const result = await response.json();
        
        tbody.innerHTML = '';
        
        if (result.success && result.rooms && result.rooms.length > 0) {
            result.rooms.forEach(room => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${room.name}</td>
                    <td>${room.event_title || 'N/A'}</td>
                    <td><span class="status-badge status-${room.room_type}">${room.room_type}</span></td>
                    <td>${room.participant_count || 0}/${room.max_participants || 'N/A'}</td>
                    <td>${room.message_count || 0}</td>
                    <td><span class="status-badge status-${room.status || 'active'}">${room.status || 'active'}</span></td>
                    <td>${new Date(room.created_at).toLocaleDateString()}</td>
                    <td>
                        <button class="btn-primary" onclick="openChatRoom(${room.id}, '${room.name}')">
                            <i class="fas fa-comments"></i> Open
                        </button>
                        <button class="btn-secondary" onclick="manageChatRoom(${room.id})">
                            <i class="fas fa-cog"></i> Manage
                        </button>
                    </td>
                `;
                tbody.appendChild(row);
            });
        } else {
            tbody.innerHTML = '<tr><td colspan="8" class="text-center">No chat rooms found</td></tr>';
        }
    } catch (error) {
        console.error('Error loading chat rooms table:', error);
        const tbody = document.getElementById('chatRoomsTableBody');
        tbody.innerHTML = '<tr><td colspan="8" class="text-center">Error loading chat rooms</td></tr>';
    }
}

async function loadEventsForChatFilter() {
    try {
        const response = await fetch('/GHTouristApp/api/events.php?action=organizer-events&organizer_id=6');
        
        const result = await response.json();
        const eventFilter = document.getElementById('chatEventFilter');
        const chatRoomEvent = document.getElementById('chatRoomEvent');
        
        if (result.success && result.data) {
            eventFilter.innerHTML = '<option value="">All Events</option>';
            chatRoomEvent.innerHTML = '<option value="">Select event</option>';
            
            result.data.forEach(event => {
                eventFilter.innerHTML += `<option value="${event.id}">${event.title}</option>`;
                chatRoomEvent.innerHTML += `<option value="${event.id}">${event.title}</option>`;
            });
        } else {
            console.log('No events found or API error:', result);
            eventFilter.innerHTML = '<option value="">All Events</option>';
            chatRoomEvent.innerHTML = '<option value="">No events available</option>';
        }
    } catch (error) {
        console.error('Error loading events for chat filter:', error);
        const eventFilter = document.getElementById('chatEventFilter');
        const chatRoomEvent = document.getElementById('chatRoomEvent');
        eventFilter.innerHTML = '<option value="">All Events</option>';
        chatRoomEvent.innerHTML = '<option value="">Error loading events</option>';
    }
}

function showCreateChatRoomModal() {
    document.getElementById('createChatRoomModal').style.display = 'block';
}

async function openChatRoom(roomId, roomName) {
    currentChatRoom = roomId;
    document.getElementById('activeChatTitle').textContent = roomName;
    document.getElementById('activeChatSection').style.display = 'block';
    
    await loadChatMessages(roomId);
    startChatPolling();
}

function closeChatInterface() {
    document.getElementById('activeChatSection').style.display = 'none';
    currentChatRoom = null;
    stopChatPolling();
}

async function loadChatMessages(roomId) {
    try {
        const messagesContainer = document.getElementById('chatMessages');
        
        // Sample messages for demonstration
        const sampleMessages = [
            {
                id: 1,
                message_type: 'system',
                content: 'Welcome to the chat room! Please be respectful and follow community guidelines.',
                created_at: new Date(Date.now() - 3600000).toISOString()
            },
            {
                id: 2,
                message_type: 'text',
                content: 'Hello everyone! Excited for the event!',
                sender_name: 'John Doe',
                created_at: new Date(Date.now() - 1800000).toISOString()
            }
        ];
        
        messagesContainer.innerHTML = '';
        
        sampleMessages.forEach(message => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `chat-message ${message.message_type}`;
            
            if (message.message_type === 'system') {
                messageDiv.innerHTML = `
                    <div class="chat-message-content">${message.content}</div>
                `;
            } else {
                messageDiv.innerHTML = `
                    <div class="chat-message-header">
                        ${message.sender_name} - ${new Date(message.created_at).toLocaleTimeString()}
                    </div>
                    <div class="chat-message-content">${message.content}</div>
                `;
            }
            
            messagesContainer.appendChild(messageDiv);
        });
        
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    } catch (error) {
        console.error('Error loading chat messages:', error);
    }
}

async function sendChatMessage() {
    const input = document.getElementById('chatMessageInput');
    const content = input.value.trim();
    
    if (!content || !currentChatRoom) return;
    
    try {
        const messagesContainer = document.getElementById('chatMessages');
        const messageDiv = document.createElement('div');
        messageDiv.className = 'chat-message own';
        
        messageDiv.innerHTML = `
            <div class="chat-message-header">
                You - ${new Date().toLocaleTimeString()}
            </div>
            <div class="chat-message-content">${content}</div>
        `;
        
        messagesContainer.appendChild(messageDiv);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        input.value = '';
    } catch (error) {
        console.error('Error sending message:', error);
    }
}

function handleChatKeyPress(event) {
    if (event.key === 'Enter') {
        sendChatMessage();
    }
}

function startChatPolling() {
    if (chatPollingInterval) clearInterval(chatPollingInterval);
    
    chatPollingInterval = setInterval(() => {
        if (currentChatRoom) {
            console.log('Polling for new messages...');
        }
    }, 3000);
}

function stopChatPolling() {
    if (chatPollingInterval) {
        clearInterval(chatPollingInterval);
        chatPollingInterval = null;
    }
}

function refreshChatRooms() {
    loadChatRooms();
}

function filterChatRooms() {
    loadChatRoomsTable();
}

function manageChatRoom(roomId) {
    alert(`Managing chat room ${roomId} - Feature coming soon!`);
}

// ==================== NOTIFICATIONS FUNCTIONALITY ====================

async function loadNotifications() {
    try {
        await loadNotificationStats();
        await loadNotificationsList();
    } catch (error) {
        console.error('Error loading notifications:', error);
    }
}

async function loadNotificationStats() {
    try {
        const response = await fetch('/GHTouristApp/api/notifications.php?action=stats&user_id=6');
        
        const result = await response.json();
        
        if (result.success) {
            document.getElementById('totalNotifications').textContent = result.stats.total || 0;
            document.getElementById('unreadNotifications').textContent = result.stats.unread || 0;
            document.getElementById('urgentNotifications').textContent = result.stats.urgent || 0;
            document.getElementById('todayNotifications').textContent = result.stats.today || 0;
        }
    } catch (error) {
        console.error('Error loading notification stats:', error);
        document.getElementById('totalNotifications').textContent = '0';
        document.getElementById('unreadNotifications').textContent = '0';
        document.getElementById('urgentNotifications').textContent = '0';
        document.getElementById('todayNotifications').textContent = '0';
    }
}

async function loadNotificationsList() {
    try {
        const response = await fetch('/GHTouristApp/api/notifications.php?action=list&user_id=6');
        
        const result = await response.json();
        const container = document.getElementById('notificationsContainer');
        
        if (result.success && result.notifications) {
            container.innerHTML = '';
            
            if (result.notifications.length === 0) {
                container.innerHTML = `
                    <div class="notification-item">
                        <div class="notification-content">
                            <p>No notifications found</p>
                        </div>
                    </div>
                `;
                return;
            }
            
            result.notifications.forEach(notification => {
                const notificationDiv = document.createElement('div');
                notificationDiv.className = `notification-item ${!notification.is_read ? 'unread' : ''} ${notification.priority === 'urgent' ? 'urgent' : ''}`;
                notificationDiv.onclick = () => markNotificationAsRead(notification.id);
                
                notificationDiv.innerHTML = `
                    <div class="notification-header">
                        <h4 class="notification-title">${notification.title}</h4>
                        <div>
                            <span class="notification-type ${notification.type}">${notification.type.replace('_', ' ')}</span>
                            <span class="notification-time">${new Date(notification.created_at).toLocaleString()}</span>
                        </div>
                    </div>
                    <div class="notification-content">
                        ${notification.message}
                    </div>
                    ${!notification.is_read ? `
                        <div class="notification-actions">
                            <button class="btn-primary" onclick="markNotificationAsRead(${notification.id}); event.stopPropagation();">
                                Mark as Read
                            </button>
                        </div>
                    ` : ''}
                `;
                
                container.appendChild(notificationDiv);
            });
        } else {
            container.innerHTML = `
                <div class="notification-item">
                    <div class="notification-content">
                        <p>No notifications available</p>
                    </div>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading notifications list:', error);
        const container = document.getElementById('notificationsContainer');
        container.innerHTML = `
            <div class="notification-item">
                <div class="notification-content">
                    <p>Error loading notifications</p>
                </div>
            </div>
        `;
    }
}

async function markNotificationAsRead(notificationId) {
    try {
        await loadNotifications();
    } catch (error) {
        console.error('Error marking notification as read:', error);
    }
}

async function markAllAsRead() {
    try {
        await loadNotifications();
        alert('All notifications marked as read');
    } catch (error) {
        console.error('Error marking all notifications as read:', error);
    }
}

function refreshNotifications() {
    loadNotifications();
}

function filterNotifications() {
    loadNotificationsList();
}
