// Tourist Dashboard JavaScript
class TouristDashboard {
    constructor() {
        this.token = localStorage.getItem('auth_token');
        this.user = JSON.parse(localStorage.getItem('user_data') || '{}');
        this.currentEventId = null;
        this.init();
    }

    async init() {
        if (!this.token || !this.user.id) {
            window.location.href = 'index.html';
            return;
        }

        this.updateUserInfo();
        await this.loadDashboardData();
        this.setupEventListeners();
    }

    updateUserInfo() {
        const userName = document.getElementById('userName');
        const welcomeName = document.getElementById('welcomeName');
        const userAvatar = document.getElementById('userAvatar');

        if (this.user.full_name) {
            userName.textContent = this.user.full_name;
            welcomeName.textContent = this.user.full_name.split(' ')[0];
            
            // Create initials for avatar
            const initials = this.user.full_name
                .split(' ')
                .map(name => name.charAt(0))
                .join('')
                .toUpperCase()
                .substring(0, 2);
            userAvatar.textContent = initials;
        }

        // Update profile form
        document.getElementById('profileName').value = this.user.full_name || '';
        document.getElementById('profileEmail').value = this.user.email || '';
        document.getElementById('profilePhone').value = this.user.phone || '';
    }

    async loadDashboardData() {
        try {
            await Promise.all([
                this.loadStats(),
                this.loadRecentActivities(),
                this.loadEvents(),
                this.loadBookings()
            ]);
        } catch (error) {
            console.error('Error loading dashboard data:', error);
        }
    }

    async loadStats() {
        try {
            // Load user statistics
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=stats&user_id=${this.user.id}`);
            const result = await response.json();
            
            if (result.success) {
                document.getElementById('totalBookings').textContent = result.stats.total_bookings || 0;
                document.getElementById('upcomingEvents').textContent = result.stats.upcoming_events || 0;
                document.getElementById('accommodationBookings').textContent = result.stats.accommodation_bookings || 0;
                document.getElementById('reviewsGiven').textContent = result.stats.reviews_given || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
            // Set default values
            document.getElementById('totalBookings').textContent = '5';
            document.getElementById('upcomingEvents').textContent = '3';
            document.getElementById('accommodationBookings').textContent = '2';
            document.getElementById('reviewsGiven').textContent = '8';
        }
    }

    async loadRecentActivities() {
        try {
            const container = document.getElementById('recentActivities');
            
            // Sample recent activities
            const activities = [
                {
                    type: 'booking',
                    title: 'Ghana Music Festival',
                    description: 'Booked VIP ticket',
                    date: '2025-09-15',
                    icon: 'fas fa-ticket-alt'
                },
                {
                    type: 'review',
                    title: 'Cultural Heritage Day',
                    description: 'Left a 5-star review',
                    date: '2025-09-14',
                    icon: 'fas fa-star'
                }
            ];

            container.innerHTML = activities.map(activity => `
                <div class="activity-item">
                    <div class="activity-icon">
                        <i class="${activity.icon}"></i>
                    </div>
                    <div class="activity-content">
                        <div class="activity-title">${activity.title}</div>
                        <div class="activity-description">${activity.description}</div>
                        <div class="activity-date">${new Date(activity.date).toLocaleDateString()}</div>
                    </div>
                </div>
            `).join('');
        } catch (error) {
            console.error('Error loading recent activities:', error);
        }
    }

    async loadEvents() {
        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=list');
            const result = await response.json();
            
            const container = document.getElementById('eventsGrid');
            if (!container) return;
            
            if (result.success && result.events && result.events.length > 0) {
                container.innerHTML = result.events.map(event => `
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">${event.title}</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> ${event.venue_name || 'TBA'}</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> ${new Date(event.start_datetime).toLocaleDateString()}
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> ${new Date(event.start_datetime).toLocaleTimeString()}
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${event.ticket_price || 50}</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">${event.description || 'Join us for an amazing event experience!'}</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">${event.category_name || 'Event'}</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">${event.approval_status || 'Open'}</span>
                        </div>
                        <button onclick="bookEvent(${event.id}, '${event.title}', ${event.ticket_price || 50})" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                `).join('');
            } else {
                // Show sample events with Ghana events data
                container.innerHTML = `
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Accra Conference Centre</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 15, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Experience the best of Ghanaian music with top artists performing live!</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Music</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Live Performance</span>
                        </div>
                        <button onclick="bookEvent(1, 'Ghana Music Festival', 150)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Labadi Beach Hotel</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 20, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Celebrate Ghana's rich cultural heritage with traditional performances and exhibitions.</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Cultural</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Family Friendly</span>
                        </div>
                        <button onclick="bookEvent(2, 'Homowo Festival', 75)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Tech Innovation Summit</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Kumasi Cultural Centre</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> January 10, 2026
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 9:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 200</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Join industry leaders discussing the future of technology in Ghana and Africa.</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Technology</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Business</span>
                        </div>
                        <button onclick="bookEvent(3, 'Tech Innovation Summit', 200)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading events:', error);
            const container = document.getElementById('eventsGrid');
            if (container) {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-calendar-times" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>Unable to Load Events</h3>
                        <p>Please check your connection and try again.</p>
                        <button onclick="dashboard.loadEvents()" style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; margin-top: 15px;">
                            Retry
                        </button>
                    </div>
                `;
            }
        }
    }

    async loadBookings() {
        try {
            const response = await fetch(`/GHTouristApp/api/tickets.php?action=user-tickets&user_id=${this.user.id}`);
            const result = await response.json();
            
            const container = document.getElementById('bookingsGrid');
            if (!container) return;
            
            if (result.success && result.tickets && result.tickets.length > 0) {
                container.innerHTML = result.tickets.map(ticket => `
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">${ticket.event_title}</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> ${ticket.ticket_type} Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> ${new Date(ticket.event_date).toLocaleDateString()}
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> ${new Date(ticket.event_date).toLocaleTimeString()}
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${ticket.price_paid}</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: ${ticket.payment_status === 'paid' ? '#e8f5e8' : '#fff3cd'}; color: ${ticket.payment_status === 'paid' ? '#28a745' : '#856404'}; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-${ticket.payment_status === 'paid' ? 'check-circle' : 'clock'}"></i> ${ticket.payment_status}
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> ${ticket.ticket_code}
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('${ticket.ticket_code}')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(${ticket.event_id})" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                `).join('');
            } else {
                // Show sample bookings
                container.innerHTML = `
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> VIP Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 15, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-check-circle"></i> Confirmed
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> GMF2025001
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('GMF2025001')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(1)" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> Regular Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 20, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-check-circle"></i> Confirmed
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> HMW2025002
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('HMW2025002')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(2)" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading bookings:', error);
            const container = document.getElementById('bookingsGrid');
            if (container) {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-ticket-alt" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>No Bookings Yet</h3>
                        <p>Book your first event to see your tickets here!</p>
                        <button onclick="showSection('discoverEventsSection')" style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; margin-top: 15px;">
                            Discover Events
                        </button>
                    </div>
                `;
            }
        }
    }

    setupEventListeners() {
        // Sidebar navigation
        document.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const sectionId = link.getAttribute('onclick')?.match(/showSection\('(.+?)'\)/)?.[1];
                if (sectionId) {
                    this.showSection(sectionId);
                }
            });
        });

        // Profile form
        const profileForm = document.getElementById('profileForm');
        if (profileForm) {
            profileForm.addEventListener('submit', (e) => this.updateProfile(e));
        }

        // Logout
        const logoutBtn = document.getElementById('logoutBtn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => this.logout());
        }
    }

    showSection(sectionId) {
        // Hide all sections
        document.querySelectorAll('.content-section').forEach(section => {
            section.style.display = 'none';
        });

        // Show selected section
        const targetSection = document.getElementById(sectionId);
        if (targetSection) {
            targetSection.style.display = 'block';
        }

        // Update active nav
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });

        const activeLink = document.querySelector(`[onclick*="${sectionId}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }

        // Load section-specific data
        switch(sectionId) {
            case 'discoverEventsSection':
                this.loadEvents();
                break;
            case 'myBookingsSection':
                this.loadBookings();
                break;
        }
    }

    async updateProfile(e) {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const profileData = {
            full_name: formData.get('full_name'),
            email: formData.get('email'),
            phone: formData.get('phone')
        };

        try {
            const response = await fetch('/GHTouristApp/api/tourist.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    action: 'update_profile',
                    ...profileData
                })
            });

            const result = await response.json();
            
            if (result.success) {
                // Update local user data
                this.user = { ...this.user, ...profileData };
                localStorage.setItem('user_data', JSON.stringify(this.user));
                this.updateUserInfo();
                
                alert('Profile updated successfully!');
            } else {
                alert('Error updating profile: ' + result.message);
            }
        } catch (error) {
            console.error('Error updating profile:', error);
            alert('Error updating profile. Please try again.');
        }
    }

    logout() {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        window.location.href = 'index.html';
    }
}

// Global functions for HTML onclick handlers
function showSection(sectionId) {
    if (window.dashboard) {
        window.dashboard.showSection(sectionId);
    }
}

function bookEvent(eventId, eventTitle, price) {
    alert(`Booking ${eventTitle} for GHS ${price}. This feature will be implemented soon!`);
}

function downloadTicket(ticketCode) {
    alert(`Downloading ticket ${ticketCode}. This feature will be implemented soon!`);
}

function joinEventChat(eventId) {
    alert(`Joining chat for event ${eventId}. This feature will be implemented soon!`);
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.dashboard = new TouristDashboard();
});
