// Tourist Dashboard JavaScript
class TouristDashboard {
    constructor() {
        this.token = localStorage.getItem('auth_token');
        this.user = JSON.parse(localStorage.getItem('user_data') || '{}');
        this.currentEventId = null;
        this.init();
    }

    async init() {
        console.log('Initializing dashboard...');
        console.log('Token:', this.token);
        console.log('User:', this.user);
        
        if (!this.token || !this.user.id) {
            console.log('Missing token or user ID, redirecting to login');
            window.location.href = 'index.html';
            return;
        }

        this.updateUserInfo();
        await this.loadDashboardData();
        this.setupEventListeners();
        
        console.log('Dashboard initialized successfully');
    }

    updateUserInfo() {
        const userName = document.getElementById('userName');
        const welcomeName = document.getElementById('welcomeName');
        const userAvatar = document.getElementById('userAvatar');

        if (this.user.full_name) {
            if (userName) userName.textContent = this.user.full_name;
            if (welcomeName) welcomeName.textContent = this.user.full_name.split(' ')[0];
            
            // Create initials for avatar
            const initials = this.user.full_name
                .split(' ')
                .map(name => name.charAt(0))
                .join('')
                .toUpperCase()
                .substring(0, 2);
            if (userAvatar) userAvatar.textContent = initials;
        }

        // Update profile form
        const profileName = document.getElementById('profileName');
        const profileEmail = document.getElementById('profileEmail');
        const profilePhone = document.getElementById('profilePhone');
        
        if (profileName) profileName.value = this.user.full_name || '';
        if (profileEmail) profileEmail.value = this.user.email || '';
        if (profilePhone) profilePhone.value = this.user.phone || '';
    }

    async loadDashboardData() {
        try {
            await Promise.all([
                this.loadStats(),
                this.loadRecentActivities()
            ]);
        } catch (error) {
            console.error('Error loading dashboard data:', error);
        }
    }

    async loadStats() {
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=stats&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.data) {
                const totalEvents = document.getElementById('totalEvents');
                const upcomingEvents = document.getElementById('upcomingEvents');
                const totalSpent = document.getElementById('totalSpent');
                
                if (totalEvents) totalEvents.textContent = result.data.total_events || 0;
                if (upcomingEvents) upcomingEvents.textContent = result.data.upcoming_events || 0;
                if (totalSpent) totalSpent.textContent = `GHS ${result.data.total_spent || 0}`;
            } else {
                // Set default values when API fails or returns no data
                const totalEvents = document.getElementById('totalEvents');
                const upcomingEvents = document.getElementById('upcomingEvents');
                const totalSpent = document.getElementById('totalSpent');
                
                if (totalEvents) totalEvents.textContent = '0';
                if (upcomingEvents) upcomingEvents.textContent = '0';
                if (totalSpent) totalSpent.textContent = 'GHS 0';
            }
        } catch (error) {
            console.error('Error loading stats:', error);
            // Set default values on error
            const totalEvents = document.getElementById('totalEvents');
            const upcomingEvents = document.getElementById('upcomingEvents');
            const totalSpent = document.getElementById('totalSpent');
            
            if (totalEvents) totalEvents.textContent = '0';
            if (upcomingEvents) upcomingEvents.textContent = '0';
            if (totalSpent) totalSpent.textContent = 'GHS 0';
        }
    }

    async loadRecentActivities() {
        const container = document.getElementById('recentActivities');
        if (!container) return;
        
        const activities = [
            {
                icon: 'fas fa-ticket-alt',
                text: 'Booked ticket for Ghana Music Festival',
                time: '2 hours ago'
            },
            {
                icon: 'fas fa-heart',
                text: 'Added Homowo Festival to favorites',
                time: '1 day ago'
            },
            {
                icon: 'fas fa-star',
                text: 'Rated Tech Innovation Summit',
                time: '3 days ago'
            }
        ];
        
        container.innerHTML = activities.map(activity => `
            <div style="display: flex; align-items: center; padding: 10px 0; border-bottom: 1px solid #f0f0f0;">
                <i class="${activity.icon}" style="color: #667eea; margin-right: 15px; width: 20px;"></i>
                <div style="flex: 1;">
                    <p style="margin: 0; color: #333; font-size: 0.9rem;">${activity.text}</p>
                    <small style="color: #666;">${activity.time}</small>
                </div>
            </div>
        `).join('');
    }

    showSection(sectionId) {
        // Hide all sections
        document.querySelectorAll('.content-section').forEach(section => {
            section.classList.remove('active');
        });
        
        // Show selected section
        const targetSection = document.getElementById(sectionId);
        if (targetSection) {
            targetSection.classList.add('active');
        }
        
        // Update sidebar active state
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });
        
        const activeLink = document.querySelector(`[onclick="showSection('${sectionId}')"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }
        
        // Load section-specific data
        if (sectionId === 'eventsSection') {
            this.loadEvents();
        } else if (sectionId === 'bookingsSection') {
            this.loadBookings();
        } else if (sectionId === 'chatSection') {
            this.loadEventChats();
        } else if (sectionId === 'safetySection') {
            this.loadSafetyReports();
        }
    }

    async loadEvents() {
        const container = document.getElementById('eventsGrid');
        if (!container) return;
        
        container.innerHTML = '<div style="text-align: center; padding: 40px; color: #666;"><i class="fas fa-spinner fa-spin"></i> Loading events...</div>';
        
        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=list&status=approved', {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.data && result.data.length > 0) {
                container.innerHTML = result.data.map(event => this.createEventCard(event)).join('');
            } else {
                // Fallback to static events if API fails
                container.innerHTML = this.getStaticEvents();
            }
        } catch (error) {
            console.error('Error loading events:', error);
            // Fallback to static events
            container.innerHTML = this.getStaticEvents();
        }
    }
    
    createEventCard(event) {
        const eventDate = new Date(event.start_datetime);
        const formattedDate = eventDate.toLocaleDateString('en-GB', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        const formattedTime = eventDate.toLocaleTimeString('en-GB', {
            hour: '2-digit',
            minute: '2-digit'
        });
        
        return `
            <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">${event.title}</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> ${event.venue_name || 'Venue TBA'}</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> ${formattedDate}
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> ${formattedTime}
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${event.ticket_price || 50}</div>
                        <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                    </div>
                </div>
                <p style="color: #666; margin-bottom: 15px;">${event.description || 'Join us for this amazing event!'}</p>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">${event.category || 'Event'}</span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Live</span>
                </div>
                <button onclick="bookEvent(${event.id}, '${event.title}', ${event.ticket_price || 50})" 
                        style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                    <i class="fas fa-ticket-alt"></i> Book Now
                </button>
            </div>
        `;
    }
    
    getStaticEvents() {
        return `
            <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Accra Conference Centre</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> December 15, 2025
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                        <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                    </div>
                </div>
                <p style="color: #666; margin-bottom: 15px;">Experience the best of Ghanaian music with top artists performing live!</p>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Music</span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Live Performance</span>
                </div>
                <button onclick="bookEvent(1, 'Ghana Music Festival', 150)" 
                        style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                    <i class="fas fa-ticket-alt"></i> Book Now
                </button>
            </div>
            <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Labadi Beach Hotel</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> December 20, 2025
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                        <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                    </div>
                </div>
                <p style="color: #666; margin-bottom: 15px;">Celebrate Ghana's rich cultural heritage with traditional performances and exhibitions.</p>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Cultural</span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Family Friendly</span>
                </div>
                <button onclick="bookEvent(2, 'Homowo Festival', 75)" 
                        style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                    <i class="fas fa-ticket-alt"></i> Book Now
                </button>
            </div>
            <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">Tech Innovation Summit</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Kumasi Cultural Centre</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> January 10, 2026
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> 9:00 AM
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 200</div>
                        <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                    </div>
                </div>
                <p style="color: #666; margin-bottom: 15px;">Join industry leaders discussing the future of technology in Ghana and Africa.</p>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Technology</span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Business</span>
                </div>
                <button onclick="bookEvent(3, 'Tech Innovation Summit', 200)" 
                        style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                    <i class="fas fa-ticket-alt"></i> Book Now
                </button>
            </div>
        `;
    }

    async loadBookings() {
        const container = document.getElementById('bookingsGrid');
        if (!container) return;
        
        container.innerHTML = '<div style="text-align: center; padding: 40px; color: #666;"><i class="fas fa-spinner fa-spin"></i> Loading bookings...</div>';
        
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=bookings&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.tickets && result.tickets.length > 0) {
                container.innerHTML = result.tickets.map(ticket => this.createBookingCard(ticket)).join('');
            } else {
                container.innerHTML = '<div style="text-align: center; padding: 40px; color: #666;"><i class="fas fa-ticket-alt"></i><br><br>No bookings found. Book your first event!</div>';
            }
        } catch (error) {
            console.error('Error loading bookings:', error);
            // Fallback to static bookings
            container.innerHTML = `
            <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> VIP Ticket</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> December 15, 2025
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                        <div style="color: #666; font-size: 0.9rem;">paid</div>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-check-circle"></i> Confirmed
                    </span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-hashtag"></i> GMF2025001
                    </span>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button onclick="downloadTicket('GMF2025001')" 
                            style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-download"></i> Download
                    </button>
                    <button onclick="joinEventChat(1)" 
                            style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-comments"></i> Chat
                    </button>
                </div>
            </div>
            <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> Regular Ticket</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> December 20, 2025
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                        <div style="color: #666; font-size: 0.9rem;">paid</div>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-check-circle"></i> Confirmed
                    </span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-hashtag"></i> HMW2025002
                    </span>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button onclick="downloadTicket('HMW2025002')" 
                            style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-download"></i> Download
                    </button>
                    <button onclick="joinEventChat(2)" 
                            style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-comments"></i> Chat
                    </button>
                </div>
            </div>
        `;
        }
    }
    
    createBookingCard(ticket) {
        const eventDate = new Date(ticket.event_date || ticket.start_datetime);
        const formattedDate = eventDate.toLocaleDateString('en-GB', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        });
        const formattedTime = eventDate.toLocaleTimeString('en-GB', {
            hour: '2-digit',
            minute: '2-digit'
        });
        
        const statusColor = ticket.payment_status === 'completed' ? '#28a745' : '#ffc107';
        const statusText = ticket.payment_status === 'completed' ? 'Confirmed' : 'Pending';
        const statusIcon = ticket.payment_status === 'completed' ? 'check-circle' : 'clock';
        
        return `
            <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h3 style="color: #333; margin-bottom: 5px;">${ticket.event_title}</h3>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> ${ticket.ticket_type || 'Regular'} Ticket</p>
                        <div style="color: #667eea; margin-bottom: 10px;">
                            <i class="fas fa-calendar"></i> ${formattedDate}
                            <i class="fas fa-clock" style="margin-left: 15px;"></i> ${formattedTime}
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${ticket.price_paid}</div>
                        <div style="color: #666; font-size: 0.9rem;">paid</div>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                    <span style="background: ${statusColor}20; color: ${statusColor}; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-${statusIcon}"></i> ${statusText}
                    </span>
                    <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                        <i class="fas fa-hashtag"></i> ${ticket.ticket_code}
                    </span>
                </div>
                <div style="display: flex; gap: 10px;">
                    <button onclick="downloadTicket('${ticket.ticket_code}')" 
                            style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-download"></i> Download
                    </button>
                    <button onclick="joinEventChat(${ticket.event_id})" 
                            style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                        <i class="fas fa-comments"></i> Chat
                    </button>
                </div>
            </div>
        `;
    }

    setupEventListeners() {
        // Booking form submission
        const bookingForm = document.getElementById('bookingForm');
        if (bookingForm) {
            bookingForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.processBooking();
            });
        }

        // Logout button
        const logoutBtn = document.getElementById('logoutBtn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => {
                this.logout();
            });
        }
    }

    async updateProfile() {
        const formData = {
            full_name: document.getElementById('profileName').value,
            email: document.getElementById('profileEmail').value,
            phone: document.getElementById('profilePhone').value
        };

        try {
            const response = await fetch('/GHTouristApp/api/tourist.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    action: 'update_profile',
                    user_id: this.user.id,
                    ...formData
                })
            });

            const result = await response.json();
            if (result.success) {
                this.user = { ...this.user, ...formData };
                localStorage.setItem('user_data', JSON.stringify(this.user));
                this.updateUserInfo();
                alert('Profile updated successfully!');
            } else {
                alert('Error updating profile: ' + (result.message || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error updating profile:', error);
            alert('Error updating profile. Please try again.');
        }
    }

    async processBooking() {
        console.log('Current user:', this.user);
        console.log('Current booking:', window.currentBooking);
        
        if (!this.user || !this.user.id) {
            this.showNotification('User not authenticated. Please login again.', 'error');
            return;
        }
        
        if (!window.currentBooking || !window.currentBooking.eventId) {
            this.showNotification('No event selected for booking.', 'error');
            return;
        }
        
        const bookingData = {
            event_id: window.currentBooking.eventId,
            user_id: this.user.id,
            ticket_type: document.getElementById('ticketType').value,
            quantity: parseInt(document.getElementById('ticketQuantity').value),
            payment_method: document.getElementById('paymentMethod').value
        };
        
        // Show loading state
        const submitBtn = document.querySelector('#bookingForm button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
        submitBtn.disabled = true;
        
        try {
            console.log('Booking data being sent:', bookingData);
            
            const response = await fetch('/GHTouristApp/api/tourist.php?action=book-event', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify(bookingData)
            });
            
            console.log('Response status:', response.status);
            const responseText = await response.text();
            console.log('Raw response:', responseText);
            
            let result;
            try {
                result = JSON.parse(responseText);
            } catch (parseError) {
                console.error('JSON parse error:', parseError);
                throw new Error('Invalid server response');
            }
            
            console.log('Parsed result:', result);
            
            if (result.success) {
                // Close modal
                closeModal('bookingModal');
                
                // Show success message
                this.showNotification('Booking successful! Your ticket code is: ' + result.ticket_code, 'success');
                
                // Refresh bookings and stats
                await this.loadStats();
                if (document.getElementById('bookingsSection').classList.contains('active')) {
                    await this.loadBookings();
                }
            } else {
                this.showNotification('Booking failed: ' + (result.error || result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('Booking error:', error);
            this.showNotification('Booking failed: ' + error.message, 'error');
        } finally {
            // Restore button state
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    }
    
    showNotification(message, type = 'info') {
        // Remove existing notifications
        const existingNotifications = document.querySelectorAll('.notification');
        existingNotifications.forEach(n => n.remove());
        
        // Create notification
        const notification = document.createElement('div');
        notification.className = 'notification';
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 15px 20px;
            border-radius: 10px;
            color: white;
            font-weight: 500;
            z-index: 10000;
            max-width: 400px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.2);
            animation: slideIn 0.3s ease-out;
        `;
        
        // Set background color based on type
        const colors = {
            success: '#28a745',
            error: '#dc3545',
            info: '#667eea'
        };
        notification.style.background = colors[type] || colors.info;
        notification.textContent = message;
        
        // Add to page
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.style.animation = 'slideOut 0.3s ease-in';
                setTimeout(() => notification.remove(), 300);
            }
        }, 5000);
    }
    
    async loadEventChats() {
        const container = document.getElementById('chatMessages');
        if (!container) return;
        
        try {
            // Get user's booked events for chat access
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=bookings&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.tickets && result.tickets.length > 0) {
                // Show available event chats
                container.innerHTML = `
                    <div style="padding: 20px;">
                        <h4 style="color: #333; margin-bottom: 15px;">Available Event Chats</h4>
                        ${result.tickets.map(ticket => `
                            <div style="background: #f8f9fa; padding: 15px; border-radius: 10px; margin-bottom: 10px; cursor: pointer;" 
                                 onclick="window.dashboard.openEventChat(${ticket.event_id}, '${ticket.event_title}')">
                                <div style="display: flex; justify-content: space-between; align-items: center;">
                                    <div>
                                        <strong style="color: #333;">${ticket.event_title}</strong>
                                        <div style="color: #666; font-size: 0.9rem;">Click to join chat</div>
                                    </div>
                                    <i class="fas fa-comments" style="color: #667eea; font-size: 1.2rem;"></i>
                                </div>
                            </div>
                        `).join('')}
                    </div>
                `;
            } else {
                container.innerHTML = `
                    <div style="text-align: center; padding: 50px; color: #666;">
                        <i class="fas fa-comments" style="font-size: 3rem; margin-bottom: 20px; opacity: 0.3;"></i>
                        <h4>No Event Chats Available</h4>
                        <p>Book an event to access its chat room</p>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading event chats:', error);
            container.innerHTML = `
                <div style="text-align: center; padding: 50px; color: #666;">
                    <i class="fas fa-exclamation-triangle" style="font-size: 3rem; margin-bottom: 20px; color: #dc3545;"></i>
                    <h4>Error Loading Chats</h4>
                    <p>Please try again later</p>
                </div>
            `;
        }
    }
    
    async openEventChat(eventId, eventTitle) {
        const container = document.getElementById('chatMessages');
        this.currentChatEventId = eventId;
        this.currentEventTitle = eventTitle;
        
        // Update chat interface
        container.innerHTML = `
            <div style="display: flex; flex-direction: column; height: 100%;">
                <div style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 15px; border-radius: 10px 10px 0 0; margin-bottom: 10px;">
                    <div style="display: flex; justify-content: space-between; align-items: center;">
                        <div>
                            <h4 style="margin: 0;">${eventTitle} Chat</h4>
                            <small>Event Discussion Room</small>
                        </div>
                        <button onclick="window.dashboard.loadEventChats()" style="background: rgba(255,255,255,0.2); border: none; color: white; padding: 5px 10px; border-radius: 5px; cursor: pointer;">
                            <i class="fas fa-arrow-left"></i> Back
                        </button>
                    </div>
                </div>
                <div id="chatMessagesList" style="flex: 1; overflow-y: auto; padding: 10px; background: #f8f9fa; border-radius: 10px; margin-bottom: 10px; min-height: 300px;">
                    <div style="text-align: center; padding: 20px; color: #666;">
                        <i class="fas fa-spinner fa-spin"></i> Loading messages...
                    </div>
                </div>
            </div>
        `;
        
        // Enable chat input
        const chatInput = document.getElementById('chatInput');
        if (chatInput) {
            chatInput.disabled = false;
            chatInput.placeholder = `Type your message for ${eventTitle}...`;
        }
        
        // First, join or create the chat room for this event
        await this.joinEventChatRoom(eventId);
        
        // Load messages for this event
        await this.loadChatMessages(eventId);
        
        // Start polling for new messages
        this.startChatPolling(eventId);
    }
    
    async joinEventChatRoom(eventId) {
        try {
            // Try the main chat API first
            const response = await fetch(`/GHTouristApp/api/chat.php?endpoint=rooms&event_id=${eventId}&organizer_id=1`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            
            const result = await response.json();
            
            if (result.success && result.rooms && result.rooms.length > 0) {
                // Use existing room
                this.currentChatRoomId = result.rooms[0].id;
                this.useMainChatAPI = true;
            } else {
                // Create new room for this event
                await this.createEventChatRoom(eventId);
            }
            
            // Join the chat room if using main API
            if (this.currentChatRoomId && this.useMainChatAPI) {
                const joinResponse = await fetch('/GHTouristApp/api/chat.php?endpoint=join', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${this.token}`
                    },
                    body: JSON.stringify({
                        room_id: this.currentChatRoomId
                    })
                });
                
                const joinResult = await joinResponse.json();
                console.log('Join room result:', joinResult);
            }
            
        } catch (error) {
            console.error('Main chat API not available, using simple chat:', error);
            // Fallback to simple chat
            this.useMainChatAPI = false;
            this.currentChatRoomId = eventId; // Use event ID as room identifier
        }
    }
    
    async createEventChatRoom(eventId) {
        try {
            const response = await fetch('/GHTouristApp/api/chat.php?endpoint=rooms&organizer_id=1', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    name: `${this.currentEventTitle} Discussion`,
                    description: `Chat room for ${this.currentEventTitle} attendees`,
                    event_id: eventId,
                    room_type: 'public',
                    moderated: false,
                    max_participants: 1000
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.currentChatRoomId = result.room_id;
                console.log('Created chat room:', this.currentChatRoomId);
            } else {
                console.error('Failed to create chat room:', result);
            }
            
        } catch (error) {
            console.error('Error creating chat room:', error);
        }
    }
    
    async loadChatMessages(eventId) {
        try {
            const messagesList = document.getElementById('chatMessagesList');
            if (!messagesList || !this.currentChatRoomId) {
                // Show fallback message if no room available
                if (messagesList) {
                    messagesList.innerHTML = `
                        <div style="text-align: center; padding: 20px; color: #666;">
                            <i class="fas fa-comments" style="font-size: 2rem; margin-bottom: 10px; opacity: 0.3;"></i>
                            <p>Setting up chat room...</p>
                        </div>
                    `;
                }
                return;
            }
            
            let response, result;
            
            if (this.useMainChatAPI) {
                // Use main chat API
                response = await fetch(`/GHTouristApp/api/chat.php?endpoint=messages&room_id=${this.currentChatRoomId}`, {
                    headers: {
                        'Authorization': `Bearer ${this.token}`
                    }
                });
            } else {
                // Use simple chat API
                response = await fetch(`/GHTouristApp/api/tourist.php?action=get-chat-messages&event_id=${eventId}`, {
                    headers: {
                        'Authorization': `Bearer ${this.token}`
                    }
                });
            }
            
            result = await response.json();
            
            if (result.success && result.messages) {
                if (result.messages.length > 0) {
                    messagesList.innerHTML = result.messages.map(msg => this.createMessageElement(msg)).join('');
                } else {
                    messagesList.innerHTML = `
                        <div style="text-align: center; padding: 20px; color: #666;">
                            <i class="fas fa-comments" style="font-size: 2rem; margin-bottom: 10px; opacity: 0.3;"></i>
                            <p>No messages yet. Be the first to say hello!</p>
                        </div>
                    `;
                }
                messagesList.scrollTop = messagesList.scrollHeight;
            } else {
                console.error('Failed to load messages:', result);
            }
            
        } catch (error) {
            console.error('Error loading chat messages:', error);
            const messagesList = document.getElementById('chatMessagesList');
            if (messagesList) {
                messagesList.innerHTML = `
                    <div style="text-align: center; padding: 20px; color: #dc3545;">
                        <i class="fas fa-exclamation-triangle" style="font-size: 2rem; margin-bottom: 10px;"></i>
                        <p>Error loading messages. Please try again.</p>
                    </div>
                `;
            }
        }
    }
    
    createMessageElement(message) {
        const time = new Date(message.created_at).toLocaleTimeString('en-GB', {
            hour: '2-digit',
            minute: '2-digit'
        });
        
        const isSystem = message.message_type === 'system' || message.message_type === 'notification';
        const isCurrentUser = message.sender_name === this.user.full_name || message.user_id == this.user.id;
        
        if (isSystem) {
            return `
                <div style="text-align: center; margin: 10px 0; padding: 8px; background: #e3f2fd; border-radius: 15px; font-style: italic; color: #1976d2;">
                    ${message.content}
                </div>
            `;
        }
        
        return `
            <div style="margin-bottom: 15px; display: flex; ${isCurrentUser ? 'justify-content: flex-end;' : 'justify-content: flex-start;'}">
                <div style="max-width: 70%; ${isCurrentUser ? 'background: linear-gradient(135deg, #667eea, #764ba2); color: white;' : 'background: white; color: #333;'} padding: 10px 15px; border-radius: 18px; box-shadow: 0 2px 5px rgba(0,0,0,0.1);">
                    ${!isCurrentUser ? `<div style="font-weight: bold; font-size: 0.8rem; margin-bottom: 5px; color: #667eea;">${message.sender_name || 'Anonymous'}</div>` : ''}
                    <div>${this.escapeHtml(message.content)}</div>
                    <div style="font-size: 0.7rem; opacity: 0.7; margin-top: 5px; text-align: right;">${time}</div>
                </div>
            </div>
        `;
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    startChatPolling(eventId) {
        // Clear existing polling
        if (this.chatPollingInterval) {
            clearInterval(this.chatPollingInterval);
        }
        
        // Poll for new messages every 3 seconds
        this.chatPollingInterval = setInterval(() => {
            if (this.currentChatEventId === eventId) {
                this.loadChatMessages(eventId);
            } else {
                clearInterval(this.chatPollingInterval);
            }
        }, 3000);
    }
    
    async sendMessage() {
        const chatInput = document.getElementById('chatInput');
        if (!chatInput || !chatInput.value.trim() || !this.currentChatRoomId) {
            if (!this.currentChatRoomId) {
                this.showNotification('Chat room not ready. Please wait...', 'error');
            }
            return;
        }
        
        const message = chatInput.value.trim();
        const originalMessage = message;
        chatInput.value = '';
        
        // Disable input while sending
        chatInput.disabled = true;
        
        try {
            let response, requestBody;
            
            if (this.useMainChatAPI) {
                // Use main chat API
                response = await fetch('/GHTouristApp/api/chat.php?endpoint=messages', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${this.token}`
                    },
                    body: JSON.stringify({
                        room_id: this.currentChatRoomId,
                        content: message,
                        message_type: 'text'
                    })
                });
            } else {
                // Use simple chat API
                response = await fetch('/GHTouristApp/api/tourist.php?action=send-chat-message', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Authorization': `Bearer ${this.token}`
                    },
                    body: JSON.stringify({
                        event_id: this.currentChatEventId,
                        user_id: this.user.id,
                        message: message
                    })
                });
            }
            
            const result = await response.json();
            
            if (result.success) {
                // Message sent successfully, reload messages to show it
                await this.loadChatMessages(this.currentChatEventId);
            } else {
                // Restore message if failed
                chatInput.value = originalMessage;
                this.showNotification('Failed to send message: ' + (result.error || 'Unknown error'), 'error');
            }
            
        } catch (error) {
            console.error('Error sending message:', error);
            // Restore message if failed
            chatInput.value = originalMessage;
            this.showNotification('Failed to send message. Please try again.', 'error');
        } finally {
            // Re-enable input
            chatInput.disabled = false;
            chatInput.focus();
        }
    }
    
    async joinEventChat(eventId) {
        // Switch to chat section and open specific event chat
        this.showSection('chatSection');
        
        // Get event title from bookings
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=bookings&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.tickets) {
                const ticket = result.tickets.find(t => t.event_id == eventId);
                if (ticket) {
                    await this.openEventChat(eventId, ticket.event_title);
                } else {
                    this.showNotification('You need to book this event to join its chat', 'error');
                }
            }
        } catch (error) {
            console.error('Error joining chat:', error);
            this.showNotification('Error joining chat', 'error');
        }
    }
    
    async downloadTicket(ticketCode) {
        try {
            this.showNotification('Preparing your ticket download...', 'info');
            
            // Get ticket details
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=bookings&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.tickets) {
                const ticket = result.tickets.find(t => t.ticket_code === ticketCode);
                if (ticket) {
                    this.generateTicketPDF(ticket);
                } else {
                    this.showNotification('Ticket not found', 'error');
                }
            }
        } catch (error) {
            console.error('Error downloading ticket:', error);
            this.showNotification('Error downloading ticket', 'error');
        }
    }
    
    generateTicketPDF(ticket) {
        // Create a simple HTML ticket that can be printed or saved as PDF
        const ticketHTML = `
            <!DOCTYPE html>
            <html>
            <head>
                <title>Event Ticket - ${ticket.event_title}</title>
                <style>
                    body { font-family: Arial, sans-serif; margin: 0; padding: 20px; background: #f5f5f5; }
                    .ticket { background: white; max-width: 600px; margin: 0 auto; border-radius: 15px; overflow: hidden; box-shadow: 0 10px 30px rgba(0,0,0,0.1); }
                    .ticket-header { background: linear-gradient(135deg, #667eea, #764ba2); color: white; padding: 30px; text-align: center; }
                    .ticket-body { padding: 30px; }
                    .ticket-title { font-size: 2rem; font-weight: bold; margin-bottom: 10px; }
                    .ticket-subtitle { font-size: 1.1rem; opacity: 0.9; }
                    .ticket-info { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 20px 0; }
                    .info-item { padding: 15px; background: #f8f9fa; border-radius: 10px; }
                    .info-label { font-weight: bold; color: #667eea; font-size: 0.9rem; margin-bottom: 5px; }
                    .info-value { color: #333; font-size: 1.1rem; }
                    .ticket-code { text-align: center; margin: 30px 0; padding: 20px; background: #e8f4fd; border-radius: 10px; }
                    .code-label { color: #667eea; font-weight: bold; margin-bottom: 10px; }
                    .code-value { font-size: 1.5rem; font-weight: bold; color: #333; font-family: monospace; letter-spacing: 2px; }
                    .ticket-footer { text-align: center; padding: 20px; background: #f8f9fa; color: #666; font-size: 0.9rem; }
                    .qr-placeholder { width: 100px; height: 100px; background: #ddd; margin: 0 auto 15px; border-radius: 10px; display: flex; align-items: center; justify-content: center; color: #999; }
                </style>
            </head>
            <body>
                <div class="ticket">
                    <div class="ticket-header">
                        <div class="ticket-title">${ticket.event_title}</div>
                        <div class="ticket-subtitle">Event Ticket</div>
                    </div>
                    <div class="ticket-body">
                        <div class="ticket-info">
                            <div class="info-item">
                                <div class="info-label">Event Date</div>
                                <div class="info-value">${new Date(ticket.event_date || ticket.start_datetime).toLocaleDateString('en-GB', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}</div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">Event Time</div>
                                <div class="info-value">${new Date(ticket.event_date || ticket.start_datetime).toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit' })}</div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">Venue</div>
                                <div class="info-value">${ticket.venue_name || 'Venue TBA'}</div>
                            </div>
                            <div class="info-item">
                                <div class="info-label">Ticket Type</div>
                                <div class="info-value">${ticket.ticket_type || 'Regular'}</div>
                            </div>
                        </div>
                        <div class="ticket-code">
                            <div class="qr-placeholder">QR CODE</div>
                            <div class="code-label">Ticket Code</div>
                            <div class="code-value">${ticket.ticket_code}</div>
                        </div>
                    </div>
                    <div class="ticket-footer">
                        <p>Present this ticket at the venue entrance. Keep this ticket safe and do not share the ticket code.</p>
                        <p>Generated on ${new Date().toLocaleDateString('en-GB')} | Ghana Events Platform</p>
                    </div>
                </div>
            </body>
            </html>
        `;
        
        // Open in new window for printing/saving
        const printWindow = window.open('', '_blank');
        printWindow.document.write(ticketHTML);
        printWindow.document.close();
        
        // Auto-trigger print dialog
        setTimeout(() => {
            printWindow.print();
        }, 500);
        
        this.showNotification('Ticket opened in new window. Use Ctrl+P to print or save as PDF.', 'success');
    }
    
    async testAPI() {
        try {
            const response = await fetch('/GHTouristApp/api/tourist.php?action=test', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({ test: 'data' })
            });
            
            const result = await response.json();
            console.log('API Test Result:', result);
            return result;
        } catch (error) {
            console.error('API Test Error:', error);
            return null;
        }
    }
    
    async submitSafetyReport() {
        const form = document.getElementById('safetyReportForm');
        if (!form) return;
        
        const formData = new FormData(form);
        const reportData = {
            user_id: this.user.id,
            type: document.getElementById('issueType').value,
            location: document.getElementById('issueLocation').value,
            description: document.getElementById('issueDescription').value,
            urgency: document.getElementById('urgencyLevel').value
        };
        
        // Validate required fields
        if (!reportData.type || !reportData.location || !reportData.description || !reportData.urgency) {
            this.showNotification('Please fill in all required fields', 'error');
            return;
        }
        
        // Show loading state
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Submitting...';
        submitBtn.disabled = true;
        
        try {
            const response = await fetch('/GHTouristApp/api/tourist.php?action=safety-report', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify(reportData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Close modal
                closeModal('safetyReportModal');
                
                // Show success message
                this.showNotification('Safety report submitted successfully. Thank you for helping keep our events safe!', 'success');
                
                // Refresh safety reports if we're on that section
                if (document.getElementById('safetySection').classList.contains('active')) {
                    this.loadSafetyReports();
                }
            } else {
                this.showNotification('Failed to submit report: ' + (result.error || result.message || 'Unknown error'), 'error');
            }
        } catch (error) {
            console.error('Safety report error:', error);
            this.showNotification('Failed to submit report. Please try again.', 'error');
        } finally {
            // Restore button state
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    }
    
    async loadSafetyReports() {
        const container = document.getElementById('safetyReports');
        if (!container) return;
        
        container.innerHTML = '<div style="text-align: center; padding: 40px; color: #666;"><i class="fas fa-spinner fa-spin"></i> Loading safety reports...</div>';
        
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=safety-reports&user_id=${this.user.id}`, {
                headers: {
                    'Authorization': `Bearer ${this.token}`
                }
            });
            const result = await response.json();
            
            if (result.success && result.reports && result.reports.length > 0) {
                container.innerHTML = result.reports.map(report => this.createSafetyReportCard(report)).join('');
            } else {
                container.innerHTML = `
                    <div style="text-align: center; padding: 50px; color: #666;">
                        <i class="fas fa-shield-alt" style="font-size: 3rem; margin-bottom: 20px; opacity: 0.3;"></i>
                        <h4>No Safety Reports</h4>
                        <p>You haven't submitted any safety reports yet.</p>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading safety reports:', error);
            container.innerHTML = `
                <div style="text-align: center; padding: 50px; color: #dc3545;">
                    <i class="fas fa-exclamation-triangle" style="font-size: 3rem; margin-bottom: 20px;"></i>
                    <h4>Error Loading Reports</h4>
                    <p>Please try again later</p>
                </div>
            `;
        }
    }
    
    createSafetyReportCard(report) {
        const statusColors = {
            'reported': '#ffc107',
            'acknowledged': '#17a2b8',
            'in_progress': '#fd7e14',
            'resolved': '#28a745',
            'false_alarm': '#6c757d'
        };
        
        const urgencyColors = {
            'low': '#28a745',
            'medium': '#ffc107',
            'high': '#fd7e14',
            'critical': '#dc3545'
        };
        
        return `
            <div style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                    <div>
                        <h4 style="color: #333; margin-bottom: 5px;">${report.report_type.replace('_', ' ').charAt(0).toUpperCase() + report.report_type.replace('_', ' ').slice(1)} Issue</h4>
                        <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> ${report.location_description || 'No location specified'}</p>
                    </div>
                    <div style="text-align: right;">
                        <span style="background: ${statusColors[report.status] || '#6c757d'}; color: white; padding: 4px 12px; border-radius: 20px; font-size: 0.8rem;">
                            ${report.status.replace('_', ' ').charAt(0).toUpperCase() + report.status.replace('_', ' ').slice(1)}
                        </span>
                        <div style="margin-top: 5px;">
                            <span style="background: ${urgencyColors[report.severity] || '#6c757d'}; color: white; padding: 2px 8px; border-radius: 10px; font-size: 0.7rem;">
                                ${report.severity.toUpperCase()}
                            </span>
                        </div>
                    </div>
                </div>
                <p style="color: #555; margin-bottom: 15px;">${report.description}</p>
                <div style="display: flex; justify-content: space-between; align-items: center; color: #888; font-size: 0.9rem;">
                    <span><i class="fas fa-calendar"></i> ${new Date(report.created_at).toLocaleDateString()}</span>
                    <span>Report #${report.id}</span>
                </div>
            </div>
        `;
    }
    
    logout() {
        // Show logout notification
        this.showNotification('Logging out...', 'info');
        
        // Clear any active intervals
        if (this.chatPollingInterval) {
            clearInterval(this.chatPollingInterval);
        }
        
        // Clear localStorage
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        
        // Small delay to show the notification
        setTimeout(() => {
            window.location.href = 'index.html';
        }, 500);
    }

    // Media sharing functionality
    toggleMediaOptions() {
        const mediaOptions = document.getElementById('mediaOptions');
        if (mediaOptions) {
            mediaOptions.style.display = mediaOptions.style.display === 'none' ? 'flex' : 'none';
        }
    }

    selectMediaType(type) {
        const fileInput = document.getElementById('mediaFileInput');
        if (!fileInput) return;

        // Set appropriate file filters based on type
        switch (type) {
            case 'image':
                fileInput.accept = 'image/*';
                break;
            case 'video':
                fileInput.accept = 'video/*';
                break;
            case 'file':
                fileInput.accept = '.pdf,.doc,.docx,.txt,.zip,.rar';
                break;
            default:
                fileInput.accept = '*/*';
        }

        // Trigger file selection
        fileInput.click();
        
        // Hide media options
        this.toggleMediaOptions();
    }

    handleFileSelection(event) {
        const files = Array.from(event.target.files);
        if (files.length === 0) return;

        // Validate files
        const validFiles = this.validateFiles(files);
        if (validFiles.length === 0) return;

        // Store selected files
        this.selectedFiles = validFiles;

        // Show preview
        this.showMediaPreview(validFiles);
    }

    validateFiles(files) {
        const maxFileSize = 10 * 1024 * 1024; // 10MB
        const allowedTypes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp',
            'video/mp4', 'video/webm', 'video/quicktime',
            'application/pdf', 'application/msword', 
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'text/plain', 'application/zip', 'application/x-rar-compressed'
        ];

        const validFiles = [];

        for (const file of files) {
            // Check file size
            if (file.size > maxFileSize) {
                this.showNotification(`File "${file.name}" is too large. Maximum size is 10MB.`, 'error');
                continue;
            }

            // Check file type
            if (!allowedTypes.includes(file.type)) {
                this.showNotification(`File type "${file.type}" is not supported.`, 'error');
                continue;
            }

            validFiles.push(file);
        }

        return validFiles;
    }

    showMediaPreview(files) {
        const preview = document.getElementById('mediaPreview');
        if (!preview) return;

        preview.style.display = 'block';
        preview.innerHTML = '<h5 style="margin-bottom: 15px; color: #333;">Selected Media:</h5>';

        files.forEach((file, index) => {
            const previewItem = document.createElement('div');
            previewItem.className = 'media-preview-item';

            if (file.type.startsWith('image/')) {
                const img = document.createElement('img');
                img.className = 'media-preview-image';
                img.src = URL.createObjectURL(file);
                previewItem.appendChild(img);
            } else if (file.type.startsWith('video/')) {
                const video = document.createElement('video');
                video.className = 'media-preview-video';
                video.src = URL.createObjectURL(file);
                video.controls = false;
                video.muted = true;
                previewItem.appendChild(video);
            } else {
                const fileDiv = document.createElement('div');
                fileDiv.className = 'media-preview-file';
                fileDiv.innerHTML = `
                    <i class="fas fa-file" style="font-size: 1.5rem; margin-bottom: 5px;"></i>
                    <div style="font-size: 0.7rem; text-align: center;">${file.name.substring(0, 15)}${file.name.length > 15 ? '...' : ''}</div>
                `;
                previewItem.appendChild(fileDiv);
            }

            // Add remove button
            const removeBtn = document.createElement('button');
            removeBtn.className = 'media-remove-btn';
            removeBtn.innerHTML = '×';
            removeBtn.onclick = () => this.removeFile(index);
            previewItem.appendChild(removeBtn);

            preview.appendChild(previewItem);
        });

        // Add send button
        const sendBtn = document.createElement('button');
        sendBtn.style.cssText = `
            margin-top: 15px;
            padding: 10px 20px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 10px;
            cursor: pointer;
            width: 100%;
        `;
        sendBtn.innerHTML = '<i class="fas fa-paper-plane"></i> Send Media';
        sendBtn.onclick = () => this.sendMediaMessage();
        preview.appendChild(sendBtn);
    }

    removeFile(index) {
        if (this.selectedFiles && this.selectedFiles.length > index) {
            this.selectedFiles.splice(index, 1);
            
            if (this.selectedFiles.length === 0) {
                const preview = document.getElementById('mediaPreview');
                if (preview) {
                    preview.style.display = 'none';
                    preview.innerHTML = '';
                }
                this.selectedFiles = null;
            } else {
                this.showMediaPreview(this.selectedFiles);
            }
        }
    }

    async sendMediaMessage() {
        if (!this.selectedFiles || this.selectedFiles.length === 0) {
            this.showNotification('No files selected', 'error');
            return;
        }

        if (!this.currentChatRoomId) {
            this.showNotification('Chat room not ready. Please wait...', 'error');
            return;
        }

        // Show uploading state
        const preview = document.getElementById('mediaPreview');
        const sendBtn = preview.querySelector('button');
        if (sendBtn) {
            sendBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Uploading...';
            sendBtn.disabled = true;
        }

        try {
            // Upload files one by one
            for (const file of this.selectedFiles) {
                await this.uploadAndSendFile(file);
            }

            // Clear preview and selected files
            this.selectedFiles = null;
            if (preview) {
                preview.style.display = 'none';
                preview.innerHTML = '';
            }

            // Reload messages to show uploaded media
            await this.loadChatMessages(this.currentChatEventId);

            this.showNotification('Media sent successfully!', 'success');

        } catch (error) {
            console.error('Error sending media:', error);
            this.showNotification('Failed to send media. Please try again.', 'error');
        }
    }

    async uploadAndSendFile(file) {
        const formData = new FormData();
        formData.append('file', file);
        formData.append('event_id', this.currentChatEventId);
        formData.append('user_id', this.user.id);
        formData.append('room_id', this.currentChatRoomId);
        formData.append('message_type', this.getMessageType(file.type));

        const response = await fetch('/GHTouristApp/api/chat-media.php', {
            method: 'POST',
            headers: {
                'Authorization': `Bearer ${this.token}`
            },
            body: formData
        });

        const result = await response.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Upload failed');
        }

        return result;
    }

    getMessageType(fileType) {
        if (fileType.startsWith('image/')) return 'image';
        if (fileType.startsWith('video/')) return 'video';
        return 'file';
    }

    createMessageElement(message) {
        const time = new Date(message.created_at).toLocaleTimeString('en-GB', {
            hour: '2-digit',
            minute: '2-digit'
        });
        
        const isSystem = message.message_type === 'system' || message.message_type === 'notification';
        const isCurrentUser = message.sender_name === this.user.full_name || message.user_id == this.user.id;
        
        if (isSystem) {
            return `
                <div style="text-align: center; margin: 10px 0; padding: 8px; background: #e3f2fd; border-radius: 15px; font-style: italic; color: #1976d2;">
                    ${message.content}
                </div>
            `;
        }

        let mediaContent = '';
        
        // Handle media messages
        if (message.message_type === 'image' && message.file_url) {
            mediaContent = `
                <div class="chat-message-media">
                    <img src="${message.file_url}" alt="Shared image" class="chat-message-image" onclick="openImageModal('${message.file_url}')">
                </div>
            `;
        } else if (message.message_type === 'video' && message.file_url) {
            mediaContent = `
                <div class="chat-message-media">
                    <video src="${message.file_url}" class="chat-message-video" controls></video>
                </div>
            `;
        } else if (message.message_type === 'file' && message.file_url) {
            const fileName = message.file_name || 'Download File';
            const fileSize = message.file_size ? this.formatFileSize(message.file_size) : '';
            mediaContent = `
                <div class="chat-message-file" onclick="downloadFile('${message.file_url}', '${fileName}')">
                    <div class="file-icon">
                        <i class="fas fa-file"></i>
                    </div>
                    <div class="file-info">
                        <div class="file-name">${fileName}</div>
                        ${fileSize ? `<div class="file-size">${fileSize}</div>` : ''}
                    </div>
                    <i class="fas fa-download"></i>
                </div>
            `;
        }
        
        return `
            <div style="margin-bottom: 15px; display: flex; ${isCurrentUser ? 'justify-content: flex-end;' : 'justify-content: flex-start;'}">
                <div style="max-width: 70%; ${isCurrentUser ? 'background: linear-gradient(135deg, #667eea, #764ba2); color: white;' : 'background: white; color: #333;'} padding: 10px 15px; border-radius: 18px; box-shadow: 0 2px 5px rgba(0,0,0,0.1);">
                    ${!isCurrentUser ? `<div style="font-weight: bold; font-size: 0.8rem; margin-bottom: 5px; color: #667eea;">${message.sender_name || 'Anonymous'}</div>` : ''}
                    ${mediaContent}
                    ${message.content ? `<div>${this.escapeHtml(message.content)}</div>` : ''}
                    <div style="font-size: 0.7rem; opacity: 0.7; margin-top: 5px; text-align: right;">${time}</div>
                </div>
            </div>
        `;
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
}

// Global functions for HTML onclick handlers
function showSection(sectionId) {
    if (window.dashboard) {
        window.dashboard.showSection(sectionId);
    }
}

function bookEvent(eventId, eventTitle, price) {
    // Store current booking details
    window.currentBooking = { eventId, eventTitle, price };
    
    // Update modal title and price display
    const modal = document.getElementById('bookingModal');
    const modalTitle = modal.querySelector('.modal-title');
    if (modalTitle) {
        modalTitle.textContent = `Book Ticket - ${eventTitle}`;
    }
    
    // Update ticket type options with current event price
    const ticketType = document.getElementById('ticketType');
    if (ticketType) {
        ticketType.innerHTML = `
            <option value="regular">Regular - GHS ${price}</option>
            <option value="vip">VIP - GHS ${Math.round(price * 1.5)}</option>
            <option value="early_bird">Early Bird - GHS ${Math.round(price * 0.8)}</option>
        `;
    }
    
    // Show the modal
    modal.style.display = 'block';
    
    // Update total price when quantity or ticket type changes
    updateBookingTotal();
}

function downloadTicket(ticketCode) {
    if (window.dashboard) {
        window.dashboard.downloadTicket(ticketCode);
    }
}

function joinEventChat(eventId) {
    if (window.dashboard) {
        window.dashboard.joinEventChat(eventId);
    }
}

function closeModal(modalId) {
    const modal = document.getElementById(modalId);
    if (modal) {
        modal.style.display = 'none';
    }
}

function updateBookingTotal() {
    const ticketType = document.getElementById('ticketType');
    const quantity = document.getElementById('ticketQuantity');
    
    if (!ticketType || !quantity || !window.currentBooking) return;
    
    const selectedOption = ticketType.options[ticketType.selectedIndex];
    const priceText = selectedOption.text.match(/GHS (\d+)/);
    const price = priceText ? parseInt(priceText[1]) : window.currentBooking.price;
    const total = price * parseInt(quantity.value);
    
    // Update button text with total
    const submitBtn = document.querySelector('#bookingForm button[type="submit"]');
    if (submitBtn && !submitBtn.disabled) {
        submitBtn.innerHTML = `<i class="fas fa-credit-card"></i> Pay GHS ${total}`;
    }
}

function sendMessage() {
    if (window.dashboard) {
        window.dashboard.sendMessage();
    }
}

function logout() {
    // Optional: Add confirmation dialog
    if (confirm('Are you sure you want to logout?')) {
        if (window.dashboard) {
            window.dashboard.logout();
        } else {
            // Fallback logout if dashboard not available
            localStorage.removeItem('auth_token');
            localStorage.removeItem('user_data');
            window.location.href = 'index.html';
        }
    }
}

// Media sharing global functions
function toggleMediaOptions() {
    if (window.dashboard) {
        window.dashboard.toggleMediaOptions();
    }
}

function selectMediaType(type) {
    if (window.dashboard) {
        window.dashboard.selectMediaType(type);
    }
}

function handleFileSelection(event) {
    if (window.dashboard) {
        window.dashboard.handleFileSelection(event);
    }
}

function openImageModal(imageUrl) {
    const modal = document.getElementById('imageModal');
    const modalImage = document.getElementById('modalImage');
    
    if (modal && modalImage) {
        modalImage.src = imageUrl;
        modal.style.display = 'block';
    }
}

function downloadFile(fileUrl, fileName) {
    const link = document.createElement('a');
    link.href = fileUrl;
    link.download = fileName;
    link.target = '_blank';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function showReportModal() {
    const modal = document.getElementById('safetyReportModal');
    if (modal) {
        modal.style.display = 'block';
        
        // Reset form
        const form = document.getElementById('safetyReportForm');
        if (form) {
            form.reset();
        }
        
        // Load events into dropdown
        if (window.dashboard) {
            window.dashboard.loadEventsForReport();
        }
    }
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.dashboard = new TouristDashboard();
    
    // Add event listeners for booking modal
    const ticketType = document.getElementById('ticketType');
    const ticketQuantity = document.getElementById('ticketQuantity');
    
    if (ticketType) {
        ticketType.addEventListener('change', updateBookingTotal);
    }
    
    if (ticketQuantity) {
        ticketQuantity.addEventListener('input', updateBookingTotal);
    }
    
    // Add Enter key support for chat input
    const chatInput = document.getElementById('chatInput');
    if (chatInput) {
        chatInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
    }
    
    // Add safety report form submission
    const safetyReportForm = document.getElementById('safetyReportForm');
    if (safetyReportForm) {
        safetyReportForm.addEventListener('submit', (e) => {
            e.preventDefault();
            if (window.dashboard) {
                window.dashboard.submitSafetyReport();
            }
        });
    }
});
