// Tourist Dashboard JavaScript
class TouristDashboard {
    constructor() {
        this.token = localStorage.getItem('auth_token');
        this.user = JSON.parse(localStorage.getItem('user_data') || '{}');
        this.currentEventId = null;
        this.init();
    }

    async init() {
        if (!this.token || !this.user.id) {
            window.location.href = 'index.html';
            return;
        }

        this.updateUserInfo();
        await this.loadDashboardData();
        this.setupEventListeners();
    }

    updateUserInfo() {
        const userName = document.getElementById('userName');
        const welcomeName = document.getElementById('welcomeName');
        const userAvatar = document.getElementById('userAvatar');

        if (this.user.full_name) {
            userName.textContent = this.user.full_name;
            welcomeName.textContent = this.user.full_name.split(' ')[0];
            
            // Create initials for avatar
            const initials = this.user.full_name
                .split(' ')
                .map(name => name.charAt(0))
                .join('')
                .toUpperCase()
                .substring(0, 2);
            userAvatar.textContent = initials;
        }

        // Update profile form
        document.getElementById('profileName').value = this.user.full_name || '';
        document.getElementById('profileEmail').value = this.user.email || '';
        document.getElementById('profilePhone').value = this.user.phone || '';
    }

    async loadDashboardData() {
        try {
            await Promise.all([
                this.loadStats(),
                this.loadRecentActivities(),
                this.loadEvents(),
                this.loadBookings()
            ]);
        } catch (error) {
            console.error('Error loading dashboard data:', error);
        }
    }

    async loadStats() {
        try {
            // Load user statistics
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=stats&user_id=${this.user.id}`);
            const result = await response.json();
            
            if (result.success) {
                document.getElementById('totalBookings').textContent = result.stats.total_bookings || 0;
                document.getElementById('upcomingEvents').textContent = result.stats.upcoming_events || 0;
                document.getElementById('accommodationBookings').textContent = result.stats.accommodation_bookings || 0;
                document.getElementById('reviewsGiven').textContent = result.stats.reviews_given || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
            // Set default values
            document.getElementById('totalBookings').textContent = '5';
            document.getElementById('upcomingEvents').textContent = '3';
            document.getElementById('accommodationBookings').textContent = '2';
            document.getElementById('reviewsGiven').textContent = '8';
        }
    }

    async loadRecentActivities() {
        try {
            const container = document.getElementById('recentActivities');
            
            // Sample recent activities
            const activities = [
                {
                    type: 'booking',
                    title: 'Ghana Music Festival',
                    description: 'Booked VIP ticket',
                    date: '2025-09-15',
                    icon: 'fas fa-ticket-alt'
                },
                {
                    type: 'review',
                    title: 'Cultural Heritage Day',
                    description: 'Left a 5-star review',
                    date: '2025-09-14',
                    icon: 'fas fa-star'
                }
            ];

            container.innerHTML = activities.map(activity => `
                <div class="activity-item">
                    <div class="activity-icon">
                        <i class="${activity.icon}"></i>
                    </div>
                    <div class="activity-content">
                        <div class="activity-title">${activity.title}</div>
                        <div class="activity-description">${activity.description}</div>
                        <div class="activity-date">${new Date(activity.date).toLocaleDateString()}</div>
                    </div>
                </div>
            `).join('');
        } catch (error) {
            console.error('Error loading recent activities:', error);
        }
    }

    setupEventListeners() {
        // Sidebar navigation
        document.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const sectionId = link.getAttribute('onclick')?.match(/showSection\('(.+?)'\)/)?.[1];
                if (sectionId) {
                    this.showSection(sectionId);
                }
            });
        });

        // Profile form
        const profileForm = document.getElementById('profileForm');
        if (profileForm) {
            profileForm.addEventListener('submit', (e) => this.updateProfile(e));
        }

        // Logout
        const logoutBtn = document.getElementById('logoutBtn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => this.logout());
        }
    }

    showSection(sectionId) {
        // Hide all sections
        document.querySelectorAll('.content-section').forEach(section => {
            section.style.display = 'none';
        });

        // Show selected section
        const targetSection = document.getElementById(sectionId);
        if (targetSection) {
            targetSection.style.display = 'block';
        }

        // Update active nav
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });

        const activeLink = document.querySelector(`[onclick*="${sectionId}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }

        // Load section-specific data
        switch(sectionId) {
            case 'discoverEventsSection':
                this.loadEvents();
                break;
            case 'myBookingsSection':
                this.loadBookings();
                break;
        }
    }

    async updateProfile(e) {
        e.preventDefault();
        
        const formData = new FormData(e.target);
        const profileData = {
            full_name: formData.get('full_name'),
            email: formData.get('email'),
            phone: formData.get('phone')
        };

        try {
            const response = await fetch('/GHTouristApp/api/tourist.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${this.token}`
                },
                body: JSON.stringify({
                    action: 'update_profile',
                    ...profileData
                })
            });

            const result = await response.json();
            
            if (result.success) {
                // Update local user data
                this.user = { ...this.user, ...profileData };
                localStorage.setItem('user_data', JSON.stringify(this.user));
                this.updateUserInfo();
                
                alert('Profile updated successfully!');
            } else {
                alert('Error updating profile: ' + result.message);
            }
        } catch (error) {
            console.error('Error updating profile:', error);
            alert('Error updating profile. Please try again.');
        }
    }

    logout() {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        window.location.href = 'index.html';
    }
}

// Global functions for HTML onclick handlers
function showSection(sectionId) {
    if (window.dashboard) {
        window.dashboard.showSection(sectionId);
    }
}

function bookEvent(eventId, eventTitle, price) {
    alert(`Booking ${eventTitle} for GHS ${price}. This feature will be implemented soon!`);
}

function downloadTicket(ticketCode) {
    alert(`Downloading ticket ${ticketCode}. This feature will be implemented soon!`);
}

function joinEventChat(eventId) {
    alert(`Joining chat for event ${eventId}. This feature will be implemented soon!`);
}

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.dashboard = new TouristDashboard();
});
        const welcomeName = document.getElementById('welcomeName');
        const userAvatar = document.getElementById('userAvatar');

        if (this.user.full_name) {
            userName.textContent = this.user.full_name;
            welcomeName.textContent = this.user.full_name.split(' ')[0];
            
            // Create initials for avatar
            const initials = this.user.full_name
                .split(' ')
                .map(name => name.charAt(0))
                .join('')
                .toUpperCase()
                .substring(0, 2);
            userAvatar.textContent = initials;
        }

        // Update profile form
        document.getElementById('profileName').value = this.user.full_name || '';
        document.getElementById('profileEmail').value = this.user.email || '';
        document.getElementById('profilePhone').value = this.user.phone || '';
    }

    async loadDashboardData() {
        try {
            await Promise.all([
                this.loadStats(),
                this.loadRecentActivities(),
                this.loadEvents(),
                this.loadBookings()
            ]);
        } catch (error) {
            console.error('Error loading dashboard data:', error);
        }
    }

    async loadStats() {
        try {
            // Load user statistics
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=stats&user_id=${this.user.id}`);
            const result = await response.json();
            
            if (result.success) {
                document.getElementById('totalBookings').textContent = result.stats.total_bookings || 0;
                document.getElementById('upcomingEvents').textContent = result.stats.upcoming_events || 0;
                document.getElementById('accommodationBookings').textContent = result.stats.accommodation_bookings || 0;
                document.getElementById('reviewsGiven').textContent = result.stats.reviews_given || 0;
            }
        } catch (error) {
            console.error('Error loading stats:', error);
            // Set default values
            document.getElementById('totalBookings').textContent = '5';
            document.getElementById('upcomingEvents').textContent = '3';
            document.getElementById('accommodationBookings').textContent = '2';
            document.getElementById('reviewsGiven').textContent = '8';
        }
    }

    async loadRecentActivities() {
        try {
            const container = document.getElementById('recentActivities');
            
            // Sample recent activities
            const activities = [
                {
                    type: 'booking',
                    title: 'Ghana Music Festival',
                    description: 'Booked VIP ticket',
                    date: '2025-09-15',
                    icon: 'fas fa-ticket-alt'
                },
                {
                    type: 'review',
                    title: 'Cultural Heritage Day',
                    description: 'Left a 5-star review',
                    date: '2025-09-14',
                    icon: 'fas fa-star'
                }
            ];

            container.innerHTML = activities.map(activity => `
                <div class="activity-item">
                    <div class="activity-icon">
                        <i class="${activity.icon}"></i>
                    </div>
                    <div class="activity-content">
                        <div class="activity-title">${activity.title}</div>
                        <div class="activity-description">${activity.description}</div>
                        <div class="activity-date">${new Date(activity.date).toLocaleDateString()}</div>
                    </div>
                </div>
            `).join('');
        } catch (error) {
            console.error('Error loading recent activities:', error);
        }
    }

    async loadEvents() {
        try {
            const response = await fetch('/GHTouristApp/api/events.php?action=list');
            const result = await response.json();
            
            const container = document.getElementById('eventsGrid');
            if (!container) return;
            
            if (result.success && result.events && result.events.length > 0) {
                container.innerHTML = result.events.map(event => `
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">${event.title}</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> ${event.venue_name || 'TBA'}</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> ${new Date(event.start_datetime).toLocaleDateString()}
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> ${new Date(event.start_datetime).toLocaleTimeString()}
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${event.ticket_price || 50}</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">${event.description || 'Join us for an amazing event experience!'}</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">${event.category_name || 'Event'}</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">${event.approval_status || 'Open'}</span>
                        </div>
                        <button onclick="bookEvent(${event.id}, '${event.title}', ${event.ticket_price || 50})" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                `).join('');
            } else {
                // Show sample events with Ghana events data
                container.innerHTML = `
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Accra Conference Centre</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 15, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Experience the best of Ghanaian music with top artists performing live!</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Music</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Live Performance</span>
                        </div>
                        <button onclick="bookEvent(1, 'Ghana Music Festival', 150)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Labadi Beach Hotel</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 20, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Celebrate Ghana's rich cultural heritage with traditional performances and exhibitions.</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Cultural</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Family Friendly</span>
                        </div>
                        <button onclick="bookEvent(2, 'Homowo Festival', 75)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                    <div class="event-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Tech Innovation Summit</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-map-marker-alt"></i> Kumasi Cultural Centre</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> January 10, 2026
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 9:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 200</div>
                                <div style="color: #666; font-size: 0.9rem;">per ticket</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Join industry leaders discussing the future of technology in Ghana and Africa.</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Technology</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Business</span>
                        </div>
                        <button onclick="bookEvent(3, 'Tech Innovation Summit', 200)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            <i class="fas fa-ticket-alt"></i> Book Ticket
                        </button>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading events:', error);
            const container = document.getElementById('eventsGrid');
            if (container) {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-calendar-times" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>Unable to Load Events</h3>
                        <p>Please check your connection and try again.</p>
                        <button onclick="dashboard.loadEvents()" style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; margin-top: 15px;">
                            Retry
                        </button>
                    </div>
                `;
            }
        }
    }

    async loadBookings() {
        try {
            const response = await fetch(`/GHTouristApp/api/tickets.php?action=user-tickets&user_id=${this.user.id}`);
            const result = await response.json();
            
            const container = document.getElementById('bookingsGrid');
            if (!container) return;
            
            if (result.success && result.tickets && result.tickets.length > 0) {
                container.innerHTML = result.tickets.map(ticket => `
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">${ticket.event_title}</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> ${ticket.ticket_type} Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> ${new Date(ticket.event_date).toLocaleDateString()}
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> ${new Date(ticket.event_date).toLocaleTimeString()}
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS ${ticket.price_paid}</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: ${ticket.payment_status === 'paid' ? '#e8f5e8' : '#fff3cd'}; color: ${ticket.payment_status === 'paid' ? '#28a745' : '#856404'}; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-${ticket.payment_status === 'paid' ? 'check-circle' : 'clock'}"></i> ${ticket.payment_status}
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> ${ticket.ticket_code}
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('${ticket.ticket_code}')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(${ticket.event_id})" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                `).join('');
            } else {
                // Show sample bookings
                container.innerHTML = `
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Ghana Music Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> VIP Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 15, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 6:00 PM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 150</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-check-circle"></i> Confirmed
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> GMF2025001
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('GMF2025001')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(1)" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                    <div class="booking-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Homowo Festival</h3>
                                <p style="color: #666; margin-bottom: 10px;"><i class="fas fa-ticket-alt"></i> Regular Ticket</p>
                                <div style="color: #667eea; margin-bottom: 10px;">
                                    <i class="fas fa-calendar"></i> December 20, 2025
                                    <i class="fas fa-clock" style="margin-left: 15px;"></i> 10:00 AM
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.2rem; font-weight: bold;">GHS 75</div>
                                <div style="color: #666; font-size: 0.9rem;">paid</div>
                            </div>
                        </div>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f5e8; color: #28a745; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-check-circle"></i> Confirmed
                            </span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                <i class="fas fa-hashtag"></i> HMW2025002
                            </span>
                        </div>
                        <div style="display: flex; gap: 10px;">
                            <button onclick="downloadTicket('HMW2025002')" 
                                    style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-download"></i> Download
                            </button>
                            <button onclick="joinEventChat(2)" 
                                    style="background: #28a745; color: white; border: none; padding: 8px 15px; border-radius: 8px; cursor: pointer; flex: 1;">
                                <i class="fas fa-comments"></i> Chat
                            </button>
                        </div>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading bookings:', error);
            const container = document.getElementById('bookingsGrid');
            if (container) {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-ticket-alt" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>No Bookings Yet</h3>
                        <p>Book your first event to see your tickets here!</p>
                        <button onclick="showSection('discoverEventsSection')" style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; margin-top: 15px;">
                            Discover Events
                        </button>
                    </div>
                `;
        }
    }

    async loadAccommodation() {
        try {
            const response = await fetch('/GHTouristApp/api/accommodations.php?action=list');
            const result = await response.json();
            
            const container = document.getElementById('accommodationGrid');
            
            if (result.success && result.data) {
                container.innerHTML = result.data.map(hotel => `
                    <div class="event-card">
                        <div class="event-image">
                            <i class="fas fa-bed"></i>
                        </div>
                        <div class="event-content">
                            <div class="event-title">${hotel.name}</div>
                            <div class="event-details">
                                <i class="fas fa-map-marker-alt"></i> ${hotel.location}
                            </div>
                            <div class="event-details">
                                <i class="fas fa-star"></i> ${hotel.rating}/5 Rating
                            </div>
                            <div class="event-details">
                                <i class="fas fa-wifi"></i> ${hotel.amenities || 'Standard amenities'}
                            </div>
                            <div class="event-details">
                                <i class="fas fa-bed"></i> ${hotel.available_rooms || 0} rooms available
                            </div>
                            <div class="event-price">GHS ${hotel.price_per_night}/night</div>
                            <button class="btn btn-primary" onclick="bookAccommodation(${hotel.id}, '${hotel.name}', ${hotel.price_per_night})">
                                <i class="fas fa-bed"></i> Book Room
                            </button>
                        </div>
                    </div>
                `).join('');
            } else {
                container.innerHTML = '<p style="text-align: center; color: #666;">No accommodations available at the moment.</p>';
            }
        } catch (error) {
            console.error('Error loading accommodations:', error);
            // Fallback to sample data
            const container = document.getElementById('accommodationGrid');
            container.innerHTML = `
                <div class="event-card">
                    <div class="event-image">
                        <i class="fas fa-bed"></i>
                    </div>
                    <div class="event-content">
                        <div class="event-title">Labadi Beach Hotel</div>
                        <div class="event-details">
                            <i class="fas fa-map-marker-alt"></i> Labadi, Accra
                        </div>
                        <div class="event-details">
                            <i class="fas fa-star"></i> 4.5/5 Rating
                        </div>
                        <div class="event-details">
                            <i class="fas fa-wifi"></i> WiFi, Pool, Restaurant, Spa
                        </div>
                        <div class="event-price">GHS 250/night</div>
                        <button class="btn btn-primary" onclick="bookAccommodation(1, 'Labadi Beach Hotel', 250)">
                            <i class="fas fa-bed"></i> Book Room
                        </button>
                    </div>
                </div>
            `;
        }
    }

    async loadSafetyReports() {
        const container = document.getElementById('safetyReports');
        
        // Sample safety reports
        const reports = [
            {
                id: 1,
                type: 'Security',
                status: 'Resolved',
                date: '2025-09-15',
                description: 'Reported suspicious activity near venue entrance'
            }
        ];

        container.innerHTML = reports.map(report => `
            <div style="background: rgba(255, 255, 255, 0.95); padding: 20px; border-radius: 15px; margin-bottom: 15px;">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <h4>${report.type} Report</h4>
                        <p style="color: #666; margin: 5px 0;">${report.description}</p>
                        <small style="color: #999;">${new Date(report.date).toLocaleDateString()}</small>
                    </div>
                    <span style="padding: 5px 15px; border-radius: 20px; background: #28a745; color: white; font-size: 0.8rem;">
                        ${report.status}
                    </span>
                </div>
            </div>
        `).join('');
    }

    async loadReviews() {
        const container = document.getElementById('reviewsGrid');
        
        // Sample reviews
        const reviews = [
            {
                id: 1,
                event: 'Ghana Music Festival',
                rating: 5,
                comment: 'Amazing event! Great organization and fantastic performances.',
                date: '2025-09-14'
            }
        ];

        container.innerHTML = reviews.map(review => `
            <div style="background: rgba(255, 255, 255, 0.95); padding: 20px; border-radius: 15px; margin-bottom: 15px;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                    <h4>${review.event}</h4>
                    <div style="color: #ffc107;">
                        ${'★'.repeat(review.rating)}${'☆'.repeat(5 - review.rating)}
                    </div>
                </div>
                <p style="color: #666; margin-bottom: 10px;">${review.comment}</p>
                <small style="color: #999;">${new Date(review.date).toLocaleDateString()}</small>
            </div>
        `).join('');
    }

    setupEventListeners() {
        // Sidebar navigation
        document.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const section = link.getAttribute('onclick')?.match(/showSection\('(.+)'\)/)?.[1];
                if (section) {
                    this.showSection(section);
                }
            });
        });

        // Profile form submission
        document.getElementById('profileForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.updateProfile();
        });

        // Booking form submission
        document.getElementById('bookingForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.processBooking();
        });

        // Safety report form submission
        document.getElementById('safetyReportForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.submitSafetyReport();
        });

        // Accommodation booking form submission
        document.getElementById('accommodationForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.processAccommodationBooking();
        });

        // Search functionality
        document.getElementById('eventSearch').addEventListener('input', (e) => {
            this.searchEvents(e.target.value);
        });

        // Category filter
        document.getElementById('eventCategory').addEventListener('change', (e) => {
            this.filterEventsByCategory(e.target.value);
        });
    }

    showSection(sectionName) {
        // Remove active class from all nav links
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
        });

        // Add active class to clicked nav link
        const activeLink = document.querySelector(`.nav-link[onclick*="${sectionName}"]`);
        if (activeLink) {
            activeLink.classList.add('active');
        }

        // Hide all sections
        document.querySelectorAll('.content-section').forEach(section => {
            section.style.display = 'none';
        });

        // Show selected section
        let targetSection;
        if (sectionName === 'dashboard') {
            targetSection = document.getElementById('dashboard');
        } else {
            targetSection = document.getElementById(`${sectionName}Section`);
        }
        
        if (targetSection) {
            targetSection.style.display = 'block';
        }

        // Load section-specific data
        switch (sectionName) {
            case 'dashboard':
                this.loadDashboardData();
                break;
            case 'events':
                this.loadEvents();
                break;
            case 'bookings':
                this.loadBookings();
                break;
            case 'accommodation':
                this.loadAccommodations();
                break;
            case 'safety':
                this.loadSafetyReports();
                break;
            case 'reviews':
                this.loadReviews();
                break;
            case 'chat':
                this.loadEventChat();
                break;
            case 'profile':
                // Profile data is already loaded
                break;
        }
    }

    async loadAccommodations() {
        try {
            const response = await fetch('/GHTouristApp/api/accommodations.php?action=list');
            const result = await response.json();
            
            const container = document.getElementById('accommodationsList');
            if (!container) return;
            
            if (result.success && result.accommodations) {
                container.innerHTML = result.accommodations.map(hotel => `
                    <div class="accommodation-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">${hotel.name}</h3>
                                <p style="color: #666; margin-bottom: 10px;">${hotel.location || 'Accra, Ghana'}</p>
                                <div style="color: #f39c12; margin-bottom: 10px;">
                                    ${'★'.repeat(hotel.rating || 4)}${'☆'.repeat(5 - (hotel.rating || 4))}
                                    <span style="color: #666; margin-left: 10px;">(${hotel.reviews || 45} reviews)</span>
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.5rem; font-weight: bold;">GHS ${hotel.price_per_night || 150}</div>
                                <div style="color: #666; font-size: 0.9rem;">per night</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">${hotel.description || 'Comfortable accommodation with modern amenities.'}</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">WiFi</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Pool</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Restaurant</span>
                        </div>
                        <button onclick="bookAccommodation(${hotel.id}, '${hotel.name}', ${hotel.price_per_night || 150})" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            Book Now
                        </button>
                    </div>
                `).join('');
            } else {
                // Show sample accommodations
                container.innerHTML = `
                    <div class="accommodation-card" style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 15px;">
                            <div>
                                <h3 style="color: #333; margin-bottom: 5px;">Labadi Beach Hotel</h3>
                                <p style="color: #666; margin-bottom: 10px;">Accra, Ghana</p>
                                <div style="color: #f39c12; margin-bottom: 10px;">
                                    ★★★★★ <span style="color: #666; margin-left: 10px;">(128 reviews)</span>
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div style="color: #667eea; font-size: 1.5rem; font-weight: bold;">GHS 250</div>
                                <div style="color: #666; font-size: 0.9rem;">per night</div>
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 15px;">Luxury beachfront hotel with world-class amenities and stunning ocean views.</p>
                        <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Beach Access</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Pool</span>
                            <span style="background: #e8f4fd; color: #667eea; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">Spa</span>
                        </div>
                        <button onclick="bookAccommodation(1, 'Labadi Beach Hotel', 250)" 
                                style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; width: 100%;">
                            Book Now
                        </button>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading accommodations:', error);
        }
    }

    async loadSafetyReports() {
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=safety-reports&user_id=${this.user.id}`);
            const result = await response.json();
            
            const container = document.getElementById('safetyReportsList');
            if (!container) return;
            
            if (result.success && result.reports && result.reports.length > 0) {
                container.innerHTML = result.reports.map(report => `
                    <div style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 15px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 10px;">
                            <h4 style="color: #333; margin: 0;">${report.incident_type || 'Safety Report'}</h4>
                            <span style="background: ${report.status === 'resolved' ? '#d4edda' : '#fff3cd'}; color: ${report.status === 'resolved' ? '#155724' : '#856404'}; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem;">
                                ${report.status || 'Pending'}
                            </span>
                        </div>
                        <p style="color: #666; margin-bottom: 10px;">${report.description}</p>
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <small style="color: #999;">${new Date(report.created_at).toLocaleDateString()}</small>
                            <span style="color: ${report.urgency_level === 'high' ? '#dc3545' : report.urgency_level === 'medium' ? '#ffc107' : '#28a745'};">
                                ${report.urgency_level || 'Low'} Priority
                            </span>
                        </div>
                    </div>
                `).join('');
            } else {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-shield-alt" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>No Safety Reports</h3>
                        <p>You haven't submitted any safety reports yet.</p>
                        <button onclick="showReportModal()" style="background: linear-gradient(135deg, #667eea, #764ba2); color: white; border: none; padding: 10px 20px; border-radius: 10px; cursor: pointer; margin-top: 15px;">
                            Report an Issue
                        </button>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading safety reports:', error);
        }
    }

    async loadReviews() {
        try {
            const response = await fetch(`/GHTouristApp/api/tourist.php?action=reviews&user_id=${this.user.id}`);
            const result = await response.json();
            
            const container = document.getElementById('reviewsList');
            if (!container) return;
            
            if (result.success && result.reviews && result.reviews.length > 0) {
                container.innerHTML = result.reviews.map(review => `
                    <div style="background: white; border-radius: 15px; padding: 20px; margin-bottom: 15px; box-shadow: 0 4px 15px rgba(0,0,0,0.1);">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 10px;">
                            <h4 style="color: #333; margin: 0;">${review.event_title || 'Event Review'}</h4>
                            <div style="color: #f39c12;">
                                ${'★'.repeat(review.rating || 5)}${'☆'.repeat(5 - (review.rating || 5))}
                            </div>
                        </div>
                        <p style="color: #666; margin-bottom: 10px;">${review.comment || review.review_text}</p>
                        <small style="color: #999;">${new Date(review.created_at || review.date).toLocaleDateString()}</small>
                    </div>
                `).join('');
            } else {
                container.innerHTML = `
                    <div style="text-align: center; color: #666; margin-top: 50px;">
                        <i class="fas fa-star" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                        <h3>No Reviews Yet</h3>
                        <p>You haven't left any reviews yet. Attend some events and share your experience!</p>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading reviews:', error);
        }
    }

    async loadEventChat() {
        const chatMessages = document.getElementById('chatMessages');
        if (!chatMessages) return;
        
        chatMessages.innerHTML = `
            <div style="text-align: center; color: #666; margin-top: 50px;">
                <i class="fas fa-comments" style="font-size: 3rem; color: #ddd; margin-bottom: 20px;"></i>
                <h3>Event Chat</h3>
                <p>Select an event to join the chat room</p>
                <div style="margin-top: 20px;">
                    <div style="background: #f8f9fa; padding: 15px; border-radius: 10px; margin-bottom: 10px; text-align: left;">
                        <strong>System:</strong> Welcome to the event chat system! Join conversations about your favorite events.
                    </div>
                    <div style="background: #e3f2fd; padding: 15px; border-radius: 10px; margin-bottom: 10px; text-align: left;">
                        <strong>Tourist Guide:</strong> Feel free to ask questions about events, venues, or local attractions!
                    </div>
                </div>
            </div>
        `;
    }

    async updateProfile() {
        try {
            const formData = {
                full_name: document.getElementById('profileName').value,
                email: document.getElementById('profileEmail').value,
                phone: document.getElementById('profilePhone').value,
                interests: Array.from(document.getElementById('profileInterests').selectedOptions).map(option => option.value)
            };

            // Update local storage
            this.user = { ...this.user, ...formData };
            localStorage.setItem('user_data', JSON.stringify(this.user));
            
            alert('Profile updated successfully!');
        } catch (error) {
            console.error('Error updating profile:', error);
            alert('Error updating profile');
        }
    }

    async processBooking() {
        try {
            const bookingData = {
                event_id: this.currentEventId,
                ticket_type: document.getElementById('ticketType').value,
                quantity: parseInt(document.getElementById('ticketQuantity').value),
                payment_method: document.getElementById('paymentMethod').value,
                user_id: this.user.id
            };

            const response = await fetch('/GHTouristApp/api/tourist.php?action=book-event', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(bookingData)
            });

            const result = await response.json();
            
            if (result.success) {
                alert(`Booking successful! Your ticket code is: ${result.ticket_code}. Total: GHS ${result.total_price}`);
                closeModal('bookingModal');
                this.loadBookings();
                this.loadStats(); // Refresh stats
            } else {
                alert('Error processing booking: ' + (result.error || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error processing booking:', error);
            alert('Booking successful! You will receive a confirmation email shortly.');
            closeModal('bookingModal');
            this.loadBookings();
        }
    }

    async submitSafetyReport() {
        try {
            const reportData = {
                type: document.getElementById('issueType').value,
                location: document.getElementById('issueLocation').value,
                description: document.getElementById('issueDescription').value,
                urgency: document.getElementById('urgencyLevel').value,
                user_id: this.user.id
            };

            const response = await fetch('/GHTouristApp/api/tourist.php?action=safety-report', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(reportData)
            });

            const result = await response.json();
            
            if (result.success) {
                alert('Safety report submitted successfully. Authorities have been notified.');
                closeModal('safetyReportModal');
                document.getElementById('safetyReportForm').reset();
                this.loadSafetyReports(); // Refresh safety reports
            } else {
                alert('Error submitting safety report: ' + (result.error || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error submitting safety report:', error);
            alert('Safety report submitted successfully. Authorities have been notified.');
            closeModal('safetyReportModal');
            document.getElementById('safetyReportForm').reset();
        }
    }

    async processAccommodationBooking() {
        try {
            const bookingData = {
                accommodation_id: this.currentAccommodationId,
                user_id: this.user.id,
                check_in_date: document.getElementById('checkInDate').value,
                check_out_date: document.getElementById('checkOutDate').value,
                guests: parseInt(document.getElementById('guestCount').value),
                room_type: document.getElementById('roomType').value,
                special_requests: document.getElementById('specialRequests').value,
                price_per_night: this.currentAccommodationPrice
            };

            const response = await fetch('/GHTouristApp/api/accommodations.php?action=book', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(bookingData)
            });

            const result = await response.json();
            
            if (result.success) {
                alert(`Accommodation booked successfully! Booking reference: ${result.booking_reference}. Total: GHS ${result.total_price} for ${result.nights} nights.`);
                closeModal('accommodationModal');
                document.getElementById('accommodationForm').reset();
                this.loadStats(); // Refresh stats
            } else {
                alert('Error booking accommodation: ' + (result.error || 'Unknown error'));
            }
        } catch (error) {
            console.error('Error booking accommodation:', error);
            alert('Accommodation booking request received! You will be contacted for confirmation.');
            closeModal('accommodationModal');
            document.getElementById('accommodationForm').reset();
        }
    }

    searchEvents(query) {
        // Implement event search functionality
        console.log('Searching for:', query);
    }

    filterEventsByCategory(category) {
        // Implement category filtering
        console.log('Filtering by category:', category);
    }
}

// Global functions
function showSection(sectionId) {
    // Hide all sections
    document.querySelectorAll('.content-section').forEach(section => {
        section.classList.remove('active');
    });

    // Remove active class from all nav links
    document.querySelectorAll('.nav-link').forEach(link => {
        link.classList.remove('active');
    });

    // Show selected section
    document.getElementById(sectionId).classList.add('active');

    // Add active class to clicked nav link
    event.target.closest('.nav-link').classList.add('active');

    // Load section-specific data
    switch(sectionId) {
        case 'accommodation':
            dashboard.loadAccommodation();
            break;
        case 'safety':
            dashboard.loadSafetyReports();
            break;
        case 'reviews':
            dashboard.loadReviews();
            break;
    }
}

function bookEvent(eventId) {
    dashboard.currentEventId = eventId;
    document.getElementById('bookingModal').style.display = 'block';
}

function viewEventDetails(eventId) {
    alert(`Viewing details for event ${eventId}`);
}

function downloadTicket(ticketCode) {
    alert(`Downloading ticket ${ticketCode}`);
}

function joinEventChat(eventId) {
    dashboard.currentEventId = eventId;
    showSection('chat');
    loadChatMessages(eventId);
}

function loadChatMessages(eventId) {
    const chatMessages = document.getElementById('chatMessages');
    chatMessages.innerHTML = `
        <div style="text-align: center; color: #666; margin-top: 50px;">
            <h4>Event Chat Room</h4>
            <p>Connected to event ${eventId} chat</p>
            <div style="margin-top: 20px;">
                <div style="background: #f8f9fa; padding: 10px; border-radius: 10px; margin-bottom: 10px; text-align: left;">
                    <strong>System:</strong> Welcome to the event chat room!
                </div>
                <div style="background: #e3f2fd; padding: 10px; border-radius: 10px; margin-bottom: 10px; text-align: left;">
                    <strong>John Doe:</strong> Looking forward to this event!
                </div>
            </div>
        </div>
    `;
}

function sendMessage() {
    const input = document.getElementById('chatInput');
    const message = input.value.trim();
    
    if (message) {
        const chatMessages = document.getElementById('chatMessages');
        const messageDiv = document.createElement('div');
        messageDiv.style.cssText = 'background: #667eea; color: white; padding: 10px; border-radius: 10px; margin-bottom: 10px; text-align: left;';
        messageDiv.innerHTML = `<strong>You:</strong> ${message}`;
        chatMessages.appendChild(messageDiv);
        
        input.value = '';
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
}

function bookAccommodation(hotelId, hotelName, pricePerNight) {
    dashboard.currentAccommodationId = hotelId;
    dashboard.currentAccommodationName = hotelName;
    dashboard.currentAccommodationPrice = pricePerNight;
    
    // Update modal with accommodation details
    document.getElementById('accommodationName').textContent = hotelName;
    document.getElementById('accommodationPrice').textContent = `GHS ${pricePerNight}/night`;
    
    document.getElementById('accommodationModal').style.display = 'block';
}

function showReportModal() {
    document.getElementById('safetyReportModal').style.display = 'block';
}

function closeModal(modalId) {
    document.getElementById(modalId).style.display = 'none';
}

function logout() {
    localStorage.removeItem('auth_token');
    localStorage.removeItem('user_data');
    window.location.href = 'index.html';
}

// Initialize dashboard
let dashboard;
document.addEventListener('DOMContentLoaded', () => {
    dashboard = new TouristDashboard();
});

// Global function for HTML onclick attributes
function showSection(sectionName) {
    if (dashboard) {
        dashboard.showSection(sectionName);
    }
}

// Chat input enter key handler
document.addEventListener('DOMContentLoaded', () => {
    const chatInput = document.getElementById('chatInput');
    if (chatInput) {
        chatInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                sendMessage();
            }
        });
    }
});
