// Venues Page JavaScript
class VenuesManager {
    constructor() {
        this.currentCategory = 'all';
        this.filters = {};
        this.venues = [];
        this.totalVenues = 0;
        
        this.init();
    }

    async init() {
        await this.loadVenues();
        this.setupEventListeners();
        this.updateAuthUI();
    }

    setupEventListeners() {
        // Search form
        const searchForm = document.getElementById('searchForm');
        if (searchForm) {
            searchForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.handleSearch();
            });
        }

        // Real-time search
        const searchQuery = document.getElementById('searchQuery');
        if (searchQuery) {
            let searchTimeout;
            searchQuery.addEventListener('input', (e) => {
                clearTimeout(searchTimeout);
                searchTimeout = setTimeout(() => {
                    this.handleSearch();
                }, 500);
            });
        }

        // Filter changes
        const filters = ['locationFilter', 'ratingFilter'];
        filters.forEach(filterId => {
            const element = document.getElementById(filterId);
            if (element) {
                element.addEventListener('change', () => {
                    this.handleSearch();
                });
            }
        });
    }

    async loadVenues() {
        this.showLoading();
        
        try {
            const params = new URLSearchParams({
                endpoint: 'list',
                category: this.currentCategory === 'all' ? '' : this.currentCategory,
                ...this.filters
            });

            const response = await fetch(`./api/venues.php?${params}`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                }
            });

            if (response.ok) {
                const data = await response.json();
                this.venues = data.venues || [];
                this.totalVenues = data.total || 0;
                
                this.displayVenues();
                this.updateResultsCount();
                this.hideLoading();
            } else {
                throw new Error('Failed to load venues');
            }
        } catch (error) {
            console.error('Error loading venues:', error);
            this.showError('Failed to load venues. Please try again.');
            this.hideLoading();
        }
    }

    switchCategory(category) {
        this.currentCategory = category;
        
        // Update active tab
        document.querySelectorAll('.tab-btn').forEach(btn => {
            btn.classList.remove('active');
        });
        document.querySelector(`[data-category="${category}"]`).classList.add('active');
        
        // Update section title
        const titles = {
            'all': 'All Venues & Vendors',
            'venue': 'Event Venues',
            'hotel': 'Hotels',
            'restaurant': 'Restaurants',
            'catering': 'Catering Services',
            'entertainment': 'Entertainment Services'
        };
        
        document.getElementById('sectionTitle').textContent = titles[category] || 'Venues & Vendors';
        
        // Load venues for this category
        this.loadVenues();
    }

    handleSearch() {
        // Collect filter values
        this.filters = {};
        
        const searchQuery = document.getElementById('searchQuery').value.trim();
        if (searchQuery) {
            this.filters.search = searchQuery;
        }

        const location = document.getElementById('locationFilter').value;
        if (location) {
            this.filters.location = location;
        }

        const rating = document.getElementById('ratingFilter').value;
        if (rating) {
            this.filters.min_rating = rating;
        }

        // Load venues with filters
        this.loadVenues();
    }

    displayVenues() {
        const venuesGrid = document.getElementById('venuesGrid');
        const emptyState = document.getElementById('emptyState');

        if (this.venues.length === 0) {
            venuesGrid.style.display = 'none';
            emptyState.style.display = 'block';
            return;
        }

        emptyState.style.display = 'none';
        venuesGrid.style.display = 'grid';
        venuesGrid.innerHTML = this.venues.map(venue => this.createVenueCard(venue)).join('');
    }

    createVenueCard(venue) {
        const rating = parseFloat(venue.average_rating) || 0;
        const stars = this.generateStars(rating);
        const reviewCount = venue.review_count || 0;
        
        const priceText = venue.price_range ? 
            `From GHS ${venue.price_range}` : 
            'Contact for pricing';

        return `
            <div class="venue-card" data-venue-id="${venue.id}">
                <div class="venue-image">
                    ${venue.image_url ? 
                        `<img src="${venue.image_url}" alt="${venue.name}" onerror="this.style.display='none'">` : 
                        ''
                    }
                    ${venue.government_verified ? 
                        '<div class="venue-badge verified">Government Verified</div>' : 
                        '<div class="venue-badge">Pending Verification</div>'
                    }
                </div>
                <div class="venue-content">
                    <div class="venue-type">${venue.type || 'Venue'}</div>
                    <h3 class="venue-name">${venue.name}</h3>
                    <div class="venue-meta">
                        <div class="venue-meta-item">
                            <i class="fas fa-map-marker-alt"></i>
                            <span>${venue.location}</span>
                        </div>
                        ${venue.capacity ? `
                            <div class="venue-meta-item">
                                <i class="fas fa-users"></i>
                                <span>Capacity: ${venue.capacity}</span>
                            </div>
                        ` : ''}
                        ${venue.contact_phone ? `
                            <div class="venue-meta-item">
                                <i class="fas fa-phone"></i>
                                <span>${venue.contact_phone}</span>
                            </div>
                        ` : ''}
                    </div>
                    <div class="venue-rating">
                        <div class="stars">${stars}</div>
                        <span class="rating-text">${rating.toFixed(1)} (${reviewCount} reviews)</span>
                    </div>
                    <div class="venue-price">${priceText}</div>
                    <div class="venue-actions">
                        <button class="btn-book" onclick="venuesManager.bookVenue(${venue.id})">
                            <i class="fas fa-calendar-plus"></i>
                            Book Now
                        </button>
                        <button class="btn-details" onclick="venuesManager.showVenueDetails(${venue.id})">
                            Details
                        </button>
                    </div>
                </div>
            </div>
        `;
    }

    generateStars(rating) {
        let stars = '';
        for (let i = 1; i <= 5; i++) {
            if (i <= rating) {
                stars += '<i class="fas fa-star star"></i>';
            } else if (i - 0.5 <= rating) {
                stars += '<i class="fas fa-star-half-alt star"></i>';
            } else {
                stars += '<i class="fas fa-star star empty"></i>';
            }
        }
        return stars;
    }

    async showVenueDetails(venueId) {
        try {
            const response = await fetch(`./api/venues.php?endpoint=details&id=${venueId}`, {
                headers: {
                    'Authorization': `Bearer ${localStorage.getItem('token') || ''}`
                }
            });

            if (response.ok) {
                const data = await response.json();
                console.log('Venue data received:', data); // Debug log
                if (data.success && data.venue) {
                    this.displayVenueModal(data.venue);
                } else {
                    throw new Error('Invalid venue data received');
                }
            } else {
                throw new Error('Failed to load venue details');
            }
        } catch (error) {
            console.error('Error loading venue details:', error);
            this.showNotification('Failed to load venue details', 'error');
        }
    }

    displayVenueModal(venue) {
        const modal = document.getElementById('venueDetailModal');
        if (!modal) {
            console.error('Modal element not found');
            return;
        }

        const rating = parseFloat(venue.average_rating) || 0;
        const stars = this.generateStars(rating);
        const reviewCount = venue.review_count || 0;

        // Fix location reference - use address and city from venue data
        const location = venue.address ? `${venue.address}, ${venue.city || ''}` : (venue.city || 'Location not specified');

        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h2>${venue.name}</h2>
                    <button class="modal-close" onclick="this.closest('.modal').style.display='none'">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div class="modal-body">
                    ${venue.images && venue.images.length > 0 ? `
                        <img src="${venue.images[0]}" alt="${venue.name}" style="width: 100%; height: 300px; object-fit: cover; border-radius: 12px; margin-bottom: 1.5rem;">
                    ` : ''}
                    
                    <div class="venue-details-grid">
                        <div class="venue-info">
                            <h3>Venue Information</h3>
                            <div class="info-item">
                                <i class="fas fa-tag"></i>
                                <span><strong>Type:</strong> ${venue.type || 'Venue'}</span>
                            </div>
                            <div class="info-item">
                                <i class="fas fa-map-marker-alt"></i>
                                <span><strong>Location:</strong> ${location}</span>
                            </div>
                            ${venue.capacity ? `
                                <div class="info-item">
                                    <i class="fas fa-users"></i>
                                    <span><strong>Capacity:</strong> ${venue.capacity} people</span>
                                </div>
                            ` : ''}
                            ${venue.contact_phone ? `
                                <div class="info-item">
                                    <i class="fas fa-phone"></i>
                                    <span><strong>Phone:</strong> ${venue.contact_phone}</span>
                                </div>
                            ` : ''}
                            ${venue.contact_email ? `
                                <div class="info-item">
                                    <i class="fas fa-envelope"></i>
                                    <span><strong>Email:</strong> ${venue.contact_email}</span>
                                </div>
                            ` : ''}
                            <div class="info-item">
                                <i class="fas fa-star"></i>
                                <span><strong>Rating:</strong> ${stars} ${rating.toFixed(1)} (${reviewCount} reviews)</span>
                            </div>
                            ${venue.government_verified ? `
                                <div class="info-item verified">
                                    <i class="fas fa-check-circle"></i>
                                    <span><strong>Government Verified</strong></span>
                                </div>
                            ` : ''}
                        </div>
                        
                        <div class="venue-description">
                            <h3>Description</h3>
                            <p>${venue.description || 'No description available.'}</p>
                            
                            ${venue.amenities && venue.amenities.length > 0 ? `
                                <h4>Amenities</h4>
                                <ul>
                                    ${venue.amenities.map(amenity => `<li>${amenity}</li>`).join('')}
                                </ul>
                            ` : ''}

                            ${venue.upcoming_events && venue.upcoming_events.length > 0 ? `
                                <h4>Upcoming Events</h4>
                                <ul>
                                    ${venue.upcoming_events.map(event => `
                                        <li>${event.title} - ${new Date(event.start_datetime).toLocaleDateString()}</li>
                                    `).join('')}
                                </ul>
                            ` : ''}
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-outline" onclick="this.closest('.modal').style.display='none'">Close</button>
                    <button class="btn btn-primary" onclick="venuesManager.bookVenue(${venue.id})">
                        <i class="fas fa-calendar-plus"></i>
                        Book Now
                    </button>
                </div>
            </div>
        `;

        console.log('Displaying modal'); // Debug log
        modal.style.display = 'flex';
        
        // Ensure modal is visible by adding a small delay
        setTimeout(() => {
            modal.style.opacity = '1';
        }, 10);
    }

    async bookVenue(venueId) {
        const token = localStorage.getItem('token');
        if (!token) {
            this.showNotification('Please login to book venues', 'warning');
            app.showLoginModal();
            return;
        }

        // Show booking modal
        this.showBookingModal(venueId);
    }

    showBookingModal(venueId) {
        const modal = document.getElementById('bookingModal');
        
        modal.innerHTML = `
            <div class="modal-content">
                <div class="modal-header">
                    <h2>Book Venue</h2>
                    <button class="modal-close" onclick="this.closest('.modal').style.display='none'">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="bookingForm">
                        <div class="form-group">
                            <label for="eventDate">Event Date</label>
                            <input type="date" id="eventDate" name="event_date" required>
                        </div>
                        <div class="form-group">
                            <label for="eventTime">Event Time</label>
                            <input type="time" id="eventTime" name="event_time" required>
                        </div>
                        <div class="form-group">
                            <label for="duration">Duration (hours)</label>
                            <input type="number" id="duration" name="duration" min="1" max="24" value="4" required>
                        </div>
                        <div class="form-group">
                            <label for="guestCount">Expected Guests</label>
                            <input type="number" id="guestCount" name="guest_count" min="1" required>
                        </div>
                        <div class="form-group">
                            <label for="eventType">Event Type</label>
                            <select id="eventType" name="event_type" required>
                                <option value="">Select event type</option>
                                <option value="wedding">Wedding</option>
                                <option value="corporate">Corporate Event</option>
                                <option value="birthday">Birthday Party</option>
                                <option value="conference">Conference</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="specialRequests">Special Requests</label>
                            <textarea id="specialRequests" name="special_requests" rows="3" placeholder="Any special requirements or requests..."></textarea>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-outline" onclick="this.closest('.modal').style.display='none'">Cancel</button>
                    <button class="btn btn-primary" onclick="venuesManager.submitBooking(${venueId})">
                        <i class="fas fa-check"></i>
                        Submit Booking Request
                    </button>
                </div>
            </div>
        `;

        // Set minimum date to today
        const today = new Date().toISOString().split('T')[0];
        document.getElementById('eventDate').min = today;

        modal.style.display = 'flex';
    }

    async submitBooking(venueId) {
        const form = document.getElementById('bookingForm');
        const formData = new FormData(form);
        
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }

        try {
            const bookingData = {
                venue_id: venueId,
                event_date: formData.get('event_date'),
                event_time: formData.get('event_time'),
                duration: formData.get('duration'),
                guest_count: formData.get('guest_count'),
                event_type: formData.get('event_type'),
                special_requests: formData.get('special_requests')
            };

            const response = await fetch('./api/venues.php?endpoint=bookings', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${localStorage.getItem('token')}`
                },
                body: JSON.stringify(bookingData)
            });

            const data = await response.json();

            if (response.ok) {
                this.showNotification('Booking request submitted successfully!', 'success');
                document.getElementById('bookingModal').style.display = 'none';
            } else {
                throw new Error(data.message || 'Failed to submit booking');
            }
        } catch (error) {
            console.error('Error submitting booking:', error);
            this.showNotification(error.message || 'Failed to submit booking', 'error');
        }
    }

    updateResultsCount() {
        const resultsCount = document.getElementById('resultsCount');
        if (resultsCount) {
            resultsCount.textContent = `${this.totalVenues} venues found`;
        }
    }

    showLoading() {
        document.getElementById('loadingState').style.display = 'block';
        document.getElementById('venuesGrid').style.display = 'none';
        document.getElementById('emptyState').style.display = 'none';
    }

    hideLoading() {
        document.getElementById('loadingState').style.display = 'none';
    }

    showError(message) {
        this.showNotification(message, 'error');
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
            <span>${message}</span>
            <button onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;

        // Add to page
        document.body.appendChild(notification);

        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 5000);
    }

    updateAuthUI() {
        const token = localStorage.getItem('token');
        const authButtons = document.getElementById('authButtons');
        
        if (token && authButtons) {
            // User is logged in, show user menu
            authButtons.innerHTML = `
                <a href="dashboard.html" class="btn btn-outline">
                    <i class="fas fa-tachometer-alt"></i>
                    Dashboard
                </a>
                <button class="btn btn-primary" onclick="app.logout()">
                    <i class="fas fa-sign-out-alt"></i>
                    Logout
                </button>
            `;
        }
    }
}

// Initialize venues manager when page loads
let venuesManager;
document.addEventListener('DOMContentLoaded', () => {
    venuesManager = new VenuesManager();
});

// Add modal and form styles
const venueStyles = `
    .venue-details-grid {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 2rem;
    }

    .venue-info .info-item {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        margin-bottom: 1rem;
        padding: 0.75rem;
        background: #f7fafc;
        border-radius: 8px;
    }

    .venue-info .info-item i {
        color: #667eea;
        width: 20px;
    }

    .venue-info .info-item.verified {
        background: #f0fff4;
        color: #38a169;
    }

    .form-group {
        margin-bottom: 1.5rem;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: #1a202c;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 0.75rem;
        border: 2px solid #e2e8f0;
        border-radius: 8px;
        font-size: 1rem;
        transition: border-color 0.3s ease;
    }

    .form-group input:focus,
    .form-group select:focus,
    .form-group textarea:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    .notification {
        position: fixed;
        top: 100px;
        right: 2rem;
        background: white;
        padding: 1rem 1.5rem;
        border-radius: 8px;
        box-shadow: 0 10px 15px rgba(0,0,0,0.1);
        display: flex;
        align-items: center;
        gap: 0.75rem;
        z-index: 10001;
        border-left: 4px solid #667eea;
        animation: slideIn 0.3s ease;
    }

    .notification-success {
        border-left-color: #48bb78;
        color: #38a169;
    }

    .notification-error {
        border-left-color: #f56565;
        color: #e53e3e;
    }

    .notification-warning {
        border-left-color: #ed8936;
        color: #dd6b20;
    }

    .notification button {
        background: none;
        border: none;
        cursor: pointer;
        color: #718096;
        padding: 0.25rem;
    }

    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }

    @media (max-width: 768px) {
        .venue-details-grid {
            grid-template-columns: 1fr;
        }
    }
`;

// Inject styles
const styleSheet = document.createElement('style');
styleSheet.textContent = venueStyles;
document.head.appendChild(styleSheet);
