<?php
/**
 * Quick Fix for Media Chat Database Issue
 * Run this script to immediately fix the "Column not found: message_type" error
 */

require_once 'config/database.php';

try {
    $db = new Database();
    $pdo = $db->getConnection();
    
    echo "Fixing media chat database issue...\n\n";
    
    // Create the simple_chat_messages table with all required columns
    $sql = "
        CREATE TABLE IF NOT EXISTS `simple_chat_messages` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `event_id` int(11) NOT NULL,
            `user_id` int(11) NOT NULL,
            `message` text NOT NULL,
            `message_type` ENUM('text', 'image', 'video', 'file') DEFAULT 'text',
            `file_url` VARCHAR(500) NULL,
            `file_name` VARCHAR(255) NULL,
            `file_size` INT NULL,
            `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
            PRIMARY KEY (`id`),
            FOREIGN KEY (`event_id`) REFERENCES `events`(`id`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    
    $pdo->exec($sql);
    echo "✓ Created/updated simple_chat_messages table with media support\n";
    
    // Check if columns exist and add them if missing
    $columns = ['message_type', 'file_url', 'file_name', 'file_size'];
    
    foreach ($columns as $column) {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE table_name = 'simple_chat_messages' 
            AND column_name = ? 
            AND table_schema = DATABASE()
        ");
        $stmt->execute([$column]);
        
        if ($stmt->fetchColumn() == 0) {
            // Column doesn't exist, add it
            switch ($column) {
                case 'message_type':
                    $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN message_type ENUM('text', 'image', 'video', 'file') DEFAULT 'text' AFTER message");
                    break;
                case 'file_url':
                    $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_url VARCHAR(500) NULL AFTER message_type");
                    break;
                case 'file_name':
                    $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_name VARCHAR(255) NULL AFTER file_url");
                    break;
                case 'file_size':
                    $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_size INT NULL AFTER file_name");
                    break;
            }
            echo "✓ Added column: $column\n";
        } else {
            echo "• Column already exists: $column\n";
        }
    }
    
    // Create uploads directory
    $uploadDir = 'uploads/chat/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0755, true);
        echo "✓ Created uploads/chat directory\n";
    } else {
        echo "• uploads/chat directory already exists\n";
    }
    
    // Create .htaccess for security
    $htaccessContent = "# Prevent direct access to uploaded files
Options -Indexes
<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|mp4|webm|mov|pdf|doc|docx|txt|zip|rar)$\">
    Order Allow,Deny
    Allow from all
</FilesMatch>
<FilesMatch \"\\.php$\">
    Order Deny,Allow
    Deny from all
</FilesMatch>";
    
    file_put_contents($uploadDir . '.htaccess', $htaccessContent);
    echo "✓ Created security .htaccess file\n";
    
    echo "\n🎉 SUCCESS! Media chat is now ready to use!\n\n";
    echo "You can now:\n";
    echo "1. Go to the tourist dashboard\n";
    echo "2. Join an event chat\n";
    echo "3. Click the paperclip icon to share media\n";
    echo "4. Upload images, videos, and files up to 10MB\n\n";
    
} catch (Exception $e) {
    echo "❌ ERROR: " . $e->getMessage() . "\n";
    echo "Please check your database connection and try again.\n";
}
?>
