<?php
/**
 * Media Chat Migration Script
 * This script adds media sharing support to the simple_chat_messages table
 */

require_once 'config/database.php';

try {
    echo "<h2>Media Chat Migration</h2>\n";
    echo "<p>Adding media sharing support to simple_chat_messages table...</p>\n";
    
    $db = new Database();
    $pdo = $db->getConnection();
    
    // Start transaction
    $pdo->beginTransaction();
    
    echo "<h3>Step 1: Creating simple_chat_messages table if it doesn't exist</h3>\n";
    
    // Create the simple_chat_messages table if it doesn't exist
    $createTableSQL = "
        CREATE TABLE IF NOT EXISTS `simple_chat_messages` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `event_id` int(11) NOT NULL,
            `user_id` int(11) NOT NULL,
            `message` text NOT NULL,
            `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
            PRIMARY KEY (`id`),
            FOREIGN KEY (`event_id`) REFERENCES `events`(`id`),
            FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
    ";
    
    $pdo->exec($createTableSQL);
    echo "<p>✓ Table created/verified</p>\n";
    
    echo "<h3>Step 2: Adding media support columns</h3>\n";
    
    // Function to check if column exists
    function columnExists($pdo, $table, $column) {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE table_name = ? AND column_name = ? AND table_schema = DATABASE()
        ");
        $stmt->execute([$table, $column]);
        return $stmt->fetchColumn() > 0;
    }
    
    // Add message_type column
    if (!columnExists($pdo, 'simple_chat_messages', 'message_type')) {
        $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN message_type ENUM('text', 'image', 'video', 'file') DEFAULT 'text' AFTER message");
        echo "<p>✓ Added message_type column</p>\n";
    } else {
        echo "<p>• message_type column already exists</p>\n";
    }
    
    // Add file_url column
    if (!columnExists($pdo, 'simple_chat_messages', 'file_url')) {
        $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_url VARCHAR(500) NULL AFTER message_type");
        echo "<p>✓ Added file_url column</p>\n";
    } else {
        echo "<p>• file_url column already exists</p>\n";
    }
    
    // Add file_name column
    if (!columnExists($pdo, 'simple_chat_messages', 'file_name')) {
        $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_name VARCHAR(255) NULL AFTER file_url");
        echo "<p>✓ Added file_name column</p>\n";
    } else {
        echo "<p>• file_name column already exists</p>\n";
    }
    
    // Add file_size column
    if (!columnExists($pdo, 'simple_chat_messages', 'file_size')) {
        $pdo->exec("ALTER TABLE simple_chat_messages ADD COLUMN file_size INT NULL AFTER file_name");
        echo "<p>✓ Added file_size column</p>\n";
    } else {
        echo "<p>• file_size column already exists</p>\n";
    }
    
    echo "<h3>Step 3: Verifying table structure</h3>\n";
    
    // Show the updated table structure
    $stmt = $pdo->query("DESCRIBE simple_chat_messages");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<table border='1' style='border-collapse: collapse; margin: 10px 0;'>\n";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th><th>Extra</th></tr>\n";
    
    foreach ($columns as $column) {
        echo "<tr>";
        echo "<td>{$column['Field']}</td>";
        echo "<td>{$column['Type']}</td>";
        echo "<td>{$column['Null']}</td>";
        echo "<td>{$column['Key']}</td>";
        echo "<td>{$column['Default']}</td>";
        echo "<td>{$column['Extra']}</td>";
        echo "</tr>\n";
    }
    echo "</table>\n";
    
    echo "<h3>Step 4: Creating uploads directory</h3>\n";
    
    // Create uploads/chat directory
    $uploadDir = 'uploads/chat/';
    if (!file_exists($uploadDir)) {
        if (mkdir($uploadDir, 0755, true)) {
            echo "<p>✓ Created uploads/chat directory</p>\n";
        } else {
            echo "<p>⚠ Failed to create uploads/chat directory</p>\n";
        }
    } else {
        echo "<p>• uploads/chat directory already exists</p>\n";
    }
    
    // Create .htaccess file for security
    $htaccessContent = "# Prevent direct access to uploaded files
Options -Indexes
# Allow only specific file types
<FilesMatch \"\\.(jpg|jpeg|png|gif|webp|mp4|webm|mov|pdf|doc|docx|txt|zip|rar)$\">
    Order Allow,Deny
    Allow from all
</FilesMatch>
# Deny access to PHP files
<FilesMatch \"\\.php$\">
    Order Deny,Allow
    Deny from all
</FilesMatch>";
    
    if (file_put_contents($uploadDir . '.htaccess', $htaccessContent)) {
        echo "<p>✓ Created security .htaccess file</p>\n";
    } else {
        echo "<p>⚠ Failed to create .htaccess file</p>\n";
    }
    
    echo "<h3>Step 5: Adding sample data (optional)</h3>\n";
    
    // Check if we have any events to add sample messages to
    $stmt = $pdo->query("SELECT COUNT(*) FROM events WHERE approval_status = 'approved'");
    $eventCount = $stmt->fetchColumn();
    
    if ($eventCount > 0) {
        // Get first approved event
        $stmt = $pdo->query("SELECT id FROM events WHERE approval_status = 'approved' LIMIT 1");
        $eventId = $stmt->fetchColumn();
        
        // Get first user
        $stmt = $pdo->query("SELECT id FROM users LIMIT 1");
        $userId = $stmt->fetchColumn();
        
        if ($eventId && $userId) {
            // Check if sample messages already exist
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM simple_chat_messages WHERE event_id = ?");
            $stmt->execute([$eventId]);
            $messageCount = $stmt->fetchColumn();
            
            if ($messageCount == 0) {
                // Insert sample messages
                $sampleMessages = [
                    ['message' => 'Welcome to the event chat! 🎉', 'type' => 'text'],
                    ['message' => 'Looking forward to this event!', 'type' => 'text'],
                    ['message' => 'Media sharing is now available - you can share images, videos, and files!', 'type' => 'text']
                ];
                
                $stmt = $pdo->prepare("
                    INSERT INTO simple_chat_messages (event_id, user_id, message, message_type) 
                    VALUES (?, ?, ?, ?)
                ");
                
                foreach ($sampleMessages as $msg) {
                    $stmt->execute([$eventId, $userId, $msg['message'], $msg['type']]);
                }
                
                echo "<p>✓ Added sample chat messages</p>\n";
            } else {
                echo "<p>• Sample messages already exist</p>\n";
            }
        }
    } else {
        echo "<p>• No approved events found for sample data</p>\n";
    }
    
    // Commit transaction
    $pdo->commit();
    
    echo "<h3>✅ Migration Completed Successfully!</h3>\n";
    echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>\n";
    echo "<h4>Media Chat Features Now Available:</h4>\n";
    echo "<ul>\n";
    echo "<li>✓ Image sharing (JPEG, PNG, GIF, WebP)</li>\n";
    echo "<li>✓ Video sharing (MP4, WebM, QuickTime)</li>\n";
    echo "<li>✓ File sharing (PDF, DOC, TXT, ZIP, RAR)</li>\n";
    echo "<li>✓ 10MB file size limit</li>\n";
    echo "<li>✓ Secure file storage</li>\n";
    echo "<li>✓ File validation and security</li>\n";
    echo "</ul>\n";
    echo "<p><strong>Next Steps:</strong></p>\n";
    echo "<ol>\n";
    echo "<li>Go to the <a href='tourist-dashboard.html'>Tourist Dashboard</a></li>\n";
    echo "<li>Login and book an event</li>\n";
    echo "<li>Access the event chat room</li>\n";
    echo "<li>Click the paperclip icon (📎) to share media</li>\n";
    echo "</ol>\n";
    echo "</div>\n";
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollback();
    }
    
    echo "<h3>❌ Migration Failed</h3>\n";
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; padding: 15px; border-radius: 5px; margin: 20px 0;'>\n";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>\n";
    echo "<p><strong>File:</strong> " . $e->getFile() . "</p>\n";
    echo "<p><strong>Line:</strong> " . $e->getLine() . "</p>\n";
    echo "</div>\n";
    
    echo "<h4>Troubleshooting:</h4>\n";
    echo "<ul>\n";
    echo "<li>Make sure your database connection is working</li>\n";
    echo "<li>Ensure you have proper database permissions</li>\n";
    echo "<li>Check that the events and users tables exist</li>\n";
    echo "<li>Verify your database configuration in config/database.php</li>\n";
    echo "</ul>\n";
}
?>

<style>
body {
    font-family: Arial, sans-serif;
    max-width: 800px;
    margin: 0 auto;
    padding: 20px;
    background: #f8f9fa;
}

h2, h3 {
    color: #333;
}

p {
    margin: 5px 0;
}

table {
    width: 100%;
    background: white;
}

th, td {
    padding: 8px;
    text-align: left;
}

th {
    background: #e9ecef;
}

.success {
    color: #28a745;
}

.warning {
    color: #ffc107;
}

.error {
    color: #dc3545;
}
</style>
