<?php
/**
 * Database Setup Script
 * Creates tables and adds sample data
 */

require_once 'config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    echo "<h2>Ghana Tourist App - Database Setup</h2>\n";
    
    // Read and execute schema
    $schema = file_get_contents('database/schema.sql');
    $statements = explode(';', $schema);
    
    echo "<h3>Creating Tables...</h3>\n";
    foreach ($statements as $statement) {
        $statement = trim($statement);
        if (!empty($statement)) {
            try {
                $conn->exec($statement);
                echo "✓ Executed: " . substr($statement, 0, 50) . "...<br>\n";
            } catch (PDOException $e) {
                if (strpos($e->getMessage(), 'already exists') === false) {
                    echo "⚠ Error: " . $e->getMessage() . "<br>\n";
                }
            }
        }
    }
    
    echo "<h3>Adding Sample Data...</h3>\n";
    
    // Add sample users
    $userQuery = "INSERT IGNORE INTO users (id, username, email, password_hash, full_name, phone, status, email_verified) VALUES 
        (1, 'admin', 'admin@ghtourist.gov.gh', ?, 'System Administrator', '+233123456789', 'active', 1),
        (2, 'organizer1', 'organizer@example.com', ?, 'Event Organizer', '+233987654321', 'active', 1)";
    
    $hashedPassword = password_hash('admin123', PASSWORD_DEFAULT);
    $stmt = $conn->prepare($userQuery);
    $stmt->execute([$hashedPassword, $hashedPassword]);
    echo "✓ Sample users created<br>\n";
    
    // Add sample venues
    $venueQuery = "INSERT IGNORE INTO venues (id, owner_id, name, type, description, address, city, region, capacity, government_verified, status) VALUES 
        (1, 1, 'Accra International Conference Centre', 'event_center', 'Premier conference and event venue', 'Ridge, Accra', 'Accra', 'Greater Accra', 2000, 1, 'active'),
        (2, 1, 'Labadi Beach Hotel', 'hotel', 'Beachfront hotel and event venue', 'Labadi Beach, Accra', 'Accra', 'Greater Accra', 500, 1, 'active'),
        (3, 1, 'Kumasi Cultural Centre', 'cultural_site', 'Traditional cultural venue', 'Kumasi Central', 'Kumasi', 'Ashanti', 1000, 1, 'active')";
    
    $conn->exec($venueQuery);
    echo "✓ Sample venues created<br>\n";
    
    // Add sample events
    $eventQuery = "INSERT IGNORE INTO events (
        id, organizer_id, venue_id, category_id, title, description, event_type, 
        start_datetime, end_datetime, max_attendees, ticket_price, is_free, 
        approval_status, government_verified, security_level
    ) VALUES 
        (1, 2, 1, 1, 'Ghana Music Festival 2024', 'Annual celebration of Ghanaian music featuring top local artists', 'public', 
         '2024-12-15 18:00:00', '2024-12-15 23:00:00', 2000, 50.00, 0, 'approved', 1, 'medium'),
        (2, 2, 2, 2, 'Homowo Festival Celebration', 'Traditional Ga festival celebration with cultural performances', 'cultural', 
         '2024-11-20 16:00:00', '2024-11-20 22:00:00', 500, 0.00, 1, 'approved', 1, 'low'),
        (3, 2, 3, 4, 'Tech Innovation Summit', 'Annual technology and innovation conference', 'business', 
         '2024-11-25 09:00:00', '2024-11-25 17:00:00', 300, 100.00, 0, 'approved', 1, 'low'),
        (4, 2, 1, 5, 'Accra Food Festival', 'Celebration of Ghanaian cuisine and local delicacies', 'public', 
         '2024-12-01 12:00:00', '2024-12-01 20:00:00', 1000, 25.00, 0, 'approved', 1, 'medium'),
        (5, 2, 2, 6, 'Comedy Night Live', 'Stand-up comedy show featuring local comedians', 'entertainment', 
         '2024-11-30 19:00:00', '2024-11-30 22:00:00', 200, 30.00, 0, 'approved', 1, 'low')";
    
    $conn->exec($eventQuery);
    echo "✓ Sample events created<br>\n";
    
    // Check results
    $countStmt = $conn->query("SELECT COUNT(*) as count FROM events WHERE approval_status = 'approved'");
    $count = $countStmt->fetch()['count'];
    
    echo "<h3>Setup Complete!</h3>\n";
    echo "✓ Database tables created<br>\n";
    echo "✓ Sample data added<br>\n";
    echo "✓ Total approved events: $count<br>\n";
    echo "<br><a href='events.html'>View Events Page</a><br>\n";
    echo "<a href='debug_events.php'>Test API Debug</a><br>\n";
    
} catch (Exception $e) {
    echo "<h3>Error:</h3>\n";
    echo $e->getMessage();
}
?>
