<?php
/**
 * Test script to verify ApiLogger functionality
 */

require_once __DIR__ . '/includes/api-logger.php';

echo "Testing ApiLogger functionality...\n\n";

// Test 1: Log a request
echo "1. Testing request logging...\n";
ApiLogger::logRequest(
    'test.php?action=test',
    'GET',
    ['test' => 'value', 'page' => 1],
    ['Authorization' => 'Bearer test-token', 'Content-Type' => 'application/json']
);
echo "✓ Request logged\n\n";

// Test 2: Log a successful response
echo "2. Testing successful response logging...\n";
$testResponse = [
    'success' => true,
    'data' => ['id' => 1, 'name' => 'Test Event'],
    'message' => 'Test successful'
];
ApiLogger::logResponse('test.php?action=test', 200, $testResponse);
echo "✓ Successful response logged\n\n";

// Test 3: Log an error response
echo "3. Testing error response logging...\n";
ApiLogger::logResponse('test.php?action=error', 500, null, 'Test error message');
echo "✓ Error response logged\n\n";

// Test 4: Log a JavaScript call
echo "4. Testing JavaScript call logging...\n";
ApiLogger::logJsCall(
    'api/events.php?action=list',
    ['page' => 1, 'limit' => 10],
    ['success' => true, 'count' => 5],
    null
);
echo "✓ JavaScript call logged\n\n";

// Test 5: Check if log file was created
$logFile = __DIR__ . '/logs/api_calls.log';
if (file_exists($logFile)) {
    echo "5. ✅ Log file created successfully!\n";
    echo "   File location: $logFile\n";
    echo "   File size: " . filesize($logFile) . " bytes\n\n";
    
    // Show recent logs
    echo "6. Recent log entries:\n";
    $logs = ApiLogger::getRecentLogs(5);
    foreach ($logs as $i => $log) {
        echo "   " . ($i + 1) . ". [{$log['type']}] {$log['endpoint']} - {$log['timestamp']}\n";
    }
} else {
    echo "5. ❌ Log file was NOT created!\n";
    echo "   Expected location: $logFile\n";
    
    // Check if logs directory exists
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        echo "   ❌ Logs directory does not exist: $logDir\n";
        echo "   Creating logs directory...\n";
        if (mkdir($logDir, 0755, true)) {
            echo "   ✅ Logs directory created successfully\n";
            
            // Try logging again after creating directory
            echo "   Retrying log write...\n";
            ApiLogger::logRequest('test-retry.php', 'GET', ['retry' => true], []);
            
            if (file_exists($logFile)) {
                echo "   ✅ Log file created after directory creation!\n";
                echo "   File size: " . filesize($logFile) . " bytes\n";
            } else {
                echo "   ❌ Still failed to create log file\n";
            }
        } else {
            echo "   ❌ Failed to create logs directory\n";
        }
    } else {
        echo "   ✅ Logs directory exists: $logDir\n";
        echo "   Directory writable: " . (is_writable($logDir) ? 'yes' : 'no') . "\n";
    }
}

echo "\nTest completed. Check the debug_api_logs.php page to view the logs.\n";
?>
