<?php
/**
 * Test script to debug events API issues
 */

require_once 'config/database.php';

// Initialize database connection
$database = new Database();
$pdo = $database->getConnection();

echo "<h2>Events API Debug Test</h2>";

// Test 1: Check if events table exists and has data
echo "<h3>Test 1: Database Check</h3>";
try {
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM events");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    echo "Total events in database: " . $result['total'] . "<br>";
    
    // Check events for organizer ID 6
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM events WHERE organizer_id = ?");
    $stmt->execute([6]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    echo "Events for organizer ID 6: " . $result['total'] . "<br>";
    
    // Get sample events
    $stmt = $pdo->prepare("SELECT id, title, organizer_id, approval_status, created_at FROM events WHERE organizer_id = ? LIMIT 5");
    $stmt->execute([6]);
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($events) {
        echo "<h4>Sample Events:</h4>";
        foreach ($events as $event) {
            echo "ID: {$event['id']}, Title: {$event['title']}, Status: {$event['approval_status']}, Created: {$event['created_at']}<br>";
        }
    } else {
        echo "No events found for organizer ID 6<br>";
    }
    
} catch (Exception $e) {
    echo "Database error: " . $e->getMessage() . "<br>";
}

// Test 2: Test the API endpoint directly
echo "<h3>Test 2: API Endpoint Test</h3>";
try {
    // Simulate API call without authentication first
    $_GET['action'] = 'organizer-events';
    $_GET['organizer_id'] = '6';
    $_SERVER['REQUEST_METHOD'] = 'GET';
    
    echo "Testing API without authentication...<br>";
    
    // Test the query directly
    $organizerId = 6;
    $limit = 10;
    
    $query = "SELECT e.*, ec.name as category_name, v.name as venue_name
             FROM events e
             LEFT JOIN event_categories ec ON e.category_id = ec.id
             LEFT JOIN venues v ON e.venue_id = v.id
             WHERE e.organizer_id = :organizer_id
             ORDER BY e.created_at DESC
             LIMIT :limit";

    $stmt = $pdo->prepare($query);
    $stmt->bindParam(':organizer_id', $organizerId);
    $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "Query executed successfully. Found " . count($events) . " events.<br>";
    
    if ($events) {
        echo "<h4>API Query Results:</h4>";
        foreach ($events as $event) {
            echo "ID: {$event['id']}, Title: {$event['title']}, Category: {$event['category_name']}, Venue: {$event['venue_name']}<br>";
        }
    }
    
} catch (Exception $e) {
    echo "API test error: " . $e->getMessage() . "<br>";
}

// Test 3: Check table structure
echo "<h3>Test 3: Table Structure Check</h3>";
try {
    $stmt = $pdo->query("DESCRIBE events");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    echo "Events table columns:<br>";
    foreach ($columns as $column) {
        echo "- {$column['Field']} ({$column['Type']})<br>";
    }
} catch (Exception $e) {
    echo "Structure check error: " . $e->getMessage() . "<br>";
}

?>
